"""Synthetic client side command interface."""

#    Copyright (C) 1998-1999 Kevin O'Connor
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import sys
import os
import string
import re
import bisect

import empQueue
import empDb
import empParse
import empEval
import empPath
import empSector

# Key Ideas:

# What is contained within this file:

# This file holds much of the client-side synthetic command interface.  The
# main code implements a command checker that parses outgoing commands and
# determines if they are synthetic.  The main class is EmpParse; it
# performs these aliasing/redirection capabilities.  This file also
# contains a couple of high-level parse classes.  Please see the individual
# class documentation for more information.


# Sending a command to the server:

# To send an outgoing command, the standard method is to invoke
# viewer.ioq.Send().  (Because viewer.ioq points to the only instance of
# EmpParse, in many cases viewer.ioq will be replaced with self.  In
# general, "closer" references should be used in place of references to the
# global variable viewer.)  See the documentation with EmpParse.Send() for
# more info on the syntax and options available.


# Creating a new command:

# Creating a command is done by setting a binding in EmpParse.  EmpParse
# uses three different storage areas: preList, syntaxList, and postList.
# preList/postList are lists of 3-tuples that contain the name of the
# command, the minimum length the command must be (if negative, the command
# will match all commands that have a prefix matching the command), and the
# function to invoke when the command is detected.  preList differs from
# postList in that preList is checked before the syntaxList, while postList
# is checked after.  Items in syntaxList are regular expressions - it can
# be used to extend the command-line to accept full-blown syntax changes,
# however it is probably not a good idea to add too much here..

# Because both preList and postList must remain in sorted order, there are
# two methods that aid in adding commands: regCommand and delayedCommand.

# regCommand simply takes a command list (preList or postList), a function
# to associate with the new command, and a list of 2-tuples that describe
# the possible command forms (a command name, followed by a minimum length
# as described above).  It will then register the command into
# preList/postList as desired.

# delayedCommand is a bit more complex.  It takes the same form as
# regCommand and it essentially performs the same actions.  However, where
# regCommand sets function to be invoked immediately upon invocation,
# delayedCommand will set the binding so that the function is invoked only
# after the command queue has been completely flushed.  (Basically,
# delayedCommand will setup the command so that the function is invoked
# only after synchronizing the client and server.)  Too illustrate this
# better, lets examine the command "foreach".  CmdForeach uses a
# delayedbinding because its results depend upon its location in the queue.
# (IE, "des * g; rdbe; foreach * ?des=g ..." is entirely different from
# "foreach * ?des=g ...;des * g;rdbe") To the contrary, commands like
# "wread" (which opens a Tk window) is entirely unrelated to its position
# in the queue..  Upon a closer look, it may appear that commands like
# "rdb"/CmdRefresh should also be a delayedCommand.  However, this is not
# the case - invocation of the function CmdRefresh does not need to be
# synchronized - only the output of CmdRefresh does..  I'd describe this in
# more detail, only I don't believe this syntax will remain much
# longer.. :-/


class EmpParse:
    """Implements a set of aliasing commands.

    This class is used as a wrapper around empQueue.EmpIOQueue.  Outgoing
    commands are sent here, and if they are not client synthetic commands,
    they are sent to the server via the empQueue module.  This class is
    more 'user-friendly' then the empQueue stuff.  The major interaction
    with this class is done via the Send() module which is passed simple
    ascii strings that are to be transmitted.

    Also in this class are some smart-features.  Check for the methods that
    are labeled 'CmdXXX' for these functions.  In all likelihood these
    functions will be separated in the future..

    Note: This class generally only has one instantiation - viewer.ioq
    """

    def __init__(self, queue):
	# Pointer to the actual low-level socket manager.
	self.sock = queue

	# Copies of methods so that this class can "double" for
	# empQueue.EmpIOQueue.
	self.HandleInput = self.sock.HandleInput
	self.fileno = self.sock.fileno
	self.SendNow = self.sock.SendNow

	# Default command transmission flags:
	self.preFlag = self.postFlag = empQueue.QU_SYNC
## 	self.preFlag = self.postFlag = empQueue.QU_BURST
	self.raw = 0

	# History substitution:
	self.hisList = [""]
	self.hisPos = 0

	# Setup the syntax list:
	self.preList = []
	self.postList = []
	self.syntaxList = (
	    (re.compile(
		r"^(?P<cmd>.*?)\s+"
		+r"(?P<flags>\||(?:>>?!?))"
		+r"\s*(?P<name>.*?)\s*$"),
	     self.SyntaxRedirect),
	    (re.compile(r"(?P<cmd1>.*)\s*;\s*(?P<cmd2>.*)"),
	     self.SyntaxMulti),
	    )

	# Set the default commands:
	self.registerCmds(CmdAlias, CmdNull, CmdEval, CmdForEach
			  , CmdBurst, CmdDefList, CmdRefresh, CmdExec
			  , CmdOut, CmdNova, CmdPredict, CmdMover
			  , CmdRaw, CmdOrigin, CmdMMove, CmdEMove)

    def registerCmds(self, *args):
	"""Register a list of commands."""
	for cmdclass in args:
	    cmds = cmdclass.defaultBinding
	    if cmdclass.defaultPreList:
		lst = self.preList
	    else:
		lst = self.postList
	    for cmd in cmds:
		bisect.insort(lst, cmd + (cmdclass,))

    commandFormat = re.compile(
	r"^\s*(?P<command>\S+)(?:\s+(?P<args>.*))?$")
    def Send(self, cmd, disp=None, prepend=0,
	     preFlag=None, postFlag=None, raw=None):
	"""Search command line for smart commands.

	This method is the main way of interacting with this class.  Send
	will search the given command for known smart-commands.  If one is
	found, its associated code block is run.  Otherwise, the command is
	sent unaltered to EmpIOQueue for normal server execution.  This
	method has grown to include a number of options that may be
	specified for each outgoing command.  disp = the display parser to
	associate with the outgoing command (None indicates that the
	standard viewer should be used).  Setting prepend will instruct the
	command to be prepended to the queue - this causes the command to
	be the next command sent regardless of other waiting commands.
	preFlags/postFlags = the empQueue pre & post flags that should be
	requested for the command (assuming it is not a client-command).
	Setting raw will force this method to send the command unaltered -
	thus ignoring possible client-side commands.
	"""
	if disp is None:
	    disp = viewer

	if raw is None:
	    raw = self.raw

	# Find root word
	if not raw:
	    mm = self.commandFormat.match(cmd)
	    if mm:
		arg0 = mm.group('command')
		# Check pre-syntax list
		f = empQueue.findCmd(self.preList, arg0)
		if f is not None:
		    f(self, mm, disp)
		    return
		# Check syntax list
		for i in self.syntaxList:
		    test = i[0].match(cmd)
		    if test is not None:
			i[1](test, disp)
			return
		# Check post-syntax list
		f = empQueue.findCmd(self.postList, arg0)
		if f is not None:
		    f(self, mm, disp)
		    return
	# Command not aliased - send raw
	if preFlag is None:
	    preFlag = self.preFlag
	if postFlag is None:
	    postFlag = self.postFlag
	hdlr = empQueue.NormalHandler(cmd, disp, preFlag, postFlag)
	if prepend:
	    self.sock.AddHandler(hdlr, 0)
	else:
	    self.sock.AddHandler(hdlr)

    def HistSend(self, cmd):
	"""Send the command with support for history substitution."""
	if cmd[:2] == "!!":
	    cmd = self.hisList[1]
	    del self.hisList[1]
	elif cmd[:1] == "!":
	    cmd = cmd[1:]
	    l = len(cmd)
	    for i in range(1, len(self.hisList)):
		val = self.hisList[i]
		if val[:l] == cmd:
		    cmd = val
		    del self.hisList[i]
		    break
	    else:
		raise IndexError
	else:
	    if cmd[:1] == "^":
		pos = string.find(cmd, "^", 1)
		if pos == -1:
		    raise IndexError
		cmd = string.replace(self.hisList[1], cmd[1:pos],
				     cmd[pos+1:], 1)
		if cmd == self.hisList[1]:
		    raise IndexError

	    # Remove duplicate commands from the history list
	    del self.hisList[0]
	    try: self.hisList.remove(cmd)
	    except ValueError: pass

	# Add the command to the history list
	self.hisList[:0] = ["", cmd]
	self.hisPos = 0
	del self.hisList[100:]

	self.Send(cmd)

    def HistMove(self, offset, cmd):
	"""Note an up/down arrow command-line request; returns new cmdline."""
	if ((offset == 1 and self.hisPos >= len(self.hisList)-1)
	    or (offset == -1 and self.hisPos <= 0)):
	    raise IndexError
	if (offset == 1 and self.hisPos == 0):
	    self.hisList[0] = cmd
	self.hisPos = self.hisPos + offset
	return self.hisList[self.hisPos]

    def SyntaxMulti(self, match, out):
	"""Multi-Command line."""
	self.Send(match.group('cmd1'), out)
	self.Send(match.group('cmd2'), out)

    def SyntaxRedirect(self, match, out):
	"""Redirect output to a file."""
##  	print "redirect"
	flags = match.group('flags')
	if flags == "|":
	    type = "pipe"
	elif flags[:2] == ">>":
	    type = "append"
	elif flags[-1] == "!":
	    type = "force"
	else:
	    type = ""
	self.Send(match.group('cmd'),
		  RedirectOutput(out, match.group('name'), type))



class baseCommand:
    """Base class for synthetic commands. (Does little by itself.)"""

    description = "No help available."

    defaultPreList = defaultBindings = None
    defaultRefresh = None
    invoke = receive = transmit = None
    commandFormat = None

    def __init__(self, ioq, match, disp):
	# Establish defaults for all the standard command class commands.
	self.out = disp
	self.ioq = ioq
	self.Send = ioq.Send
	self.commandMatch = match
	# If commandFormat is defined check command parameters now.
	if self.commandFormat is not None:
	    args = match.group('args')
	    if not args:
		args = ""
	    self.parameterMatch = self.commandFormat.match(args)
	    if not self.parameterMatch:
		viewer.Error("Invalid parameters.  Form should be: %s"
			     % self.commandFormat.pattern)
		return
	# Call invoke method immediately
	if self.invoke is not None:
	    self.invoke()
	# Add receive/transmit method as a callback
	if self.receive is not None or self.transmit is not None:
	    flags = empQueue.QU_FULLSYNC
	    if self.transmit is not None:
		flags = empQueue.QU_BURST
##  	    # Send DB update commands
##  	    if self.defaultRefresh:
##  		ioq.Send("rdb"+self.defaultRefresh, self.out)
	    # Arrange for callback from queue code
	    ioq.sock.AddHandler(empQueue.DummyHandler(
		match.string, self.out, self.receive, self.transmit, flags))


class CmdDefList(baseCommand):

    description = "List all online commands."

    defaultBinding = (('define', 6),)

    def receive(self):
	if not self.out.data:
	    return
	self.out.data('Pre-syntax:')
	for i, k, j in self.ioq.preList:
	    self.out.data("%-10s - %s" % (i, j.description))
	self.out.data("")
	self.out.data("Syntax:")
	for i in self.ioq.syntaxList:
	    self.out.data("%-35s - %s" % (i[0].pattern, i[1].__doc__))
	self.out.data("")
	self.out.data("Post-syntax:")
	for i, k, j in self.ioq.postList:
	    self.out.data("%-10s - %s" % (i, j.description))

class CmdAlias(baseCommand):

    description = "Generate a definition for a simple alias."

    defaultPreList = 1
    defaultBinding = (('alias', 5),)

    EAliasFormat = re.compile(r"\$(\$|\*|\d+)")

    commandFormat = re.compile(
	r"^(?P<newName>\S+)\s+(?P<value>.*?)\s*$")
    def invoke(self):
	newcmd = self.parameterMatch.group('value')
	name = self.parameterMatch.group('newName')
	lst = self.EAliasFormat.split(newcmd)

	class dummyclass(baseCommand):
	    def invoke(self):
		"""Evaluate an alias for a given command."""
		sublst = self.substitutions
		args = self.commandMatch.group('args')
		if args is None:
		    args = ""
		argl = string.split(self.commandMatch.string)
		cmd = ""
		for part in range(len(sublst)):
		    val = sublst[part]
		    if part & 1:
			# Odd value - this is a parameter substitution.
			if val == '*':
			    cmd = cmd + args
			elif val == '$':
			    cmd = cmd + '$'
			else:
			    try:
				cmd = cmd + argl[string.atoi(val)]
			    except IndexError:
				viewer.Error("Not enough arguments for alias.")
				return
		    else:
			cmd = cmd + val
		self.Send(cmd, self.out, 1)

	dummyclass.description = "Aliased to "+repr(newcmd)
	dummyclass.defaultBinding = ((name, len(name)),)
	dummyclass.substitutions = lst

	self.ioq.registerCmds(dummyclass)

class CmdOrigin(baseCommand):
    description = "Block unsupported server command."

    defaultBinding = (('origin', -3),)

    def receive(self):
	viewer.Error("Sorry, the origin command is not supported.")
	viewer.Error("Please see TIPS.html for more information.")

class CmdRaw(baseCommand):
    description = "Send a command without client interpretation."

    defaultPreList = 1
    defaultBinding = (('raw', 3),)

    def invoke(self):
	self.ioq.Send(self.commandMatch.group('args'), raw=1)

class CmdBurst(baseCommand):

    description = "Send a multi-command line without syncing server output."

    defaultPreList = 1
    defaultBinding = (('burst', 5),)

    def invoke(self):
	args = self.commandMatch.group('args')
	if not args:
	    return
	pre, post = self.ioq.preFlag, self.ioq.postFlag
## 	self.preFlag = empQueue.QU_SYNC
## 	self.postFlag = empQueue.QU_FORCEBURST
	self.ioq.preFlag = self.ioq.postFlag = empQueue.QU_BURST
## 	self.Send(args, out, preFlag=pre, postFlag=post)
	self.Send(args, self.out)
	self.ioq.preFlag, self.ioq.postFlag = pre, post

class CmdExec(baseCommand):

    description = "Run commands from a file."

    defaultBinding = (('execute', -3), ('runfeed', 7))

    def invoke(self):
	args = self.commandMatch.group('args')
	if not args:
	    return
	try:
	    if self.commandMatch.group('command')[:1] == 'r':
		# Runfeed command
		file = os.popen(args, 'r')
	    else:
		# Execute command
		file = open(args)
	except IOError:
	    viewer.Error("Unable to open file [%s]." % args)
	    return
	lst = file.readlines()
	file.close()
	for i in lst:
	    i = string.strip(i)
	    if not i or i[0] == '#':
		continue
	    self.Send(i, self.out)

##     class ReallySilentDisp(empQueue.baseDisp):
## 	"""Display class that discards all input."""
## 	Begin = data = Answer = End = empQueue.doNothing

##     class QuietDisp(empQueue.baseDisp):
## 	"""Display class that discards some input."""
## 	data = empQueue.doNothing

class SilentDisp(empQueue.baseDisp):
    """Display class that discards most input."""
    def __init__(self, disp, name):
	empQueue.baseDisp.__init__(self, disp)
	self.name = name
    def __del__(self):
	if self.End is empQueue.doNothing:
	    self.out.End(self.name)
    def Begin(self, cmd):
	if self.Begin is not empQueue.doNothing:
	    self.out.Begin(self.name)
	    self.Begin = empQueue.doNothing
    def End(self, cmd):
	self.End = empQueue.doNothing
    data = empQueue.doNothing

class CmdNull(baseCommand):

    description = "Run command with no output."

    defaultPreList = 1
    defaultBinding = (('null', 4),)

    def invoke(self):
	self.Send(self.commandMatch.group('args')
		  , SilentDisp(self.out, self.commandMatch.string))


class CmdPredict(baseCommand):

    description = "Get predictions for a sector."

    defaultRefresh = "e"
    defaultBinding = (('Predict', 7),)

    def receive(self):
	try:
	    coord = empParse.str2Coords(self.commandMatch.group('args'))
	except ValueError:
	    viewer.Error("Invalid coordinates.")
	    return
	DB = empDb.megaDB['SECTOR'][coord]
	for i in string.split(empSector.sectorPredictions(DB), "\n"):
	    self.out.data(i)

class CmdMover(baseCommand):

    description = "Issue a multi-stage move."

    defaultRefresh = "e"
    defaultBinding = (('Mover', 5),)

    commandFormat = re.compile(
	r"^(?P<comd>\S+)\s+(?P<val>[+-@]?\d+)\s+(?P<sectors>.*)$")
    def receive(self):
	mm = self.parameterMatch
	try:
	    sectors = map(empParse.str2Coords,
			  string.split(mm.group('sectors')))
	    if len(sectors) < 2:
		raise ValueError
	    commodity = empEval.commodityTransform[mm.group('comd')]
	    quantity = getMoveQuantity(mm.group('val'), commodity, sectors)
	except (ValueError, KeyError):
	    viewer.Error("Move invalid input failure.")

	if quantity < 0:
	    quantity = -quantity
	    sectors.reverse()

	for i in range(len(sectors)-1, 0, -1):
	    last = sectors[i-1]
	    cur = sectors[i]
	    self.Send("move %s %d,%d %d %d,%d" % (
		commodity, last[0], last[1], quantity, cur[0], cur[1]),
		      self.out, 1)

class CmdRefresh(baseCommand):

    description = "Refresh client databases."

    defaultBinding = (('rdb', -3), ('rerdb', -5))

    syncFormat = re.compile(
	r"^(?P<total>re)?rdb(?P<pre>P)?(?P<flags>[eslpno]*)$")

    commandFlags = [('e', 'SECTOR', 'dump'),
		    ('l', 'LAND UNITS', 'ldump'),
		    ('s', 'SHIPS', 'sdump'),
		    ('p', 'PLANES', 'pdump'),
		    ('n', 'NUKES', 'ndump'),
		    ('o', 'LOST ITEMS', 'lost')]

    def invoke(self):
	match = self.syncFormat.match(self.commandMatch.group('command'))
	if not match:
	    viewer.Error("refresh regex failure.")
	    return

	# Set the display class class to the null-parser
	self.out = SilentDisp(self.out, match.string)

##  	self.dumpList = match.group('flags') or "elspno"
	self.dumpList = match.group('flags') or "elspo"
	self.persistent = (match.group('pre') != None)
	self.totalDump = (match.group('total') != None)

    def checkPending(self, qElem):
	"""empQueue scan callback: Don't send redundant commands."""
	if (qElem is empQueue.DummyHandler
	    and qElem.tcallback.__class__ is self.__class__):
	    # Found a refresh command pending - don't send any repeated dumps
	    for i in qElem.callback.dumpList:
		pos = string.find(self.dumpList, i)
		if pos != -1:
		    self.dumpList = self.dumpList[:pos] + self.dumpList[pos+1:]
		    if not self.dumpList:
			# No dump commands left - end scan
			return 1

    def transmit(self, qPos):
	# Check if parts of this dump can be picked up by a pending dump.
	if not self.persistent:
	    self.ioq.sock.scanQueue(self.checkPending, start=qPos+1)

	for dumpInfo in self.commandFlags:
	    # Check if this kind of dump is requested.
	    if not dumpInfo[0] in self.dumpList:
		continue

	    # Set the timestamp.
	    ts = ""
	    if (not self.totalDump
		# HACK!	 Always do a full dump of the nuke database
		and dumpInfo[0] != 'n'):
		time = empDb.megaDB[dumpInfo[1]].unofficial_timestamp
		if time != 0:
		    ts = " ?timestamp>%s" % time

	    # Generate the command.
	    qPos = qPos + 1
	    self.ioq.sock.AddHandler(empQueue.NormalHandler(
		dumpInfo[2] + " *" + ts, self.out, pre=empQueue.QU_SYNC
		, post=empQueue.QU_FORCEBURST)
				     , pos=qPos)

s_var = r"(?P<var>\w+)"
s_val = r"(?P<val>\w+)"
s_command = r"(?P<command>.*)"

class CmdEval(baseCommand):

    description = "Evaluate an expression for a sector/database."

    defaultRefresh = "e"
    defaultPreList = 1
    defaultBinding = (('eval', 4),)

    commandFormat = re.compile(
	r"^(?P<type>"+empParse.s_sector+"|nation|version)\s+"+s_command+"$")
    def receive(self):
	mm = self.parameterMatch
	type = mm.group('type')
	if type == 'nation' or type == 'version':
	    db = empDb.megaDB[type]
	else:
	    type = 'SECTOR'
	    key = tuple(map(string.atoi, mm.group('sectorX', 'sectorY')))
	    try:
		db = empDb.megaDB['SECTOR'][key]
	    except KeyError:
		viewer.Error("Could not find sector.")
		return
	try:
	    self.Send(empEval.evalString(
		empEval.estrToExpr(mm.group('command')),
		type, db),
		      self.out, 1)
	except empEval.error, e:
	    viewer.Error(e)

class CmdForEach(baseCommand):

    description = "Run given command on specified sectors."

    defaultRefresh = "e"
    defaultPreList = 1
    defaultBinding = (('foreach', 7),)

    commandFormat = re.compile(
	r"^(?P<sectors>\S+)\s+(?:\?(?P<selectors>\S+)\s+)?"
	+s_command+"$")
    def receive(self):
	mm = self.parameterMatch
	try:
	    list = empEval.foreach(
		"owner==-1 and "+empEval.selectToExpr(mm.group('sectors'),
						      mm.group('selectors')),
		empEval.estrToExpr(mm.group('command')),
		'SECTOR')
	except empEval.error, e:
	    viewer.Error(e)
	else:
	    for i in list:
		self.Send(i, self.out, 1)

class CmdOut(baseCommand):

    description = "Debugging tool - dumps database."

    defaultBinding = (('out', 3),)

##      outFormat = re.compile(r"\s+(\S+)\s*$")
    def receive(self):
	self.out.data(str(empDb.megaDB))
##	cmds = [('e', 'SECTOR', 'dump'),
##		('l', 'LAND UNITS', 'ldump'),
##		('s', 'SHIPS', 'sdump'),
##		('p', 'PLANES', 'pdump'),
## #		('n', 'NUKES', 'ndump'),
##		('o', 'LOST ITEMS', 'lost')]
##	c = match.group(1)
##	if not c:
##	    c = 'elspno'

##	mm = self.match_out.search(match.string)

##	if not mm:
##	    val = '*'
##	else:
##	    val = mm.group(1)

##	for dbt in cmds:
##	    if not dbt[0] in c:
##		continue

##	    out.data(dbt[1]+":" + `self.getSectors(val, "", dbt[1])`)

class CmdMMove(baseCommand):

    description = "Ulf Larsson's multi-move tool."

    defaultRefresh = "e"
    defaultBinding = (('mmove', 5),)

    commandFormat = re.compile(
	"^"+empParse.s_comm+"\s+"
	r"(?P<sectors>\S+)(?:\s+\?(?P<selectors>\S+))?\s+"
	r"(?P<level1>\S+)\s+(?P<mob>\S+)\s+"
	r"(?P<sectors2>\S+)(?:\s+\?(?P<selectors2>\S+))?\s+"
	r"(?P<level2>\S+)\s*$")
    def receive(self):
	mm = self.parameterMatch
	commodity, slevel, mob, dlevel = mm.group(
	    'comm', 'level1', 'mob', 'level2')
	try: commodity = empEval.commodityTransform[commodity]
	except KeyError:
	    viewer.Error("Unknown commodity.")
	    return

	# Call the empEval.foreach function.  empEval.foreach tests every
	# sector in the database to see if arg1 tests true.  If it does, then
	# arg2 is evaluated and inserted into the list.
	try:
	    # Return a list for source sectors that contains tuples of the
	    # form: ((x, y), sectDB, amount, mobility)
	    slist = empEval.foreach(
		("owner==-1 and "+empEval.selectToExpr(
		    mm.group('sectors'), mm.group('selectors'))),
		("(xloc+0,yloc+0), __db[1], int("+commodity+"-("+slevel
		 +")), int(mob-("+mob+"))"),
		'SECTOR')
	    # Return a list for destination sectors that contains tuples of the
	    # form: ((x, y), sectDB, amount)
	    dlist = empEval.foreach(
		("owner==-1 and "+empEval.selectToExpr(
		    mm.group('sectors2'), mm.group('selectors2'))),
		"(xloc+0,yloc+0), __db[1], int(("+dlevel+")-"+commodity+")",
		'SECTOR')
	except empEval.error, e:
	    viewer.Error(e)
	    return

	# Create usable dictionaries from the returned list.
	ddict = {}
	for coord, db, amount in dlist:
	    if amount > 0 and empSector.is_movable_into(db, commodity):
		ddict[coord] = amount
	sdict = {}
	for coord, db, amount, mobility in slist:
	    if (amount > 0 and mobility > 0
		and not ddict.has_key(coord)
		and empSector.is_movable_from(db, commodity)):
		sdict[coord] = (amount, mobility,
				empSector.move_weight(db, commodity))
##  	print "%s\n%s" % (sdict, ddict)

	# Send the dictionaries to the move generator
	mmove = empPath.MoveGenerator(commodity, sdict, ddict)

	# Extract the moves and pass to the command queue.
	while not mmove.empty() :
	    comm, path, amount = mmove.move()
	    self.Send("move %s %d,%d %s %d,%d" % (
		comm, path.start[0], path.start[1], amount,
		path.end[0], path.end[1]),
		      self.out, 1)
	    mmove.next()

class CmdEMove(baseCommand):

    description = "Ulf Larsson's multi-move explore tool."

    defaultRefresh = "e"
    defaultBinding = (('emove', 5),)

    commandFormat = re.compile(
	"^"+empParse.s_comm+"\s+"
	r"(?P<sectors>\S+)(?:\s+\?(?P<selectors>\S+))?\s+"
	r"(?P<level1>\S+)\s+(?P<mob>\S+)\s+"
	r"(?P<sectors2>\S+)(?:\s+\?(?P<selectors2>\S+))?\s*$")
    def receive(self):
	mm = self.parameterMatch
	commodity, slevel, mob = mm.group(
	    'comm', 'level1', 'mob')
	try: commodity = empEval.commodityTransform[commodity]
	except KeyError:
	    viewer.Error("Unknown commodity.")
	    return

	# Call the empEval.foreach function.  empEval.foreach tests every
	# sector in the database to see if arg1 tests true.  If it does, then
	# arg2 is evaluated and inserted into the list.
	try:
	    # Return a list for source sectors that contains tuples of the
	    # form: ((x, y), sectDB, amount, mobility)
	    slist = empEval.foreach(
		("owner==-1 and "+commodity+">0 and "+empEval.selectToExpr(
		    mm.group('sectors'), mm.group('selectors'))),
		("(xloc+0,yloc+0), __db[1], int("+commodity+"-("+slevel
		 +")), int(mob-("+mob+"))"),
		'SECTOR')
	    # Return a list for destination sectors that contains tuples of the
	    # form: ((x, y), sectDB)
	    dlist = empEval.foreach(
		("owner!=-1 and "+empEval.selectToExpr(
		    mm.group('sectors2'), mm.group('selectors2'))),
		"(xloc+0,yloc+0), __db[1]",
		'SECTOR')
	except empEval.error, e:
	    viewer.Error(e)
	    return

	# Create usable dictionaries from the returned list.
	ddict = {}
	for coord, db in dlist:
	    if empSector.is_explorable_into(db):
		ddict[coord] = 1
	sdict = {}
	for coord, db, amount, mobility in slist:
	    if (amount > 0 and mobility > 0
		and not ddict.has_key(coord)
		and empSector.is_movable_from(db, commodity)):
		sdict[coord] = (amount, mobility, 1.0)
##  	print "%s\n%s" % (sdict, ddict)

	# Send the dictionaries to the move generator
	mmove = empPath.MoveGenerator(commodity, sdict, ddict,
				      empPath.ExplMobCost())

	# Extract the moves and pass to the command queue.
	while not mmove.empty() :
	    comm, path, amount = mmove.move()
	    self.Send("explore %s %d,%d %s %sh" % (
		comm, path.start[0], path.start[1], amount, path.directions),
		      self.out, 1)
	    mmove.next()

class CmdNova(baseCommand):

    description = "Auto-explore sectors."

    defaultRefresh = "e"
    defaultBinding = (('nova', 4),)

    commandFormat = re.compile(
	"^"+empParse.s_comm+"\s+"+empParse.s_sector
	+"\s*(?:"+empParse.s_sector2+")?\s*$")
    def receive(self):
	mm = self.parameterMatch
	start = map(string.atoi, mm.group('sectorX', 'sectorY'))
	if mm.group('sector2X'):
	    dest = map(string.atoi, mm.group('sector2X', 'sector2Y'))
	else:
	    dest = start
	if dest == start:
	    estr = "%s,%s 1" % tuple(start)
	else:
	    estr = "%s,%s 1 %s,%s" % tuple(start+dest)
	db = empDb.megaDB['SECTOR']
	# Check if there are wilderness sectors around destination.
	CN_UNOWNED = empDb.CN_UNOWNED
	for newcoord in empDb.sectorNeighbors(dest):
	    sect = db.get(newcoord, {})
	    des = sect.get('des')
	    own = sect.get('owner')
	    if ((own == 0 or own is None or own == CN_UNOWNED)
		and des is not None and des not in ".?s\\"):
## 		print "exploring to %s,%s a (%s) owned by (%s)" % (
## 		    newcoord+(sect.get('des'), sect.get('owner')))
		break
	else:
	    # No sectors to explore
	    return
	self.Send("explore " + mm.group('comm') + " " + estr,
		  HandleNova(self.out, self.ioq, mm.group('comm'), start, dest),
		  1, empQueue.QU_SYNC, empQueue.QU_FULLSYNC)
	self.Send("des * ?newdes=-&mob<1 +", self.out,
##  		  preFlag=empQueue.QU_SYNC,
		  postFlag=empQueue.QU_FORCEBURST)
##	self.Send("esync")

class HandleNova(empQueue.baseDisp):
    """Parse class for nova handled explores.

    This class receives output after the empParse.ParseMove class.
    Basically, it automatically generates moves based on the information
    contained in the prompts.
    """

    def __init__(self, disp, slf, comd, start, dest):
	empQueue.baseDisp.__init__(self, disp)
	self.slf = slf
	self.comd = comd
	self.start = start
	self.dest = dest
	self.oldflush = None
	self.direc = 0
    promptFormat = empParse.ParseMove.promptFormat
    def flush(self, msg, hdl):
	mm = self.promptFormat.match(msg)
	if not mm:
	    # Encountered an unknown prompt
	    self.out.flush(msg, hdl)
	    return
	if self.oldflush is not None and msg != self.oldflush:
	    # Looks like we succeeded in moving.
	    self.out.flush(msg, None)
	    hdl('h')
	    return
	self.oldflush = msg
	db = empDb.megaDB['SECTOR']
	CN_UNOWNED = empDb.CN_UNOWNED
	# Failed to move.
	neighbors = empDb.sectorNeighbors(self.dest)
	pathDirections = empDb.pathDirections
	while 1:
	    if self.direc >= len(pathDirections):
		# No more directions left
		self.out.flush(msg, None)
		hdl('h')
		return
	    newcoord = neighbors[self.direc]
	    sect = db.get(newcoord, {})
	    des = sect.get('des')
	    own = sect.get('owner')
	    if (((own == 0 or own is None) and des is not None
		 and des not in ".\\")
		or (own == CN_UNOWNED and des not in "?s")):
		self.out.flush(msg, None)
		hdl(pathDirections[self.direc])
		break
	    self.direc = self.direc + 1
	self.direc = self.direc + 1
    ownSector = empParse.ParseMove.ownSector
    def data(self, msg):
	self.out.data(msg)
	mm = self.ownSector.match(msg)
	if mm:
	    if self.direc < len(empDb.pathDirections):
		CmdNova(
		    self.slf,
		    EmpParse.commandFormat.match(
			"nova %s %s,%s %s,%s" % (
			    self.comd, self.start[0], self.start[1],
			    self.dest[0], self.dest[1])),
		    self.out)
	    self.slf.Send(
		"nova %s %s,%s %s,%s" % (
		    (self.comd, self.start[0], self.start[1],
		    mm.group('sectorX'), mm.group('sectorY'))),
		self.out)

##  class HandleFlush(empQueue.baseDisp):
##      """Class to handle programable sub-prompts."""
##      def __init__(self, disp, prompts):
##  	empQueue.baseDisp.__init__(self, disp)
##  	self.prompts = list(prompts)
##      def flush(self, msg):
##  	for i in range(prompts):
##  	    val = prompts[i]
##  	    if type(val) == types.tuple:
##  		match, val = val
##  		mm = match.match(msg)
##  		if mm:
##  		    if callable(val):
##  			viewer.SendNow(val(mm))
##  		    else:
##  			viewer.SendNow(val)
##  		    break
##  	    elif callable(val):
##  		viewer.SendNow(val(msg))
##  		del self.prompts[i]
##  		break
##  	    else:
##  		viewer.SendNow(val)
##  		del self.prompts[i]
##  		break
##  	else:
##  	    self.out.flush(msg)

class ParseShow(empQueue.baseDisp):
    """Rearrange output from multiple commands into a set of columns.

    This is a hack!  - It is really only usefull for show XXX YYY.
    Now this hack got hacked by Bernhard to let capability listings
	not grow to big. So it is special for cshow. :-)
	It is important that "cap" are the last char of the last command
	which this class is called with.

    Kevin has arranged a list Body. The last element is printed first,
    so that commands with more overhead lines can append to the list.
    """

    def __init__(self, disp):
	empQueue.baseDisp.__init__(self, disp)

	self.CBody = []
	self.Body = []
	self.first = 1
	self.foundheader = 0

    def __del__(self):
	# do the output
	for i in range(1, len(self.Body)+1):
	    self.out.data(self.Body[-i])

    def End(self, cmd):
	self.out.End(cmd)
	# stuff to do at the end of each block, put CBody in Body

	if self.Body:
	    mx = max(map(len, self.Body))
	else:
	    mx = 0

	# next lines just to test that i have understood this --Bernhard
## 	sys.stdout.write('cmd=' + cmd + "\n")
	if  cmd[-3:] == "cap" :
## 	    sys.stdout.write('found cap cmd\n')
## 	    sys.stdout.flush()
	    mfc=140-mx-1  # max width for cap output (should be determined dynmc)

	al=0 	# added lines, only >0 when formatting cap output
	for i in range(len(self.CBody)):
	    if cmd[-3:] == "cap":
		while len(self.CBody[i])>mfc:
		    # insert enough additional lines in Body and truncate CBody
		    # a better approach would be to find a word to split
		    ll= len(self.CBody[i])
		    if (ll/mfc)*mfc==ll:
			ii=(ll/mfc - 1)*mfc
		    else:
			ii=(ll/mfc)*mfc
		    self.Body.insert(i+al,' '*mx + ' ' + self.CBody[i][ii:])
		    self.CBody[i]= self.CBody[i][:ii]
		    al=al+1

	    if i+1 > len(self.Body):
		self.Body.append(' '*mx + self.CBody[i])
	    else:
		self.Body[i+al] =  ( self.Body[i+al]
					+ ' '*(mx-len(self.Body[i+al]))
					+ self.CBody[i])
	self.CBody = []
	self.Bpos = -1
	self.first = 0
	self.foundheader = 0

    def data(self, msg):
	# prepend message to CBody
	if not self.foundheader:
	    if string.lstrip(msg[:25]) == '':
		self.foundheader = 1
	    else:
		return
	if self.first:
	    self.CBody[0:0] = [ msg ]
	else:
	    self.CBody[0:0] = [ msg[25:] ]

class RedirectOutput(empQueue.baseDisp):
    """Redirect a command to a file or program."""
    def __init__(self, disp, name, type):
	empQueue.baseDisp.__init__(self, disp)
	self.name = name
	self.type = type
    def Begin(self, cmd):
##  	print "opening"
	if self.out.Begin is empQueue.doNothing:
	    # File already opened.
	    return
	self.out.Begin(cmd)
	try:
	    if self.type == "pipe":
		self.file = os.popen(self.name, 'w')
	    else:
		if self.type == "append":
		    flags = "a"
		elif self.type != "force" and os.path.exists(self.name):
		    raise IOError, "File %s already exists." % self.name
		else:
		    flags = "w"
		self.file = open(self.name, flags)
	except:
	    viewer.Error('Exception %s with detail:\n"%s".'
			 % tuple(sys.exc_info()[:2]))
	    self.file = None
    def End(self, cmd):
##  	print 'end'
	self.out.End(cmd)
	if self.file is not None:
	    self.file.close()
    def data(self, msg):
##  	print 'data'
	if self.file is not None:
	    self.file.write(msg+"\n")
	else:
	    self.out.data(msg)

def getMoveQuantity(txt, commodity, sectors):
    var = string.lstrip(txt)
    flag = 0
    if var and var[0] in '-+@':
	if var[0] == '@':
	    flag = 1
	    var = var[1:]
	else:
	    flag = 2

    val = string.atoi(var)

    # Check for reverse paths.
    if flag == 2:
	# Absolute move.
	quantity = val
    elif flag == 1:
	# Reverse "smart" move.
	db = empDb.megaDB['SECTOR'][sectors[-1]]
	quantity = val - db.get(commodity, 0)
    else:
	# Normal "smart" move.
	db = empDb.megaDB['SECTOR'][sectors[0]]
	quantity = db.get(commodity, 0) - val

    return quantity

##  class telegramParser(empQueue.baseDisp):
##      def __init__(self, cmd, msg):
##  	empQueue.baseDisp.__init__(self, viewer)
##  	self.cmd = cmd
##  	self.msg = string.split(string.rstrip(msg), '\n')
##      prompt = empParse.ParseTele.prompt
##      def flush(self, msg, hdl):
##  	# Get the characters remaining in this message
##  	mm = self.prompt.match(msg)
##  	if not mm:
##  	    self.out.flush(msg, hdl)
##  	else:
##  	    response = ""
##  	    left = string.atoi(mm.group('left'))
##  	    if len(self.msg) == 0:
##  		response = "."
##  	    elif len(self.msg[0]) > left:
##  		if len(self.msg[0]) > 1024:
##  		    # Force out a partial string
##  		    response = self.msg[0][:left]
##  		    self.msg[0] = self.msg[0][left:]
##  		else:
##  		    # Finish this message
##  		    viewer.ioq.Send(self.cmd, self, 1)
##  		    response = "."
##  	    else:
##  		response = self.msg[0]
##  		del self.msg[0]
##  	    self.out.flush(msg, None)
##  	    hdl(response)

def sendTelegram(cmd, msg):
    """Send a telegram to the server.

    This function takes a telegram/announcement command stored in cmd, and
    a multi-line string stored in msg and then bursts the telegram command
    and the body of the message to the server.  The results of this command
    are not synchronized or checked in any way.  Assumptions are made about
    the form and acceptable content of a telegram, and then all the info is
    bursted out.  The advantage of bursting telegrams, is that it makes the
    transmission extremely fast.
    """
    msg = string.split(string.rstrip(msg), '\n')
    if not msg:
	return
    tot = 0
    QU_BURST = empQueue.QU_BURST
    Send = viewer.ioq.Send
    # Start the tele command
    Send(cmd, postFlag=QU_BURST)
    while msg:
	val = msg[0]
	del msg[0]
	# Alter any suspicious lines so the server doesn't choke on them.
	if val in ("~u", "~p", "~q", "."):
	    val = val+" "
	l = len(val)
	# Assume the total bytes in a telegram is 1024.
	if l > 1024:
	    l = 1024-l
	    msg[:0].insert(val[l:], 0)
	    val = val[:l]
	elif tot + l > 1024:
	    # Continue message on another tele command
	    Send(".", preFlag=QU_BURST, postFlag=QU_BURST, raw=1)
	    Send(cmd, preFlag=QU_BURST, postFlag=QU_BURST)
	    tot = 0
	Send(val, preFlag=QU_BURST, postFlag=QU_BURST, raw=1)
	tot = tot + l + 1
    Send(".", preFlag=QU_BURST, raw=1)
