/* Copyright (c) 1997-1999 Guenter Geiger, Miller Puckette, Larry Troxler,
* Winfried Ritsch, and others.
* For information on usage and redistribution, and for a DISCLAIMER OF ALL
* WARRANTIES, see the file, "LICENSE.txt," in this distribution.  */

/* this file implements the sys_ functions profiled in m_imp.h for
 audio and MIDI I/O.  In Linux there might be several APIs for doing the
 audio part; right now there are three (OSS, ALSA, RME); the third is
 for the RME 9652 driver by Winfried Ritsch.
 
 FUNCTION PREFIXES.
    sys_ -- functions which must be exported to Pd on all platforms
    linux_ -- linux-specific objects which don't depend on API,
    	mostly static but some exported.
    oss_, alsa_, rme_ -- API-specific functions, all of which are
    	static.
     
 For MIDI, we only offer the OSS API; ALSA has to emulate OSS for us.
*/

/* OSS include (whether we're doing OSS audio or not we need this for MIDI) */

#include <linux/soundcard.h>

#ifdef HAVE_ALSA   /* ALSA includes */

#include <sys/asoundlib.h>
#include <linux/asound.h>
#endif

#include "m_imp.h"
#include <stdio.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sched.h>
#include <sys/mman.h>


/* midi lib include */

#include "s_linux_mididefs.h"
#include "s_linux_midi_io.h"
#include "s_linux_midiparse.h"

/* local function prototypes */

static void linux_close_midi( void);

static int oss_open_audio(int inchans, int outchans, int rate);
static void oss_close_audio(void);
static int oss_send_dacs(void);
static void oss_reportidle(void);

#ifdef HAVE_ALSA
static int alsa_open_audio(int inchans, int outchans, int rate);
static void alsa_close_audio(void);
static int alsa_send_dacs(void);
static void alsa_reportidle(void);
#endif
 
#ifdef HAVE_RME
static int rme9652_open_audio(int inchans, int outchans, int rate);
static void rme9652_close_audio(void);
static int rme9652_send_dacs(void);
static void rme9652_reportidle(void);
#endif

/* Defines */
#define DEBUG(x) x
#define DEBUG2(x) {x;}

#define OSS_DEFAULTCH 2
typedef int16_t t_oss_sample;
#define OSS_SAMPLEWIDTH sizeof(t_oss_sample)
#define OSS_BYTESPERCHAN (DACBLKSIZE * OSS_SAMPLEWIDTH) 
#define OSS_XFERSAMPS (OSSCHPERDEV*DACBLKSIZE)
#define OSS_XFERSIZE (OSS_SAMPLEWIDTH * OSS_XFERSAMPS)

#ifdef HAVE_RME
#define RME_DEFAULTCH 8
typedef int32_t t_rme_sample;
#define RME_SAMPLEWIDTH sizeof(t_rme_sample)
#define RME_BYTESPERCHAN (DACBLKSIZE * RME_SAMPLEWIDTH)
#endif

#define MAXDEV 4    	/* maximum number of input or output devices */

/* GLOBALS */

static int linux_whichapi = API_OSS;
static int linux_inchannels;
static int linux_outchannels;
static int linux_advance_samples; /* scheduler advance in samples */
static int linux_meters;    	/* true if we're metering */
static float linux_inmax;    	/* max input amplitude */
static float linux_outmax;    	/* max output amplitude */

/* our device handles */

typedef struct _dh {
#ifdef HAVE_ALSA
     snd_pcm_t *handle; /* handle for snd_pcm_open call, etc */
     int channels;  	/* ALSA only; OSS is always 2ch per device */
#endif
     int fd;
     unsigned int space; 	/* bytes available for writing/reading  */
     int bufsize;   	/* total buffer size in blocks for this device */
     int dropcount; 	/* number of buffers to drop for resync (output only) */
} t_dac_handle;

static t_dac_handle linux_dacs[MAXDEV];
static t_dac_handle linux_adcs[MAXDEV];
static int linux_ndacs = 0;
static int linux_nadcs = 0;

    /* exported variables */
int sys_schedadvance = 50000; 	/* scheduler advance in microseconds */
float sys_dacsr;
int sys_hipriority = 0;
t_sample *sys_soundout;
t_sample *sys_soundin;

    /* OSS-specific private variables */
#define OSSCHPERDEV 2	/* channels per OSS device */
static int oss_oldaudio = 0;	/* flag to use "fragments" below  */
  /* log base 2 of audio "fragment size" in bytes */
static int oss_logfragsize = 0xb;
static int oss_nfragment = 4;
static char ossdsp[] = "/dev/dsp%d"; 


/* ------------- private routines for all APIS ------------------- */
static void linux_flush_all_underflows_to_zero(void)
{
/*
    TODO: Implement similar thing for linux (GGeiger) 

    One day we will figure this out, I hope, because it 
    costs CPU time dearly on Intel  - LT
  */
     /*    union fpc_csr f;
	   f.fc_word = get_fpc_csr();
	   f.fc_struct.flush = 1;
	   set_fpc_csr(f.fc_word);
     */
}

static void linux_setchsr(int chin,int chout, int sr)
{
    int nblk;
    int inbytes = chin * (DACBLKSIZE*sizeof(float));
    int outbytes = chout * (DACBLKSIZE*sizeof(float));

    linux_inchannels = chin;
    linux_outchannels = chout;
    sys_dacsr = sr;
    linux_advance_samples = (sys_schedadvance * sys_dacsr) / (1000000.);
    if (linux_advance_samples < 3 * DACBLKSIZE)
    	linux_advance_samples = 3 * DACBLKSIZE;

    if (sys_soundin)
    	free(sys_soundin);
    sys_soundin = (t_float *)malloc(inbytes);
    memset(sys_soundin, 0, inbytes);

    if (sys_soundout)
    	free(sys_soundout);
    sys_soundout = (t_float *)malloc(outbytes);
    memset(sys_soundout, 0, outbytes);

    post("advance %d samples, samplerate = %f",
    	linux_advance_samples, sys_dacsr);
    post("inputchannels = %d, outputchannels = %d",
    	linux_inchannels, linux_outchannels);
}

/* ---------------- private MIDI routines -------------------------- */
#define NPORTS 16

void inmidi_rtin(int portno, int cmd);
void inmidi_mtc(int portno, int frame, int sec, int min, int hr);
void inmidi_noteon(int portno, int channel, int pitch, int velo);
void inmidi_controlchange(int portno, int channel, int ctlnumber, int value);
void inmidi_programchange(int portno, int channel, int value);
void inmidi_pitchbend(int portno, int channel, int value);
void inmidi_aftertouch(int portno, int channel, int value);
void inmidi_polyaftertouch(int portno, int channel, int pitch, int value);

static int  oss_nports;
MIDIParser *oss_mpa[NPORTS];
MIDIPort *oss_midiport[NPORTS];

static void inmidi_noteonoff(int portno, int channel, int pitch,
    int velon, int veloff)
{
    	/* ignore veloff, pd does not respect it */
    inmidi_noteon(portno, channel, pitch, velon);
}

void linux_open_midi(void)
{
    int i;
    	/* Look for MIDI devices */
    MIDIPort *mp;
    oss_nports = 0;
    if(midi_open()<=0)
    	goto bad;

    for (mp=midi_getfirstport(); oss_nports < NPORTS && mp != NULL;
    	mp = midi_getnextport())
    {
	oss_midiport[oss_nports] = mp;

	oss_mpa[oss_nports] = mp_parse_init(oss_nports);  /* .... */

	    /* initialize all reaction functions */
	oss_mpa[oss_nports]->noteonoff = inmidi_noteonoff;
	oss_mpa[oss_nports]->noteon = NULL;
	oss_mpa[oss_nports]->noteoff = NULL;

	oss_mpa[oss_nports]->controlchange = inmidi_controlchange;
	oss_mpa[oss_nports]->programchange = inmidi_programchange;
	oss_mpa[oss_nports]->pitchbend = inmidi_pitchbend;
	oss_mpa[oss_nports]->aftertouch = inmidi_aftertouch;
	oss_mpa[oss_nports]->polyaftertouch = inmidi_polyaftertouch;
	    /* system messages */
	oss_mpa[oss_nports]->mode = NULL;
	oss_mpa[oss_nports]->quarterframe = NULL; /* inmidi_mtc */
	oss_mpa[oss_nports]->mode = NULL;
	oss_mpa[oss_nports]->songposition = NULL;
	oss_mpa[oss_nports]->songselect = NULL;
	oss_mpa[oss_nports]->tunerequest = NULL;
	oss_mpa[oss_nports]->sysex = NULL;
	oss_mpa[oss_nports]->realtime = NULL; /* inmidi_rtin; */

	oss_nports++;
    };

    if (oss_nports == 1) 
    	fprintf(stderr,"MIDIlib: Found one MIDI port on %s\n",
	oss_midiport[0]->devname);
    else 
    	for(i=0;i<oss_nports;i++) 
    	    fprintf(stderr,"MIDIlib Found MIDI port %d on %s\n",
    	    	i,oss_midiport[i]->devname);		
    return;

 bad:
    error("failed to open MIDI ports; continuing without MIDI.");
    oss_nports = 0;
    return;
}

void linux_close_midi()
{
     int i;
     for(i=0;i<oss_nports;i++)
     {
	  mp_parse_exit(oss_mpa[i]);
     }
     midi_close();
}

/* ----------------------- public routines ----------------------- */

void sys_open_audio_and_midi(int midiin, int midiout,
    int inchans, int outchans, int rate)
{
    int defaultchannels =
    	(linux_whichapi == API_RME ? RME_DEFAULTCH : OSS_DEFAULTCH);
    if (rate < 1) rate = 44100;
    if (inchans < 0) inchans = defaultchannels;
    if (outchans < 0) outchans = defaultchannels;
    linux_flush_all_underflows_to_zero();
#ifdef HAVE_ALSA 
     if (linux_whichapi == API_ALSA)
     	alsa_open_audio(inchans,outchans,rate);
     else 
#endif
#ifdef HAVE_RME
     if (linux_whichapi == API_RME)
     	rme9652_open_audio(inchans,outchans,rate);
     else 
#endif
    	oss_open_audio(inchans,outchans,rate);

    if (midiin || midiout) linux_open_midi();
}

void sys_close_audio_and_midi(void)
{
#ifdef HAVE_ALSA 
    if (linux_whichapi == API_ALSA)
     	alsa_close_audio();
    else
#endif
#ifdef HAVE_RME 
    if (linux_whichapi == API_RME)
     	rme9652_close_audio();
    else
#endif
     oss_close_audio();
     
     linux_close_midi();
}

int sys_send_dacs(void)
{
    if (linux_meters)
    {
    	int i, n;
	float maxsamp;
	for (i = 0, n = linux_inchannels * DACBLKSIZE, maxsamp = linux_inmax;
	    i < n; i++)
	{
	    float f = sys_soundin[i];
	    if (f > maxsamp) maxsamp = f;
	    else if (-f > maxsamp) maxsamp = -f;
	}
	linux_inmax = maxsamp;
	for (i = 0, n = linux_outchannels * DACBLKSIZE, maxsamp = linux_outmax;
	    i < n; i++)
	{
	    float f = sys_soundout[i];
	    if (f > maxsamp) maxsamp = f;
	    else if (-f > maxsamp) maxsamp = -f;
	}
	linux_outmax = maxsamp;
    }
#ifdef HAVE_ALSA 
    if (linux_whichapi == API_ALSA)
    	return alsa_send_dacs();
#endif
#ifdef HAVE_RME
    if (linux_whichapi == API_RME)
    	return rme9652_send_dacs();
#endif
    return oss_send_dacs();
}

float sys_getsr(void)
{
     return (sys_dacsr);
}

int sys_get_outchannels(void)
{
     return (linux_outchannels); 
}

int sys_get_inchannels(void) 
{
     return (linux_inchannels);
}

void sys_audiobuf(int n)
{
     /* set the size, in milliseconds, of the audio FIFO */
     if (n < 5) n = 5;
     else if (n > 5000) n = 5000;
     fprintf(stderr, "audio buffer set to %d milliseconds\n", n);
     sys_schedadvance = n * 1000;
}

void sys_getmeters(float *inmax, float *outmax)
{
    if (inmax)
    {
    	linux_meters = 1;
	*inmax = linux_inmax;
	*outmax = linux_outmax;
    }
    else
    	linux_meters = 0;
    linux_inmax = linux_outmax = 0;
}

void sys_reportidle(void)
{
}

void sys_putmidimess(int portno, int a, int b, int c)
{
  if (oss_nports) {
       switch (md_msglen(a)){
       case 2:
	    midi_out(oss_midiport[portno],a);	 
	    midi_out(oss_midiport[portno],b);	 
	    midi_out(oss_midiport[portno],c);
	    return;
       case 1:
	    midi_out(oss_midiport[portno],a);	 
	    midi_out(oss_midiport[portno],b);	 
	    return;
       case 0:
	    midi_out(oss_midiport[portno],a);	 
	    return;
       };
  }
}


void sys_poll_midi(void)
{
    int i;

    for (i = 0; i < oss_nports; i++)
    {
	if (midi_instat(oss_midiport[i]))
	{
	    mp_parse(oss_mpa[i],midi_in(oss_midiport[i]));
    	}
    }
}

void sys_set_priority( void) 
{
    struct sched_param par;
    int p1,p2, success = 0;
#if defined(_POSIX_PRIORITY_SCHEDULING)

    p1 = sched_get_priority_min(SCHED_FIFO);
    p2 = sched_get_priority_max(SCHED_FIFO);

    par.sched_priority = 99; /* Obviously this needs to be refined - LT */

    if (sched_setscheduler(0,SCHED_FIFO,&par) != -1)
       post("warning: high priority scheduling enabled");
    else
    	sys_hipriority = 1;
#endif

#ifdef _POSIX_MEMLOCK
    if (mlockall(MCL_FUTURE) != -1) 
    	post("warning: memory locking enabled");
#endif
}

/* ------------ linux-specific command-line flags -------------- */

void linux_frags(int n) {
     oss_nfragment = n;
     oss_oldaudio = 1;
     post("fragment count specified; using old-fashioned audio I/O");
}

void linux_fragsize(int n) {
     oss_logfragsize = n;
     oss_oldaudio = 1;
     post("fragment size specified; using old-fashioned audio I/O");
}

void linux_set_sound_api(int which)
{
     linux_whichapi = which;
     post("foo %d", linux_whichapi);
}

/* -------------- Audio I/O using the OSS API ------------------ */ 

int oss_reset(int fd) {
     int err;
     if ((err = ioctl(fd,SNDCTL_DSP_RESET)) < 0)
	  error("OSS: Could not reset");
     return err;
}

void oss_configure(int dev, int fd,int srate, int dac) {
    int orig, param, nblk;
    audio_buf_info ainfo;
    orig = param = srate; 

    /* samplerate */

    if (ioctl(fd,SNDCTL_DSP_SPEED,&param) == -1)
     fprintf(stderr,"OSS: Could not set sampling rate for device\n");
    else if( orig != param )
    	fprintf(stderr,"OSS: sampling rate: wanted %d, got %d\n",
	    orig, param );				

    // setting the correct samplerate (could be different than expected)     
    srate = param;

    	/* I don't know if you can really do this in general */
    if (oss_oldaudio)
    { 

    	/* setting fragment count and size */

    	param = (oss_nfragment<<16) + oss_logfragsize;

    	post("nfrags %d, fragment %d",
    	    oss_nfragment, oss_logfragsize);

    	if (ioctl(fd,SNDCTL_DSP_SETFRAGMENT,&param) == -1)
    	    fprintf(stderr,"OSS: Could not set fragment size\n");
    }

    /* setting resolution */

    orig = param = AFMT_S16_LE;
    if (ioctl(fd,SNDCTL_DSP_SETFMT,&param) == -1)
    	fprintf(stderr,"OSS: Could not set DSP format\n");
    else if( orig != param )
    	fprintf(stderr,"OSS: DSP format: wanted %d, got %d\n",orig, param );

    /* setting channels */

    orig = param = OSSCHPERDEV;

    if (ioctl(fd,SNDCTL_DSP_CHANNELS,&param) == -1)
    	fprintf(stderr,"OSS: Could not set channels\n");
    else if( orig != param )
    	fprintf(stderr,"OSS: num channels: wanted %d, got %d\n",orig, param );

    if (dac)
    {
	/* use "free space" to learn the buffer size.  Normally you
	should set this to your own desired value; but this seems not
	to be implemented uniformly across different sound cards.  LATER
	we should figure out what to do if the requested scheduler advance
	is greater than this buffer size; for now, we just print something
	out.  */

	if (ioctl(linux_dacs[dev].fd, SOUND_PCM_GETOSPACE,&ainfo) < 0)
	   fprintf(stderr,"OSS: ioctl on output device %d failed",dev);
	linux_dacs[dev].bufsize = ainfo.bytes;

	if (linux_advance_samples * (OSS_SAMPLEWIDTH * OSSCHPERDEV) >
	    linux_dacs[dev].bufsize - OSS_XFERSIZE)
	{
	    fprintf(stderr,
	    	"OSS: requested audio buffer size %d limited to %d\n",
		    linux_advance_samples * (OSS_SAMPLEWIDTH *OSSCHPERDEV),
		    linux_dacs[dev].bufsize);
    	    linux_advance_samples =
	    	(linux_dacs[dev].bufsize - OSS_XFERSAMPS) /
		     (OSS_SAMPLEWIDTH *OSSCHPERDEV);
	}
    }
}

int oss_open_audio(int inchans, int outchans,int srate)
{  
    int orig;
    int tmp;
    int inchannels = 0,outchannels = 0;
    char devname[20];
    int i;
    char* buf[OSS_SAMPLEWIDTH*OSSCHPERDEV*DACBLKSIZE];
    int num_devs = 0;
    audio_buf_info ainfo;

    linux_nadcs = linux_ndacs = 0;

    /* First check if we can do full duplex */
    /* and open the write ports	*/

    for (num_devs=0; outchannels < outchans; num_devs++) {
	int channels = 2;
	if (num_devs)
	   sprintf(devname, ossdsp, num_devs);
	else
	   sprintf(devname,"/dev/dsp");

	DEBUG(fprintf(stderr,"OSS: device Nr. %d on %s\n",linux_ndacs+1,devname);)

	if ((tmp = open(devname,O_WRONLY)) == -1) {
	    DEBUG(fprintf(stderr,"OSS: failed to open %s writeonly\n",
	    	devname);)
	    break;
	}

	/* if (ioctl(tmp,SNDCTL_DSP_GETCAPS,&orig) == -1)
	    error("OSS: SNDCTL_DSP_GETCAPS failed %s",devname); */

	if (inchans > inchannels)
	{
	    if (1) /* used to test for DSP_CAP_DUPLEX; better just to try it */
	    {
		close(tmp);
		if ((linux_dacs[linux_ndacs].fd = 
		   linux_adcs[linux_nadcs].fd = 
		   open(devname, O_RDWR)) < 0)
		{
		    fprintf(stderr,
		    "OSS: failed to open dac; continuing without audio\n");
    	    	    linux_setchsr(2,2,44100);
		    return 0;
		}
		else
		{
		    linux_ndacs++;
		    linux_nadcs++;
		    outchannels += channels;
		    inchannels += channels;
		}
	    }
	    else
	    {
		linux_dacs[linux_ndacs].fd = tmp;
		outchannels += channels;
		linux_ndacs++;
	    }
	}
	else
	{
	   linux_dacs[linux_ndacs].fd = tmp;
	   outchannels += channels;
	   linux_ndacs++;
	}
     }

    for (; inchans > inchannels; linux_nadcs++, num_devs++) {
	int channels = 2;
	post("OSS: requiring extra adc handles");
	if (num_devs)
	   sprintf(devname,ossdsp,num_devs);
	else
	   sprintf(devname,"/dev/dsp");

	if ((tmp = open(devname,O_RDONLY)) == -1)
	{
	   DEBUG(fprintf(stderr,"OSS: failed to open %s read only\n",
		devname);)
	   break;
	}
	else 
	   linux_adcs[linux_nadcs].fd = tmp;
	inchans += channels;
    }

    	/* set sample rate; this also calculates linux_advance_samples.  */
    linux_setchsr(inchannels,outchannels,srate);

    /* configure soundcards */

    for (i=0;i<linux_nadcs;i++) {
	oss_configure(i, linux_adcs[i].fd,srate, 0);
    }
    for (i=0;i<linux_ndacs;i++) {
	oss_configure(i, linux_dacs[i].fd,srate, 1);
    }
      /* We have to do a read to start the engine. This is 
	 necessary because sys_send_dacs waits until the input
	 buffer is filled and only reads on a filled buffer.
	 This is good, because it's a way to make sure that we
	 will not block */

    if (linux_nadcs) {
	fprintf(stderr,("OSS: starting engine ... "));
	read(linux_adcs[0].fd,buf,OSS_SAMPLEWIDTH*OSSCHPERDEV*DACBLKSIZE);
	fprintf(stderr,"done\n");
    }
    return (0);
}

void oss_close_audio( void)
{
     int i;
     for (i=0;i<linux_ndacs;i++)
	  close(linux_dacs[i].fd);

     for (i=0;i<linux_nadcs;i++)
          close(linux_adcs[i].fd);
}

static int linux_dacs_write(int fd,void* buf,long bytes) {
    return write(fd, buf,bytes);
}

static int linux_adcs_read(int fd,void*  buf,long bytes) {
     return read(fd, buf,bytes);
}

    /* query audio devices for "available" data size. */
static void oss_calcspace(void)
{
    int dev;
    audio_buf_info ainfo;
    for (dev=0; dev < linux_ndacs; dev++)
    {
	if (ioctl(linux_dacs[dev].fd, SOUND_PCM_GETOSPACE,&ainfo) < 0)
	   fprintf(stderr,"OSS: ioctl on output device %d failed",dev);
	linux_dacs[dev].space = ainfo.bytes;
    }

    for (dev = 0; dev < linux_nadcs; dev++)
    {
	if (ioctl(linux_adcs[dev].fd, SOUND_PCM_GETISPACE,&ainfo) < 0)
	    fprintf(stderr, "OSS: ioctl on input device %d, fd %d failed",
    	    	dev, linux_adcs[dev].fd);
	linux_adcs[dev].space = ainfo.bytes;
    }
}

void linux_audiostatus(void)
{
    int dev;
    if (!oss_oldaudio)
    {
	oss_calcspace();
	for (dev=0; dev < linux_ndacs; dev++)
	    fprintf(stderr, "dac %d space %d\n", dev, linux_dacs[dev].space);

	for (dev = 0; dev < linux_nadcs; dev++)
	    fprintf(stderr, "adc %d space %d\n", dev, linux_adcs[dev].space);

	fprintf(stderr, "dac buf %d, advance %d\n", linux_dacs[0].bufsize,
    	    linux_advance_samples);
    }
}

/* this call resyncs audio output and input which will cause discontinuities
in audio output and/or input. */ 

static void oss_doresync( void)
{
    int dev, zeroed = 0, wantsize;
    t_oss_sample buf[OSS_XFERSAMPS];
    audio_buf_info ainfo;

    	/* 1. if any input devices are ahead (have more than 1 buffer stored),
	    drop one or more buffers worth */
    for (dev = 0; dev < linux_nadcs; dev++)
    {
    	if (linux_adcs[dev].space == 0)
	{
	    linux_adcs_read(linux_adcs[dev].fd, buf, OSS_XFERSIZE);
	}
	else while (linux_adcs[dev].space > OSS_XFERSIZE)
	{
    	    linux_adcs_read(linux_adcs[dev].fd, buf, OSS_XFERSIZE);
	    if (ioctl(linux_adcs[dev].fd, SOUND_PCM_GETISPACE, &ainfo) < 0)
	    {
	    	fprintf(stderr, "OSS: ioctl on input device %d, fd %d failed",
    	    	    dev, linux_adcs[dev].fd);
	    	break;
	    }
	    linux_adcs[dev].space = ainfo.bytes;
	}
    }

    	/* 2. if any output devices are behind, feed them zeros to catch them
	    up */
    for (dev = 0; dev < linux_ndacs; dev++)
    {
    	while (linux_dacs[dev].space > linux_dacs[dev].bufsize - 
	    linux_advance_samples * (OSSCHPERDEV * OSS_SAMPLEWIDTH))
	{
    	    if (!zeroed)
	    {
	    	int i;
		for (i = 0; i < OSS_XFERSAMPS; i++) buf[i] = 0;
		zeroed = 1;
	    }
    	    linux_dacs_write(linux_dacs[dev].fd, buf, OSS_XFERSIZE);
	    if (ioctl(linux_dacs[dev].fd, SOUND_PCM_GETOSPACE, &ainfo) < 0)
	    {
	    	fprintf(stderr, "OSS: ioctl on output device %d, fd %d failed",
    	    	    dev, linux_dacs[dev].fd);
	    	break;
	    }
	    linux_dacs[dev].space = ainfo.bytes;
	}
    }
    	/* 3. if any DAC devices are too far ahead, plan to drop the
	    number of frames which will let the others catch up. */
    for (dev = 0; dev < linux_ndacs; dev++)
    {
    	if (linux_dacs[dev].space > linux_dacs[dev].bufsize - 
	    (linux_advance_samples - 1) * (OSSCHPERDEV * OSS_SAMPLEWIDTH))
	{
    	    linux_dacs[dev].dropcount = linux_advance_samples - 1 - 
	    	(linux_dacs[dev].space - linux_dacs[dev].bufsize) /
		     (OSSCHPERDEV * OSS_SAMPLEWIDTH) ;
	}
	else linux_dacs[dev].dropcount = 0;
    }
}

int oss_send_dacs(void)
{
    float *fp1, *fp2;
    long fill;
    int i, j, dev, rtnval = SENDDACS_YES;
    t_oss_sample buf[OSSCHPERDEV * DACBLKSIZE], *sp;
    	/* the maximum number of samples we should have in the ADC buffer */
    int idle = 0;
    double timeref, timenow;

    if (!linux_nadcs && !linux_ndacs)
    	return (SENDDACS_NO);

    if (!oss_oldaudio)
    	oss_calcspace();
    
    	/* determine whether we're idle.  This is true if either (1)
    	some input device has less than one buffer to read or (2) some
	output device has fewer than (linux_advance_samples) blocks buffered
	already. */
    for (dev=0; dev < linux_ndacs; dev++)
	if (linux_dacs[dev].dropcount ||
	    (linux_dacs[dev].bufsize - linux_dacs[dev].space >
	    	linux_advance_samples * (OSS_SAMPLEWIDTH * OSSCHPERDEV)))
    	    	    idle = 1;
    for (dev=0; dev < linux_nadcs; dev++)
	if (linux_adcs[dev].space < OSS_XFERSIZE)
	    idle = 1;

    if (idle && !oss_oldaudio)
    {
    	    /* sometimes---rarely---when the ADC available-byte-count is
	    zero, it's genuine, but usually it's because we're so
	    late that the ADC has overrun its entire kernel buffer.  We
	    distinguish between the two by waiting 2 msec and asking again.
	    There should be an error flag we could check instead; look for this
    	    someday... */
    	for (dev = 0;dev < linux_nadcs; dev++) 
	    if (linux_adcs[dev].space == 0)
	{
    	    audio_buf_info ainfo;
	    sys_microsleep(2000);
	    oss_calcspace();
	    if (linux_adcs[dev].space != 0) continue;

    	    	/* here's the bad case.  Give up and resync. */
	    sys_log_error(ERR_ADCSTALL);
	    oss_doresync();
	    return (SENDDACS_NO);
	}
    	    /* check for slippage between devices, either because
	    data got lost in the driver from a previous late condition, or
	    because the devices aren't synced.  When we're idle, no
	    input device should have more than one buffer readable and
	    no output device should have less than linux_advance_samples-1
	    */
	    
	for (dev=0; dev < linux_ndacs; dev++)
	    if (!linux_dacs[dev].dropcount &&
		(linux_dacs[dev].bufsize - linux_dacs[dev].space <
	    	    (linux_advance_samples - 2) *
		    	(OSS_SAMPLEWIDTH * OSSCHPERDEV)))
    	    		goto badsync;
	for (dev=0; dev < linux_nadcs; dev++)
	    if (linux_adcs[dev].space > 3 * OSS_XFERSIZE)
		goto badsync;

	    /* return zero to tell the scheduler we're idle. */
    	return (SENDDACS_NO);
    badsync:
	sys_log_error(ERR_RESYNC);
	oss_doresync();
	return (SENDDACS_NO);
    	
    }

	/* do output */

    timeref = sys_getrealtime();
    for (dev=0; dev < linux_ndacs; dev++)
    {
    	if (linux_dacs[dev].dropcount)
	    linux_dacs[dev].dropcount--;
	else
	{
	    for (i = DACBLKSIZE,  fp1 = sys_soundout + OSSCHPERDEV*DACBLKSIZE*dev,
	    	sp = buf; i--; fp1++, sp += OSSCHPERDEV)
	    {
		for (j=0, fp2 = fp1; j<OSSCHPERDEV; j++, fp2 += DACBLKSIZE)
		{
	    		/* clip, don't wrap */
	    	    int s = *fp2 * 32767.;
		    if (s > 32767) s = 32767;
		    else if (s < -32767) s = -32767;
	    	    sp[j] = s;
		}
	    }
	    linux_dacs_write(linux_dacs[dev].fd, buf, OSS_XFERSIZE);
	    if ((timenow = sys_getrealtime()) - timeref > 0.002)
	    {
	    	if (!oss_oldaudio)
		    sys_log_error(ERR_DACSLEPT);
		else
		    rtnval = SENDDACS_SLEPT;
    	    }
	    timeref = timenow;
    	}
    }
    memset(sys_soundout, 0,
    	linux_ndacs * (sizeof(float) * OSSCHPERDEV* DACBLKSIZE));

    	/* do input */

    for (dev = 0; dev < linux_nadcs; dev++) {

    	linux_adcs_read(linux_adcs[dev].fd, buf, OSS_XFERSIZE);

	if ((timenow = sys_getrealtime()) - timeref > 0.002)
	{
	    if (!oss_oldaudio)
	    	sys_log_error(ERR_ADCSLEPT);
	    else
	    	rtnval = SENDDACS_SLEPT;
    	}
	timeref = timenow;

	for (i = DACBLKSIZE,fp1 = sys_soundin + OSSCHPERDEV*DACBLKSIZE*dev,
	    sp = buf; i--; fp1++,  sp += OSSCHPERDEV) {
	   for (j=0;j<linux_inchannels;j++)
		fp1[j*DACBLKSIZE] = (float)sp[j]*3.051850e-05;
	}	  	  
     }	
     return (rtnval);
}


/* ----------------- audio I/O using the ALSA native API ---------------- */


#ifdef HAVE_ALSA


static int num_cards = 0;

#define DEVICE_CHANNELS 2
/* KWM This should be enough and I am not going to check further down */
#define MAXCARDS  8

static t_oss_sample *alsa_buf;

static int alsa_open_audio(int inchans,int outchans,int srate)
{  
 
  /* KWM Complete rewrite of this function - 11/12/99 */

  /* KWM This is so that we can associate the number of 
     in/out devices with a particular card */
  typedef struct _card {
    int card_num;
    snd_pcm_playback_info_t pinfo[MAXDEV];
    snd_pcm_capture_info_t cinfo[MAXDEV];
    snd_ctl_hw_info_t hwinfo;
    snd_pcm_info_t pcminfo;
    int indev, outdev;
  } alsa_card;

  alsa_card ac[MAXCARDS];
  unsigned int dev;
  int inchannels, outchannels, card, err, maxiochannels, frags = 32;
  
  snd_ctl_t *handle; /* temporary handle */
  snd_pcm_format_t pcm_format;
  snd_pcm_playback_params_t pcm_playback_params;
  snd_pcm_capture_params_t pcm_capture_params;

  inchannels = 0;
  outchannels = 0;
  linux_ndacs = 0;
  linux_nadcs = 0;

 
  num_cards = snd_cards();
  if (num_cards == 0) 
  {
    fprintf(stderr,"PD-ALSA Error: no soundcards found!\n");
  }

  /* Find ALL sound cards and their capabilities */
  for (card = 0; card < num_cards; card++)
  {
    ac[card].indev = 0;
    ac[card].outdev = 0;

    if ((err = snd_ctl_open(&handle, card)) < 0)
    {
      fprintf(stderr,"PD-ALSA Error: control open on card %i failed: %s\n",
	      card, snd_strerror(err));
      break;
    }
    if ((err = snd_ctl_hw_info(handle, &ac[card].hwinfo)) < 0)
    {
      fprintf(stderr,"PD-ALSA Error: getting info on card %i failed: %s\n",
	      card, snd_strerror(err));
      break;
    }

    for (dev = 0; dev < ac[card].hwinfo.pcmdevs; dev++)
    {

      if ((err = snd_ctl_pcm_info(handle, dev, &ac[card].pcminfo)) < 0)
      {
	fprintf(stderr,"PD-ALSA Error: pcm info on card %i failed: %s\n",
		card, snd_strerror(err));
      }
      else
      {
	fprintf(stderr,"PD-ALSA: found the following cards:");
	fprintf(stderr,"  %s: %i\n",ac[card].hwinfo.name, card);
	fprintf(stderr,"  Directions: %s%s%s\n",
	  ac[card].pcminfo.flags & SND_PCM_INFO_PLAYBACK ? "playback " : "",
	  ac[card].pcminfo.flags & SND_PCM_INFO_CAPTURE ? "capture " : "",
	  ac[card].pcminfo.flags & SND_PCM_INFO_DUPLEX ? "duplex " : "");
	fprintf(stderr,"  Playback devices: %i\n", ac[card].pcminfo.playback + 1);
	fprintf(stderr,"  Capture devices: %i\n", ac[card].pcminfo.capture + 1);

	if (ac[card].pcminfo.flags & SND_PCM_INFO_PLAYBACK)
	{
	  if (linux_ndacs <= MAXDEV)
	     {
	    if ((err = snd_ctl_pcm_playback_info(handle, dev, 0, 
		 &ac[card].pinfo[dev])) < 0)
	    {
	      fprintf(stderr,"PD-ALSA: playback info on card %i failed: %s\n",
		      card, snd_strerror(err));
	    }
	    else
	    {
	      fprintf(stderr,"PD-ALSA: found %i playback channels on card %i\n",
		      ac[card].pinfo[dev].max_channels, card);
	      ac[card].outdev += 1;
	      linux_ndacs += 1;
	    }
	  }
	  else
	  {
	    fprintf(stderr,
                    "PD-ALSA: too many dacs - set MAXDEV larger and recompile\n");
	  }
	}
	if (ac[card].pcminfo.flags & SND_PCM_INFO_CAPTURE)
	{
	  if (linux_nadcs <= MAXDEV)
	  {
	    if ((err = snd_ctl_pcm_capture_info(handle, dev, 0,
	         &ac[card].cinfo[dev])) < 0)
	    {
	      fprintf(stderr,"PD-ALSA: capture info on card %i failed: %s\n",
		      card, snd_strerror(err));
	    }
	    else
	    {
	      fprintf(stderr,"PD-ALSA: found %i record channels on card %i\n",
		      ac[card].cinfo[dev].max_channels, card);
	      ac[card].indev += 1;
	      linux_nadcs += 1;
	    }
	  }
	  else
	  {
	    fprintf(stderr,
		    "PD-ALSA: too many adcs - set MAXDEV larger and recompile\n");
	  }
	}
      }
    }
    snd_ctl_close(handle);
  }

  for (card = 0; card < num_cards; card++)
  {
    for (dev = 0; dev < ac[card].hwinfo.pcmdevs; dev++)
    {
    
        if (ac[card].pcminfo.flags & SND_PCM_INFO_PLAYBACK)
	{
	  if ((err = snd_pcm_open(&linux_dacs[dev].handle, card, dev,
				  SND_PCM_OPEN_PLAYBACK)) < 0)
	  {
	    fprintf(stderr,"PD-ALSA Error: audio playback open error on card %d: %s\n",
		card, snd_strerror(err));
	  }
	  else
	  {
	    pcm_format.format = SND_PCM_SFMT_S16_LE ;
	    pcm_format.rate = srate; 
	    pcm_format.channels = ac[card].pinfo[dev].max_channels;
	  
	    if (pcm_format.channels > DEVICE_CHANNELS)
	        fprintf(stderr,
	      "ALSA - too many channels found; recompile pd with higher MAXCH\n");

  	    if( (err = snd_pcm_playback_format(linux_dacs[dev].handle,
	        &pcm_format)) < 0)
	    { 
	        fprintf(stderr,"ERROR pcm_format (DAC) failed: %s\n",
	                snd_strerror(err));
	        return 0;
	    }

	    pcm_playback_params.fragment_size = frags;
	    pcm_playback_params.fragments_max  = -1;
	    pcm_playback_params.fragments_room = 1;

	    if( (err = snd_pcm_playback_params(linux_dacs[dev].handle,
	        &pcm_playback_params))< 0)
	    { 
	        fprintf(stderr,"ERROR pcm_playback_params failed: %s\n", 
	        	      snd_strerror(err));
	        return 0;
	    }
	  
	    outchannels += linux_dacs[dev].channels =  ac[card].pinfo[dev].max_channels;
	  }
	}
	  if (ac[card].pcminfo.flags & SND_PCM_INFO_CAPTURE)
	  {
	    if ((err = snd_pcm_open(&linux_adcs[dev].handle, card, dev,
				    SND_PCM_OPEN_CAPTURE)) < 0)
	    {
	     fprintf(stderr,"PD-ALSA Error: audio capture open error on card %d: %s\n",
		  card, snd_strerror(err));
	    }
	    else
	    {
	      pcm_format.format = SND_PCM_SFMT_S16_LE ; /* 16 bit little endian */
	      pcm_format.rate = srate; 
	      pcm_format.channels = ac[card].cinfo[dev].max_channels;

	      if( (err = snd_pcm_capture_format(linux_adcs[dev].handle,
	          &pcm_format)) < 0)
	      { 
	        fprintf(stderr,"ERROR pcm_format (ADC) failed: %s\n",
	                snd_strerror(err));
	        return 0;
	      }

	      pcm_capture_params.fragment_size = frags;
	      pcm_capture_params.fragments_min = -1;

	      if( (err = snd_pcm_capture_params(linux_adcs[dev].handle,
	          &pcm_capture_params))< 0)
	      { 
	          fprintf(stderr,"ERROR pcm_capture_params failed: %s\n", 
	      	          snd_strerror(err));
	          return 0;
    	      }

	      /* necessary to initialize record */
	      /*if (linux_adcs[dev].handle != NULL)
	      {
	         snd_pcm_read(linux_adcs[dev].handle, alsa_buf,
		      OSS_SAMPLEWIDTH*pcm_format.channels*DACBLKSIZE);
		      }*/
	    }
   
	    inchannels += linux_adcs[dev].channels =  ac[card].cinfo[dev].max_channels;
	  }
    }
  }
  maxiochannels = (inchannels > outchannels ? inchannels : outchannels);
  alsa_buf = (t_oss_sample *)malloc(OSS_SAMPLEWIDTH * DACBLKSIZE * maxiochannels);
  if (!alsa_buf)
  {
    fprintf(stderr, "PD-ALSA Error: out of memory\n");
  }

  linux_setchsr(inchannels, outchannels, srate);
  return (0);
}


void alsa_close_audio(void)
{
     int i;
     for (i=0;i< linux_ndacs;i++)
	  snd_pcm_close(linux_dacs[i].handle);
     for (i=0;i< linux_nadcs;i++)
	  snd_pcm_close(linux_adcs[i].handle);
     if (alsa_buf)
     	free(alsa_buf);
}

//#define DEBUG_WRITE
//#define DEBUG_READ

int alsa_send_dacs(void)
{
     t_oss_sample *fp2;
     float *fp1;
     int i,j,err;  
     int sampstomove = linux_outchannels * DACBLKSIZE;
     struct snd_pcm_playback_status pcm_playback_status;
     struct snd_pcm_capture_status pcm_capture_status;
     struct snd_pcm_playback_info pcm_playback_info;
     struct snd_pcm_capture_info pcm_capture_info;
     int devs=0;
     int outchannels = 0;
     int inchannels = 0;
     int size;

     /* -- write -- */
     for (devs=0;devs<linux_ndacs;devs++) {
	  int channels = linux_dacs[devs].channels;
	  if((err = snd_pcm_playback_status(linux_dacs[devs].handle,
	    &pcm_playback_status)) < 0) {
	       fprintf(stderr,"ERROR pcm_playback_status %s\n",snd_strerror(err));
	       continue;
	  }
	  
	  /*
	  if(pcm_playback_status.count < OSS_BYTESPERCHAN*channels){
#ifdef DEBUG_WRITE
	       fprintf(stderr,"\nplayback_status.count %d bytes -> idle\n",
		       pcm_playback_status.count);
#endif 
	       return 0;
	  }
	  */

	  for (i=DACBLKSIZE,fp1=sys_soundout+outchannels*DACBLKSIZE, 
		    fp2 = alsa_buf;i--;fp1++,  fp2 += channels) {
	       for (j=0;j<channels;j++)
		    fp2[j] = (int) (32767.0f*fp1[DACBLKSIZE*j]);
	  }	  

#ifdef DEBUG_WRITE
	       fprintf(stderr,"try to write %d bytes of %d ...",
		       OSS_BYTESPERCHAN*channels,pcm_playback_status.count);
#endif
	  size = snd_pcm_write(linux_dacs[devs].handle, alsa_buf,
	    	OSS_BYTESPERCHAN*channels);
#ifdef DEBUG_WRITE
	       fprintf(stderr,"%d written\n",
		       OSS_BYTESPERCHAN*channels,size);
#endif
	  outchannels+=channels;
     }

     /* -- read -- */

     for (devs=0;devs<linux_nadcs;devs++) {
          int border = linux_inchannels*OSS_BYTESPERCHAN*(linux_advance_samples+1);
	  int channels = linux_adcs[devs].channels;

	  if((err = snd_pcm_capture_status(linux_adcs[devs].handle,
	    &pcm_capture_status)) < 0)
	       continue;

	  
	  if( pcm_capture_status.count > border){
	       while(pcm_capture_status.count > border){
#ifdef DEBUG_READ
		    fprintf(stderr,"byte = %d, border = %d \n",
			    pcm_capture_status.count,border);
#endif
//		    post("drop ADC buf");
		    if((err = snd_pcm_capture_status(linux_adcs[devs].handle,
		    	&pcm_capture_status)) < 0)
		    {
			fprintf(stderr,"ERROR pcm_capture_status failed: %s\n",
		    	    snd_strerror(err));
			return 0;
		    }
		    snd_pcm_read(linux_adcs[devs].handle, alsa_buf,
		    	OSS_BYTESPERCHAN*channels);
	       }
	       
	  }
	  if ( (unsigned)pcm_capture_status.count >= (OSS_BYTESPERCHAN*channels))
	  {
#ifdef DEBUG_READ
	       fprintf(stderr,"try to read %d bytes of %d ...",
		       OSS_BYTESPERCHAN*channels,pcm_capture_status.count);
#endif
	       size = snd_pcm_read(linux_adcs[devs].handle,alsa_buf,
	            OSS_BYTESPERCHAN*channels);
#ifdef DEBUG_READ
	       fprintf(stderr,"%d read\n",size);
#endif
	  }
	  else
	  {
#ifdef DEBUG_READ
	       fprintf(stderr,"only %d bytes ready to read\n",
	            pcm_capture_status.count );
#endif
	       memset(alsa_buf, 0, OSS_BYTESPERCHAN*channels);
	  } 
     
	  for (i = DACBLKSIZE,fp1 = sys_soundin+inchannels*DACBLKSIZE,
		    fp2 = alsa_buf; i--;fp1++,  fp2 += channels) {
	       for (j=0;j<channels;j++)
		    fp1[j*DACBLKSIZE] = (float)fp2[j]/32767.;
	       /*3.051850e-05;*/
	  }
	  inchannels+=channels;
     }
     memset(sys_soundout, 0, sampstomove*sizeof(float));
     return (1);
}

#endif /* HAVE_ALSA */


/***************************************************
 *  Code using the RME_9652 API
 */ 

	 /* trying native device for future use of native memory map:
		 because of busmaster if you dont use the dac, you ont need 
		 CPU Power und also no nearly no CPU-Power is used in device
	 */
/* since always all DAs and ADs are synced (else they wouldnt work)
	we use linux_dacs[0], linux_adcs[0]
*/

#ifdef HAVE_RME

#define RME9652_MAX_CHANNELS 26

#define RME9652_CH_PER_NATIVE_DEVICE 1

static int rme9652_dac_devices[RME9652_MAX_CHANNELS];
static int rme9652_adc_devices[RME9652_MAX_CHANNELS];

static char rme9652_dsp_dac[] = "/dev/rme9652/C0da%d"; 
static char rme9652_dsp_adc[] = "/dev/rme9652/C0ad%d"; 

static int num_of_rme9652_dac = 0;
static int num_of_rme9652_adc = 0;

static int rme_soundindevonset = 1;
static int rme_soundoutdevonset = 1;

void rme_soundindev(int which)
{
    rme_soundindevonset = which;
}

void rme_soundoutdev(int which)
{
    rme_soundoutdevonset = which;
}

void rme9652_configure(int dev, int fd,int srate, int dac) {
  int orig, param, nblk;
  audio_buf_info ainfo;
  orig = param = srate; 

  /* samplerate */

  fprintf(stderr,"RME9652: configuring %d, fd=%d, sr=%d\n, dac=%d\n",
			 dev,fd,srate,dac);

  if (ioctl(fd,SNDCTL_DSP_SPEED,&param) == -1)
	 fprintf(stderr,"RME9652: Could not set sampling rate for device\n");
  else if( orig != param )
	 fprintf(stderr,"RME9652: sampling rate: wanted %d, got %d\n",
				orig, param );				
  
  // setting the correct samplerate (could be different than expected)     
  srate = param;


  /* setting resolution */

  /* use ctrlpanel to change, experiment, channels 1 */

  orig = param = AFMT_S16_LE;
  if (ioctl(fd,SNDCTL_DSP_SETFMT,&param) == -1)
	 fprintf(stderr,"RME9652: Could not set DSP format\n");
  else if( orig != param )
	 fprintf(stderr,"RME9652: DSP format: wanted %d, got %d\n",orig, param );

  /* setting channels */
  orig = param = RME9652_CH_PER_NATIVE_DEVICE;

  if (ioctl(fd,SNDCTL_DSP_CHANNELS,&param) == -1)
	 fprintf(stderr,"RME9652: Could not set channels\n");
  else if( orig != param )
	 fprintf(stderr,"RME9652: num channels: wanted %d, got %d\n",orig, param );

  if (dac)
    {

	/* use "free space" to learn the buffer size.  Normally you
		should set this to your own desired value; but this seems not
		to be implemented uniformly across different sound cards.  LATER
		we should figure out what to do if the requested scheduler advance
		is greater than this buffer size; for now, we just print something
		out.  */

	if( ioctl(linux_dacs[0].fd, SOUND_PCM_GETOSPACE,&ainfo) < 0 )
	  fprintf(stderr,"RME: ioctl on output device %d failed",dev);

	linux_dacs[0].bufsize = ainfo.bytes;

	fprintf(stderr,"RME: ioctl SOUND_PCM_GETOSPACE says %d buffsize\n",
			  linux_dacs[0].bufsize);


	if (linux_advance_samples * (RME_SAMPLEWIDTH * RME9652_CH_PER_NATIVE_DEVICE) 
		 > linux_dacs[0].bufsize - RME_BYTESPERCHAN)
	  {
		 fprintf(stderr,
		    "RME: requested audio buffer size %d limited to %d\n",
		    linux_advance_samples 
		    * (RME_SAMPLEWIDTH * RME9652_CH_PER_NATIVE_DEVICE),
		    linux_dacs[0].bufsize);
    	    	linux_advance_samples =
		    (linux_dacs[0].bufsize - RME_BYTESPERCHAN) 
		    / (RME_SAMPLEWIDTH *RME9652_CH_PER_NATIVE_DEVICE);
	  }
    }
}

 
int rme9652_open_audio(int inchans, int outchans,int srate)
{  
    int orig;
    int tmp;
    int inchannels = 0,outchannels = 0;
    char devname[20];
    int i;
    char buf[RME_SAMPLEWIDTH*RME9652_CH_PER_NATIVE_DEVICE*DACBLKSIZE];
    int num_devs = 0;
    audio_buf_info ainfo;

    linux_nadcs = linux_ndacs = 0;

    post("RME open");
    /* First check if we can  */
    /* open the write ports	*/

    for (num_devs=0; outchannels < outchans; num_devs++)
    {
	int channels = RME9652_CH_PER_NATIVE_DEVICE;

	sprintf(devname, rme9652_dsp_dac, num_devs + rme_soundoutdevonset);
	if ((tmp = open(devname,O_WRONLY)) == -1) 
	{
	    DEBUG(fprintf(stderr,"RME9652: failed to open %s writeonly\n",
    	    	devname);)
	    break;
	}
	DEBUG(fprintf(stderr,"RME9652: out device Nr. %d (%d) on %s\n",
    	    linux_ndacs+1,tmp,devname);)

	if (outchans > outchannels)
	{
	    rme9652_dac_devices[linux_ndacs] = tmp;
	    linux_ndacs++;
	    outchannels += channels;
	}
	else close(tmp);
    }
    if( linux_ndacs > 0)
	linux_dacs[0].fd =  rme9652_dac_devices[0];

    /* Second check if we can  */
    /* open the read ports	*/

    for (num_devs=0; inchannels < inchans; num_devs++)
    {
	int channels = RME9652_CH_PER_NATIVE_DEVICE;

	sprintf(devname, rme9652_dsp_adc, num_devs+rme_soundindevonset);

	if ((tmp = open(devname,O_RDONLY)) == -1) 
	{
	    DEBUG(fprintf(stderr,"RME9652: failed to open %s readonly\n",
	    	devname);)
	    break;
	}
	DEBUG(fprintf(stderr,"RME9652: in device Nr. %d (%d) on %s\n",
	    linux_nadcs+1,tmp,devname);)

	if (inchans > inchannels)
	{
	    rme9652_adc_devices[linux_nadcs] = tmp;
	    linux_nadcs++;
	    inchannels += channels;
	}
	else
	     close(tmp);
    }
    if(linux_nadcs > 0)
	linux_adcs[0].fd = rme9652_adc_devices[0];

    /* configure soundcards */

	 rme9652_configure(0, linux_adcs[0].fd,srate, 0);
	 rme9652_configure(0, linux_dacs[0].fd,srate, 1);
    
	 /* We have to do a read to start the engine. This is 
		 necessary because sys_send_dacs waits until the input
		 buffer is filled and only reads on a filled buffer.
		 This is good, because it's a way to make sure that we
		 will not block */
	 
    if (linux_nadcs)
    {
	fprintf(stderr,("RME9652: starting read engine ... "));


	for (num_devs=0; num_devs < linux_nadcs; num_devs++) 
	  read(rme9652_adc_devices[num_devs],
	    buf, RME_SAMPLEWIDTH* RME9652_CH_PER_NATIVE_DEVICE*
		DACBLKSIZE);


	for (num_devs=0; num_devs < linux_ndacs; num_devs++) 
	  write(rme9652_dac_devices[num_devs],
	    buf, RME_SAMPLEWIDTH* RME9652_CH_PER_NATIVE_DEVICE*
		DACBLKSIZE);

	if(linux_ndacs)
	  ioctl(rme9652_dac_devices[0],SNDCTL_DSP_SYNC);

	fprintf(stderr,"done\n");
    }

    linux_setchsr(linux_nadcs,linux_ndacs,44100);

    num_of_rme9652_dac = linux_ndacs;
    num_of_rme9652_adc = linux_nadcs;

    if(linux_ndacs)linux_ndacs=1;
    if(linux_nadcs)linux_nadcs=1;

    /* trick RME9652 behaves as one device fromread write pointers */
    return (0);
}

void rme9652_close_audio( void)
{
     int i;
     for (i=0;i<num_of_rme9652_dac;i++)
		 close(rme9652_dac_devices[i]);

     for (i=0;i<num_of_rme9652_adc;i++)
          close(rme9652_adc_devices[i]);
}


/* query audio devices for "available" data size. */
/* not needed because oss_calcspace does the same */
static int rme9652_calcspace(void)
{
    audio_buf_info ainfo;


	 /* one for all */

	 if (ioctl(linux_dacs[0].fd, SOUND_PCM_GETOSPACE,&ainfo) < 0)
		fprintf(stderr,"RME9652: calc ioctl SOUND_PCM_GETOSPACE on output device fd %d failed\n",
				  linux_dacs[0].fd);
	 linux_dacs[0].space = ainfo.bytes;

	 if (ioctl(linux_adcs[0].fd, SOUND_PCM_GETISPACE,&ainfo) < 0)
		fprintf(stderr, 
				  "RME9652: calc ioctl SOUND_PCM_GETISPACE on input device fd %d failed\n", 
				  rme9652_adc_devices[0]);
	 linux_adcs[0].space = ainfo.bytes;

	 return 1;
}

/* this call resyncs audio output and input which will cause discontinuities
in audio output and/or input. */ 

static void rme9652_doresync( void)
{	
  if(linux_ndacs)
	 ioctl(rme9652_dac_devices[0],SNDCTL_DSP_SYNC);
}

static int mycount =0;
#ifndef INT32_MAX
#define INT32_MAX 0x7fffffff
#endif

#define CLIP(x) (((x)>INT32_MAX)?INT32_MAX:((x) < -INT32_MAX)?-INT32_MAX:(x))

int rme9652_send_dacs(void)
{
    float *fp;
    long fill;
    int i, j, dev;
	 /* the maximum number of samples we should have in the ADC buffer */
    t_rme_sample buf[RME9652_CH_PER_NATIVE_DEVICE*DACBLKSIZE], *sp;

    double timeref, timenow;

    mycount++;

    if (!linux_nadcs && !linux_ndacs) return (0);

    rme9652_calcspace();
    	 
	/* do output */
	
   timeref = sys_getrealtime();

    if(linux_ndacs){

    	if (linux_dacs[0].dropcount)
    	    linux_dacs[0].dropcount--;
    	else{
	    /* fprintf(stderr,"output %d\n", linux_outchannels);*/

	    for(j=0;j<linux_outchannels;j++){

	      t_rme_sample *a,*b,*c,*d;
	      float *fp1,*fp2,*fp3,*fp4;

	      fp1 = sys_soundout + j*DACBLKSIZE-4;
	      fp2 = fp1 + 1;
	      fp3 = fp1 + 2;
	      fp4 = fp1 + 3;
	      a = buf-4;
	      b=a+1;
	      c=a+2;
	      d=a+3; 

	      for (i = DACBLKSIZE>>2;i--;)
		 { 
		    float s1 =  *(fp1+=4) * INT32_MAX;
		    float s2 =  *(fp2+=4) * INT32_MAX;
		    float s3 =  *(fp3+=4) * INT32_MAX;
		    float s4 =  *(fp4+=4) * INT32_MAX;

		    *(a+=4) = CLIP(s1);
		    *(b+=4) = CLIP(s2);
		    *(c+=4) = CLIP(s3);
		    *(d+=4) = CLIP(s4);
    	    	} 

		linux_dacs_write(rme9652_dac_devices[j],buf,RME_BYTESPERCHAN);				
	    }
    	}

	  if ((timenow = sys_getrealtime()) - timeref > 0.002)
		 sys_log_error(ERR_DACSLEPT);
	  timeref = timenow;
    }

    memset(sys_soundout, 0,
    	linux_outchannels * (sizeof(float) * DACBLKSIZE));

	 /* do input */

    if(linux_nadcs) {

    	for(j=0;j<linux_inchannels;j++){

    	    linux_adcs_read(rme9652_adc_devices[j], buf, RME_BYTESPERCHAN);
		  
	    if ((timenow = sys_getrealtime()) - timeref > 0.002)
		 sys_log_error(ERR_ADCSLEPT);
	    timeref = timenow;
	    {
		t_rme_sample *a,*b,*c,*d;
		float *fp1,*fp2,*fp3,*fp4;

		fp1 = sys_soundin + j*DACBLKSIZE-4;
		fp2 = fp1 + 1;
		fp3 = fp1 + 2;
		fp4 = fp1 + 3;
		a = buf-4;
		b=a+1;
		c=a+2;
		d=a+3; 

		for (i = (DACBLKSIZE>>2);i--;)
		{ 
		    *(fp1+=4) = *(a+=4) * (float)(1./INT32_MAX);
		    *(fp2+=4) = *(b+=4) * (float)(1./INT32_MAX);
		    *(fp3+=4) = *(c+=4) * (float)(1./INT32_MAX);
		    *(fp4+=4) = *(d+=4) * (float)(1./INT32_MAX);
		}
	    }  
    	}	
    }
    /*	fprintf(stderr,"ready \n");*/

    return (1);
}

#endif /* HAVE_RME */
