/*
 * The contents of this file are subject to the AOLserver Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://aolserver.lcs.mit.edu/.
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is AOLserver Code and related documentation
 * distributed by AOL.
 * 
 * The Initial Developer of the Original Code is America Online,
 * Inc. Portions created by AOL are Copyright (C) 1999 America Online,
 * Inc. All Rights Reserved.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU General Public License (the "GPL"), in which case the
 * provisions of GPL are applicable instead of those above.  If you wish
 * to allow use of your version of this file only under the terms of the
 * GPL and not to allow others to use your version of this file under the
 * License, indicate your decision by deleting the provisions above and
 * replace them with the notice and other provisions required by the GPL.
 * If you do not delete the provisions above, a recipient may use your
 * version of this file under either the License or the GPL.
 */

/*
 * nsunix.c --
 *
 *	This module is a communication driver for HTTP that initially
 *	uses a unix domain socket.  It works together with the nsvhr.so
 *	module.  Initially, this driver accepts a connection on a unix
 *	domain socket and receives a message sent by nsvhr.so.  The
 *	message contains the passed file descriptor of the connection
 *	socket and the initial data from that connection socket.  The
 *	rest of the data is read from the passed file descriptor after
 *	the initial data has been consumed from the message.
 */

static const char *RCSID = "@(#) $Header: /cvsroot/aolserver/aolserver3/nsunix/nsunix.c,v 1.1.1.1 2000/03/17 07:11:34 kriston Exp $, compiled: " __DATE__ " " __TIME__;

#include <sys/types.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <sys/un.h>
#include <strings.h>

#include "ns.h"

#if defined(__linux) || defined(__FreeBSD__) || defined (__unixware) || defined(__APPLE__) || defined(__OpenBSD__)
#define HAS_CMSGHDR
#endif

#define MODULES	    "modules"
#define DRIVER_NAME "nsunix"
#if !defined(MAX)
#define MAX(a,b)    (((a) > (b)) ? (a) : (b))
#endif
#if !defined(MIN)
#define MIN(a,b)    (((a) < (b)) ? (a) : (b))
#endif
#define SELECT_TIMEOUT 30
#define HOSTNAME    "Hostname"
#define PORT        "Port"
#define SOCKETFILE  "SocketFile"
#define BUFFERSIZE  1024

#ifdef HAS_CMSGHDR
/*
 * This wraps a cmsghdr and provides space for a file descriptor.
 * It's the modern way of passing file descriptors.  NOTE: this assumes
 * that member cmsg is aligned with the beginning of struct my_cmsghdr.
 */

typedef struct {
    struct cmsghdr cmsg;
    int            sock;
} my_cmsghdr;
#endif

/*
 * This context is passed to many of the callbacks. It contains
 * information that is relevant at the driver level (as opposed
 * to the connection level).
 */

typedef struct {
    char       *location;       /* e.g., http://www.foo.com:80             */
    char       *host;           /* e.g., www.foo.com                       */
    int         port;           /* e.g., 80                                */
    char       *udsFilename;	/* Unix domain socket file name		   */
    int         listenSocket;   /* do accept() calls on this socket        */
    Ns_Driver   driver;         /* the Ns_Driver handle for this driver    */
    int         stopped;        /* 0: drv running; 1: stopping; 2: stopped */
    Ns_Mutex    lock;           /* lock around 'stopped' variable          */
} DriverContext;

/*
 * This context is passed to many of the callbacks. It contains
 * information that is relevant at the connection level.
 */

typedef struct {
    char      	  *buffer;	   /* this is the read 			   */
    char	  *base;	   /* 		       input		   */
    int            count;	   /* 			     buffer	   */
    int            unix_sock;	   /* unix domain socket		   */
    int            sock;	   /* socket that was passed via unix_sock */
    char           remoteAddr[16]; /* IP address of remote peer            */
    int            remotePort;	   /* port of remote peer		   */
    DriverContext *drvCtxPtr;
} ConnContext;

/*
 * Local functions defined in this file.
 */

static int   DrvPeerPort(void *vconnCtxPtr);
static int   DrvConnectionFd(void *vconnCtxPtr);
static void *DrvDetach(void *vconnCtxPtr);
static int   DrvSendFile(void *vconnCtxPtr, char *filename);
static int   DrvSendFd(void *vconnCtxPtr, int fd, int nsend);
static int   DrvPort(void *vconnCtxPtr);
static char *DrvHost(void *vconnCtxPtr);
static char *DrvLocation(void *vconnCtxPtr);
static char *DrvPeer(void *vconnCtxPtr);
static void  DrvFree(void *vconnCtxPtr);
static int   DrvClose(void *vconnCtxPtr);
static int   DrvWrite(void *vconnCtxPtr, void *vbufPtr, int toWrite);
static int   DrvRead(void *vconnCtxPtr, void *vbufPtr, int toread);
static int   DrvInit(void *vconnCtxPtr);
static void  DrvStop(void *vdriverCtxPtr);
static int   DrvAccept(void *drvCtxPtr, void **connCtxPtrPtr);
static int   DrvStart(char *hServer, char *hDriver, void **driverCtxPtrPtr);
static char *DrvName(void *drvCtxPtr);
static int   GetDataFromUDS(ConnContext *connCtxPtr);
static int   Listen(char *path);
static int   Accept(int lsock, struct sockaddr *addrPtr, int *addrlenPtr);
static int   ConnWait(ConnContext *connCtxPtr, int write);

/*
 * This null-terminated list of driver callbacks is passed
 * to Ns_RegisterDriver.
 */

static Ns_DrvProc drvProcs[] = {
    { Ns_DrvIdName,         (void *) DrvName },
    { Ns_DrvIdStart,        (void *) DrvStart },
    { Ns_DrvIdAccept,       (void *) DrvAccept },
    { Ns_DrvIdStop,         (void *) DrvStop },
    { Ns_DrvIdInit,         (void *) DrvInit },
    { Ns_DrvIdRead,         (void *) DrvRead },
    { Ns_DrvIdWrite,        (void *) DrvWrite },
    { Ns_DrvIdClose,        (void *) DrvClose },
    { Ns_DrvIdFree,         (void *) DrvFree },
    { Ns_DrvIdPeer,         (void *) DrvPeer },
    { Ns_DrvIdLocation,     (void *) DrvLocation },
    { Ns_DrvIdHost,         (void *) DrvHost },
    { Ns_DrvIdPort,         (void *) DrvPort },
    { Ns_DrvIdSendFd,       (void *) DrvSendFd },
    { Ns_DrvIdSendFile,     (void *) DrvSendFile },
    { Ns_DrvIdDetach,       (void *) DrvDetach },
    { Ns_DrvIdConnectionFd, (void *) DrvConnectionFd },
    { Ns_DrvIdPeerPort,     (void *) DrvPeerPort },
    
    /*
     * SSL-specific callbacks are not supported here.
     */
    
    { 0,                    NULL }
};

/*
 * Global variables.
 */

int Ns_ModuleVersion = 1;

/*
 * Static variables defined in this file.
 */

static Ns_ModLogHandle unixModLogHandle;

/*
 *==========================================================================
 * Exported functions
 *==========================================================================
 */


/*
 *----------------------------------------------------------------------
 *
 * Ns_ModuleInit --
 *
 *	Initialize the comm driver module. This listens on a port and 
 *	registers the driver callbacks. 
 *
 * Results:
 *	NS_OK/NS_ERROR 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

int
Ns_ModuleInit(char *server, char *module)
{
    DriverContext *drvCtxPtr;
    int            status = NS_ERROR;
    char 	  *path;
    char	  *udsFilename;
    Ns_DString     ds;

    Ns_DStringInit(&ds);

    Ns_ModLogRegister(module, &unixModLogHandle);

    drvCtxPtr = ns_malloc(sizeof(DriverContext));

    path = Ns_ConfigGetPath(server, module, NULL);
    if (path == NULL) {
        Ns_ModLog(Error, unixModLogHandle,
                  "no config path [ns/server/%s/module/%s]",
                  server, module);
        goto done;
    }

    drvCtxPtr->host = Ns_ConfigGetValue(path, HOSTNAME);
    if (drvCtxPtr->host == NULL) {
	drvCtxPtr->host = ns_strdup(Ns_InfoHostname());
        Ns_ModLog(Warning, unixModLogHandle, "missing %s parameter, "
	         "using %s: %s", HOSTNAME, HOSTNAME, drvCtxPtr->host);
    }

    if (Ns_ConfigGetInt(path, PORT, &drvCtxPtr->port) != NS_TRUE) {
	drvCtxPtr->port = 80;
        Ns_ModLog(Warning, unixModLogHandle, "missing %s parameter, "
	         "using %s: %d", PORT, PORT, drvCtxPtr->port);
    }

    udsFilename = Ns_ConfigGetValue(path, SOCKETFILE);
    if (udsFilename == NULL) {
	Ns_DString ds;

	Ns_DStringInit(&ds);
	Ns_DStringVarAppend(&ds, server, ".", module, NULL);
	udsFilename = Ns_DStringExport(&ds);
	Ns_ModLog(Warning, unixModLogHandle, "missing %s parameter, "
		  "using %s: %s", SOCKETFILE, SOCKETFILE, udsFilename);
	Ns_DStringFree(&ds);
    }
    Ns_DStringVarAppend(&ds, MODULES, "/", DRIVER_NAME,
			"/", udsFilename, NULL);
    drvCtxPtr->udsFilename = ns_strdup(Ns_DStringValue(&ds));

    Ns_DStringTrunc(&ds, 0);
    drvCtxPtr->location = ns_strdup(Ns_DStringPrintf(&ds, "http://%s:%d/",
				 drvCtxPtr->host, drvCtxPtr->port));

    drvCtxPtr->stopped  = 0;
    
    drvCtxPtr->driver = Ns_RegisterDriver(server, module, drvProcs, drvCtxPtr);
    status = NS_OK;
    
 done:
    Ns_DStringFree(&ds);
    return status;
}

/*
 *==========================================================================
 * Comm Driver Callbacks
 *==========================================================================
 */


/*
 *----------------------------------------------------------------------
 *
 * Return the name of this driver; e.g. nssock, nsssl --
 *
 *	Return the name of the driver, e.g. nssock, nsssl. 
 *
 * Results:
 *	The name of this driver. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static char *
DrvName(void *driverCtxPtr)
{
    DriverContext *drvCtxPtr = (DriverContext *) driverCtxPtr;

    return DRIVER_NAME;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvStart --
 *
 *	Optionally, set the driver context; also listen on the socket. 
 *
 * Results:
 *	NS_OK/NS_ERROR 
 *
 * Side effects:
 *	This may set *ctxDriverPtrPtr, but it has already been
 *	set by the call to Ns_RegisterDriver.
 *
 *----------------------------------------------------------------------
 */

static int
DrvStart(char *hServer, char *hDriver, void **driverCtxPtrPtr)
{
    DriverContext *drvCtxPtr = (DriverContext *) *driverCtxPtrPtr;
    int            status = NS_ERROR;
    
    drvCtxPtr->listenSocket = Listen(drvCtxPtr->udsFilename);
    if (drvCtxPtr->listenSocket == -1) {
	Ns_ModLog(Error, unixModLogHandle, "could not listen: %s.",
		  strerror(errno));
    } else {
	status = NS_OK;
    }
    
    return status;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvAccept --
 *
 *	Block on a socket, accept a connection, allocate a connection 
 *	context and fill it up.
 *
 * Results:
 *	NS_OK/NS_ERROR/NS_SHUTDOWN
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static int
DrvAccept(void *vdrvCtxPtr, void **vconnCtxPtrPtr)
{
    DriverContext *drvCtxPtr = (DriverContext *) vdrvCtxPtr;
    ConnContext   *connCtxPtr = *((ConnContext **) vconnCtxPtrPtr);
    int            fd;
    int            status = NS_ERROR;
    
    /*
     * Do the malloc before the accept to speed things up a bit.
     */
    
    connCtxPtr = ns_malloc(sizeof(ConnContext));
    connCtxPtr->sock = -1;
    connCtxPtr->buffer = (char *) ns_malloc(BUFFERSIZE + 1);
    assert(connCtxPtr->buffer != NULL);
    connCtxPtr->base = connCtxPtr->buffer;
    connCtxPtr->count = 0;

    for (;;) {
	int                len;
	struct sockaddr_in sa;
	
	len = sizeof(sa);
	
	Ns_ModLog(Debug, unixModLogHandle, "accepting connections on unix "
		  "domain socket...");
	
	connCtxPtr->unix_sock = Accept(drvCtxPtr->listenSocket,
				     (struct sockaddr *) &sa, &len);
	Ns_ModLog(Debug, unixModLogHandle, "accepted a new connection");
	if (connCtxPtr->unix_sock >= 0) {
	    strcpy(connCtxPtr->remoteAddr, ns_inet_ntoa(sa.sin_addr));
	    connCtxPtr->remotePort = ntohs(sa.sin_port);
	    connCtxPtr->drvCtxPtr = drvCtxPtr;
	    *vconnCtxPtrPtr = (void *) connCtxPtr;
	    Ns_ModLog(Debug, unixModLogHandle,
		      "valid connection received, unix domain socket fd: %d",
		      connCtxPtr->unix_sock);
	    status = NS_OK;
	    break;
	} else if (errno != EWOULDBLOCK) {
	    Ns_ModLog(Error, unixModLogHandle, "accept returned error: %s",
		      strerror(errno));
	    status = NS_ERROR;
	    break;
	}
    }

    Ns_MutexLock(&drvCtxPtr->lock);
    if (drvCtxPtr->stopped != 0) {
	status = NS_SHUTDOWN;
	close(connCtxPtr->unix_sock);
    }
    Ns_MutexUnlock(&drvCtxPtr->lock);

    if (status != NS_OK) {
	ns_free(connCtxPtr->buffer);
	connCtxPtr->buffer = NULL;
	ns_free(connCtxPtr);
    }

    return status;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvStop --
 *
 *	Stop the accept callback because the server is shutting down. 
 *
 * Results:
 *	None. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static void
DrvStop(void *vdriverCtxPtr)
{
    DriverContext *drvCtxPtr = (DriverContext *) vdriverCtxPtr;
    struct sockaddr_un  addr;
    int                 unix_sock;
    int                 addr_len;

    Ns_MutexLock(&drvCtxPtr->lock);
    drvCtxPtr->stopped = 1;
    Ns_MutexUnlock(&drvCtxPtr->lock);
    
    /*
     * Force the accept thread to break out of the blocking accept call.
     */

    unix_sock = socket(AF_UNIX, SOCK_STREAM, 0);
    if (unix_sock < 0) {
        Ns_ModLog(Error, unixModLogHandle, "could not create socket: %s.",
                  strerror(errno));
	return;
    }

    bzero((char *) &addr, sizeof(addr));
    strcpy(addr.sun_path, drvCtxPtr->udsFilename);
    addr.sun_family = AF_UNIX;
    addr_len = sizeof(addr.sun_family) + strlen(addr.sun_path);
    if (connect(unix_sock, (struct sockaddr *) &addr, addr_len) < 0) {
        Ns_ModLog(Error, unixModLogHandle, "could not connect to unix:%s: %s.",
                  drvCtxPtr->udsFilename, strerror(errno));
    } else {
	close(unix_sock);
	close(drvCtxPtr->listenSocket);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * DrvInit --
 *
 *	This is called when a new connection is made. It's not very
 *	useful most of the time. It's also run when a keepalive connection
 *	gets a new request.
 *
 * Results:
 *	NS_OK/NS_ERROR 
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
DrvInit(void *vconnCtxPtr)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;
    
    Ns_ModLog(Debug, unixModLogHandle, "new connection");
    return NS_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvRead --
 *
 *	Read 'toread' bytes into the buffer. 
 *
 * Results:
 *	# bytes read, 0 on EOF, -1 on error. 
 *
 * Side effects:
 *	Read from socket. 
 *
 *----------------------------------------------------------------------
 */

static int
DrvRead(void *vconnCtxPtr, void *vbufPtr, int toread)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;
    char        *buffer = (char *) vbufPtr;
    int          numRead = 0;

    while (toread > 0) {
	int toCopy;

        if (connCtxPtr->count > 0) {
	    toCopy = MIN(toread, connCtxPtr->count);
            memcpy(buffer, connCtxPtr->base, toCopy);
            connCtxPtr->base += toCopy;
            connCtxPtr->count -= toCopy;
            toread -= toCopy;
            numRead += toCopy;
            buffer += toCopy;
        }

        if (toread > 0) {
	    if (connCtxPtr->unix_sock != -1) {
		if (GetDataFromUDS(connCtxPtr) != NS_OK) {
		    return -1;
		}
	    } else if (connCtxPtr->sock != -1) {
		connCtxPtr->base = connCtxPtr->buffer;
		connCtxPtr->count = recv(connCtxPtr->sock, connCtxPtr->buffer,
					 BUFFERSIZE, 0);
		Ns_ModLog(Debug, unixModLogHandle, "received %d bytes on "
			  "socket fd: %d", numRead, connCtxPtr->sock);
		if (connCtxPtr->count == -1 && errno == EWOULDBLOCK &&
		    ConnWait(connCtxPtr, 0)) {
		    
		    connCtxPtr->count = recv(connCtxPtr->sock,
					   connCtxPtr->buffer, BUFFERSIZE, 0);
		}

		if (connCtxPtr->count <= 0) {
		    return -1;
		}
	    }
        }
    }

    return numRead;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvWrite --
 *
 *	Write 'towrite' bytes from the buffer to the socket. 
 *
 * Results:
 *	# bytes written, 0 on EOF, -1 on error. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static int
DrvWrite(void *vconnCtxPtr, void *vbufPtr, int toWrite)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;
    int numWrite;

    numWrite = write(connCtxPtr->sock, vbufPtr, toWrite);
    if (numWrite == -1 && errno == EWOULDBLOCK &&
	ConnWait(connCtxPtr, 1)) {

	numWrite = write(connCtxPtr->sock, vbufPtr, toWrite);
    }

    return numWrite;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvClose --
 *
 *	Close the connection (but don't free any memory yet). 
 *
 * Results:
 *	See close(2). 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static int
DrvClose(void *vconnCtxPtr)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;
    int status;

    close(connCtxPtr->unix_sock);
    status = close(connCtxPtr->sock);
    connCtxPtr->sock = -1;

    return status;

}


/*
 *----------------------------------------------------------------------
 *
 * DrvFree --
 *
 *	Free all memory associated with a connection. 
 *
 * Results:
 *	None. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static void
DrvFree(void *vconnCtxPtr)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;

    ns_free(connCtxPtr->buffer);
    connCtxPtr->buffer = NULL;
    ns_free(connCtxPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * DrvPeer --
 *
 *	Return the address of the peer. 
 *
 * Results:
 *	A string address--it might be something other than an IP 
 *	address in a wacky driver, but that might break bad code. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static char *
DrvPeer(void *vconnCtxPtr)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;
    
    return connCtxPtr->remoteAddr;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvLocation --
 *
 *	Return the base URL of this driver. 
 *
 * Results:
 *	A string like "http://www.foo.bar:80/" 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static char *
DrvLocation(void *vdrvCtxPtr)
{
    DriverContext *drvCtxPtr = (DriverContext *) vdrvCtxPtr;

    return drvCtxPtr->location;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvHost --
 *
 *	Return the hostname I'm listening on. 
 *
 * Results:
 *	A hostname like "localhost". 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static char *
DrvHost(void *vconnCtxPtr)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;

    return connCtxPtr->drvCtxPtr->host;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvPort --
 *
 *	Return the port we're listening on. 
 *
 * Results:
 *	A port, such as 80. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static int
DrvPort(void *vconnCtxPtr)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;

    return connCtxPtr->drvCtxPtr->port;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvSendFd --
 *
 *	Write 'nsend' bytes from 'fd' to the connection. 
 *
 * Results:
 *	# bytes written, 0 on EOF, -1 on error. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static int
DrvSendFd(void *vconnCtxPtr, int fd, int nsend)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;
    int          towrite;
    char        *mem;
    int          n;
    int          offset;
    int          written;
    
    mem = (char *) mmap(NULL, nsend, PROT_READ, MAP_PRIVATE, fd, 0);
    if (mem == NULL) {
	return -1;
    }

    offset = 0;
    towrite = nsend;
    
    while (towrite > 0) {
	n = write(connCtxPtr->sock, mem+offset, towrite);
	if (n <= 0) {
	    written = n;
	    break;
	}
	written += n;
	offset += n;
	towrite -= n;
    }
    munmap(mem, nsend);

    return written;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvSendFile --
 *
 *	Copy a file to the connection 
 *
 * Results:
 *	# bytes written, 0=EOF, -1=error 
 *
 * Side effects:
 *	None 
 *
 *----------------------------------------------------------------------
 */

static int
DrvSendFile(void *vconnCtxPtr, char *filename)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;
    struct stat  statbuf;
    int          fd;
    int          retval;
    
    if (stat(filename, &statbuf) == -1) {
	return -1;
    }
    fd = open(filename, O_RDONLY);
    if (fd == -1) {
	return -1;
    }
    retval = DrvSendFd(vconnCtxPtr, fd, statbuf.st_size);
    close(fd);
    
    return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvDetach --
 *
 *	Make a copy of the conn context for keep-alive. 
 *
 * Results:
 *	A pointer to the copied context. 
 *
 * Side effects:
 *
 *----------------------------------------------------------------------
 */

static void *
DrvDetach(void *vconnCtxPtr)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;
    ConnContext *newCtxPtr;

    newCtxPtr = ns_malloc(sizeof(ConnContext));
    assert(newCtxPtr != NULL);
    memcpy(newCtxPtr, connCtxPtr, sizeof(ConnContext));
    newCtxPtr->buffer = (char *) ns_malloc(BUFFERSIZE + 1);
    assert(newCtxPtr->buffer != NULL);
    newCtxPtr->base = newCtxPtr->buffer;
    newCtxPtr->count = connCtxPtr->count;
    memcpy(newCtxPtr->base, connCtxPtr->base, connCtxPtr->count);

    Ns_ModLog(Debug, unixModLogHandle, "detached context for keepalive.");
    
    return (void *) newCtxPtr;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvConnectionFd --
 *
 *	Returns the FD for this connection. 
 *
 * Results:
 *	A file descriptor. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static int
DrvConnectionFd(void *vconnCtxPtr)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;

    if (connCtxPtr->unix_sock != -1) {
	GetDataFromUDS(connCtxPtr);
    }
    
    return connCtxPtr->sock;
}


/*
 *----------------------------------------------------------------------
 *
 * DrvPeerPort --
 *
 *	Returns the port of the peer. 
 *
 * Results:
 *	A port number. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static int
DrvPeerPort(void *vconnCtxPtr)
{
    ConnContext *connCtxPtr = (ConnContext *) vconnCtxPtr;

    return connCtxPtr->remotePort;
}

/*
 *==========================================================================
 * Static functions
 *==========================================================================
 */


/*
 *----------------------------------------------------------------------
 *
 * Listen --
 *
 *	Listen on a host, port. If you are modifying this to use 
 *	something other than TCP, change the parameters to be, for 
 *	example, the path of a unix-domain socket. It'll still return 
 *	a socket, thus abstracting the underlying implementation. 
 *
 * Results:
 *	A valid socket to listen on, or -1 on error. 
 *
 * Side effects:
 *	None. 
 *
 *----------------------------------------------------------------------
 */

static int
Listen(char *path)
{
    int                sock;
    struct sockaddr_un name;

    sock = socket(AF_UNIX, SOCK_STREAM, 0);
    if (sock < 0) {
	return -1;
    }
    bzero((char *) &name, sizeof(name));
    unlink(path);
    name.sun_family = AF_UNIX;
    strcpy(name.sun_path, path);
    if (bind(sock, (struct sockaddr *) &name, sizeof(name)) < 0) {
	return -1;
    }
    if (listen(sock, 16) < 0) {
	return -1;
    }
    Ns_ModLog(Debug, unixModLogHandle, "listening on unix domain socket");

    return sock;
}


/*
 *----------------------------------------------------------------------
 *
 * Accept --
 *
 *	Wrapper around accept().
 *
 * Results:
 *	Result from accept().
 *
 * Side effects:
 *	May log error.
 *
 *----------------------------------------------------------------------
 */

static int
Accept(int lsock, struct sockaddr *addr, int *addrlen)
{
    int sock;

    sock = Ns_SockAccept(lsock, addr, addrlen);
    if (sock == -1) {
	Ns_ModLog(Warning, unixModLogHandle, "accept(%d) failed: %s",
		  lsock, strerror(errno));
    }

    return sock;
}
 

/*
 *----------------------------------------------------------------------
 *
 * GetDataFromUDS --
 *
 *	Receive a file descriptor and headers from the virtual host 
 *	redirector and drain all data in message. 
 *
 * Results:
 *	NS_OK/NS_ERROR 
 *
 * Side effects:
 *	Fills the ConnContext read buffer with data from unix domain
 *	until there is no more.
 *
 *----------------------------------------------------------------------
 */

static int
GetDataFromUDS(ConnContext *connCtxPtr)
{
    struct msghdr  msg;
    struct iovec   iov[1];
    int            sock;
    int            numRead;
#ifdef HAS_CMSGHDR
    my_cmsghdr     ancillary;
#endif
    
    if (connCtxPtr->count == 0) {
	iov[0].iov_base = connCtxPtr->buffer;
	iov[0].iov_len  = BUFFERSIZE;

	msg.msg_iov     = iov;
	msg.msg_iovlen  = 1;
	msg.msg_name    = NULL;
	msg.msg_namelen = 0;

#ifdef HAS_CMSGHDR
	msg.msg_control      = &ancillary;
	msg.msg_controllen   = sizeof(ancillary);
#else
	msg.msg_accrights    = (caddr_t) &sock;
	msg.msg_accrightslen = sizeof(sock);
#endif

	numRead = recvmsg(connCtxPtr->unix_sock, &msg, 0);
	Ns_ModLog(Debug, unixModLogHandle, "received %d bytes on unix "
		  "domain socket fd: %d", numRead, connCtxPtr->unix_sock);
	if (numRead < 0) {
	    Ns_ModLog(Error, unixModLogHandle, "recvmsg() failed on fd %d: %s",
		      connCtxPtr->unix_sock, strerror(errno));
	    return NS_ERROR;
	}

#ifdef HAS_CMSGHDR
	if (ancillary.cmsg.cmsg_len != sizeof(ancillary) ||
	    ancillary.cmsg.cmsg_level != SOL_SOCKET ||
	    ancillary.cmsg.cmsg_type != SCM_RIGHTS) {

	    Ns_ModLog(Error, unixModLogHandle, "unexpected ancillary data:"
		      " cmsg_len: %d, cmsg_level: %d, cmsg_type: %d",
		      ancillary.cmsg.cmsg_len, ancillary.cmsg.cmsg_level,
		      ancillary.cmsg.cmsg_type);
	    return NS_ERROR;
	}

	sock = ancillary.sock;
#endif

	if (numRead > 0) {
	    assert (numRead <= BUFFERSIZE);
	    connCtxPtr->buffer[numRead] = '\0';
	    connCtxPtr->base = connCtxPtr->buffer;
	    connCtxPtr->count = numRead;

	    if (connCtxPtr->sock == -1) {
		connCtxPtr->sock = sock;
		Ns_ModLog(Debug, unixModLogHandle, "received passed socket "
			  "fd: %d on unix domain socket fd: %d",
			  connCtxPtr->sock, connCtxPtr->unix_sock);
	    }
	} else if (numRead == 0) {
	    close(connCtxPtr->unix_sock);
	    connCtxPtr->unix_sock = -1;
	}
    }

    return NS_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * ConnWait --
 *
 *      Wait for nonblocking I/O
 *
 * Results:
 *      1 if the socket is ready, 0 if the socket is timed out
 *
 * Side effects:
 *      None
 *
 *----------------------------------------------------------------------
 */

static int
ConnWait(ConnContext *connCtxPtr, int write)
{
    fd_set set;
    struct timeval tv;
    int result;

    FD_ZERO(&set);
    FD_SET(connCtxPtr->sock, &set);
    tv.tv_sec = SELECT_TIMEOUT;
    tv.tv_usec = 0;
    if (write) {
	result = select(connCtxPtr->sock + 1, NULL, &set, NULL, &tv);
    } else {
	result = select(connCtxPtr->sock + 1, &set, NULL, NULL, &tv);
    }

    return (result > 0 ? NS_TRUE : NS_FALSE);
}

