/*  Gtk+ Font Selector
 *  Copyright (C) 1998  Damon Chaplin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <gtk/gtk.h>

#ifndef GTK_HAVE_FEATURES_1_1_0
#include "gtkfontsel.h"
#endif

/*
 * This is a self-contained font-selector program, rather like xfontsel.
 * It even accepts the -print option to print the selected font name when it
 * exits.
 */

gboolean print_on_quit = FALSE;
gchar *selection = NULL;

void on_copy_font_name (GtkWidget *widget,
			GtkWidget *fontsel);

/* We currently support GTK 1.0 and GTK 1.1.4+ selection. */
#ifdef GTK_HAVE_FEATURES_1_1_4
void selection_get (GtkWidget *widget, 
		    GtkSelectionData *selection_data,
		    guint      info,
		    guint      time,
		    gpointer   data);
#else
void selection_handler (GtkWidget        *widget,
			GtkSelectionData *selection_data,
			gpointer          data);
#endif

void quit (GtkWidget *widget,
	   gpointer data);
void usage ();
void version ();


int
main (int argc, char *argv[])
{
  GtkWidget *widget;
  GtkFontSelectionDialog *fontsel;
  gint i;
  gchar *preview_text = NULL;

#ifdef GTK_HAVE_FEATURES_1_1_4
  static GtkTargetEntry targetlist[] = {
    { "STRING", 0, GDK_SELECTION_TYPE_STRING },
  };
  static gint ntargets = sizeof(targetlist) / sizeof(targetlist[0]);
#endif

  gtk_set_locale ();
  gtk_init (&argc, &argv);

  for (i = 0; i < argc; i++)
    {
      if (!strcmp (argv[i], "-print"))
	print_on_quit = TRUE;
      else if (!strcmp (argv[i], "--help"))
	usage ();
      else if (!strcmp (argv[i], "--version"))
	version ();
      else if (i < argc - 1 
	       && (!strcmp (argv[i], "-p") || !strcmp (argv[i], "--preview")))
	preview_text = argv[i + 1];
    }

  widget = gtk_font_selection_dialog_new ("Font Selector");
  fontsel = GTK_FONT_SELECTION_DIALOG (widget);

  if (preview_text)
    gtk_font_selection_dialog_set_preview_text (fontsel, preview_text);

  gtk_label_set (GTK_LABEL (GTK_BUTTON (fontsel->ok_button)->child),
		 "Copy Font Name");
  gtk_signal_connect (GTK_OBJECT (fontsel->ok_button), "clicked",
		      GTK_SIGNAL_FUNC (on_copy_font_name), fontsel);

  gtk_label_set (GTK_LABEL (GTK_BUTTON (fontsel->cancel_button)->child),
		 "Exit");
  gtk_signal_connect_object (GTK_OBJECT (fontsel->cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT (fontsel));
  gtk_signal_connect (GTK_OBJECT (fontsel), "destroy",
		      GTK_SIGNAL_FUNC (quit), fontsel);

#ifdef GTK_HAVE_FEATURES_1_1_4
  gtk_signal_connect (GTK_OBJECT (fontsel), "selection_get",
		      GTK_SIGNAL_FUNC (selection_get), NULL);
  gtk_selection_add_targets (widget, GDK_SELECTION_PRIMARY,
			     targetlist, ntargets);
#else
  gtk_selection_add_handler (widget, GDK_SELECTION_PRIMARY, GDK_TARGET_STRING,
			     selection_handler, NULL);
#endif

  gtk_widget_show (widget);
  gtk_main ();
  return 0;
}


void
on_copy_font_name (GtkWidget *widget,
		   GtkWidget *fontsel)
{
  GtkFontSelectionDialog *fsd = GTK_FONT_SELECTION_DIALOG (fontsel);
  g_free (selection);
  selection = gtk_font_selection_dialog_get_font_name (fsd);
  gtk_selection_owner_set (fontsel, GDK_SELECTION_PRIMARY, GDK_CURRENT_TIME);
}


#ifdef GTK_HAVE_FEATURES_1_1_4
void
selection_get (GtkWidget *widget, 
	       GtkSelectionData *selection_data,
	       guint      info,
	       guint      time,
	       gpointer   data)
{
  gchar *text = selection ? selection : "<none>";
  gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
			  8 * sizeof (gchar), text, strlen (text));
}
#else
void
selection_handler (GtkWidget        *widget,
		   GtkSelectionData *selection_data,
		   gpointer          data)
{
  gchar *text = selection ? selection : "<none>";
  gtk_selection_data_set (selection_data, GDK_SELECTION_TYPE_STRING,
			  8 * sizeof (gchar), text, strlen (text));
}
#endif

void
quit (GtkWidget *widget, gpointer data)
{
  if (print_on_quit)
    {
      GtkFontSelectionDialog *fsd = GTK_FONT_SELECTION_DIALOG (widget);
      gchar *font_name = gtk_font_selection_dialog_get_font_name (fsd);
      if (font_name) g_print ("%s", font_name);
      g_free (font_name);
    }
  gtk_exit (0);
}


void
usage ()
{
  g_print ("Usage: gtkfontsel [-print] [-p preview_text]\n"
	   "\n"
	   "Options:\n"
	   "  -print                    print the selected font name to STDOUT on exit\n"
	   "  -p TEXT, --preview TEXT   show the given text in the preview area\n");
  exit(0);
}


void
version ()
{
  g_print ("gtkfontsel %s\n"
	   "\n"
	   "Copyright (C) 1998 Damon Chaplin\n"
	   "gtkfontsel comes with NO WARRANTY, to the extent permitted by law.\n"
	   "You may redistribute copies of gtkfontsel under the terms of the GNU General\n"
	   "Public License. See the COPYING file in the source distribution.\n",
	   VERSION);
  exit(0);
}

