
/*
 * DREADERD/DNS.C - dns resolution and authentication task
 *
 *	DNS authenticator for new connections.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

#define MAXKPAIR 32

Prototype void DnsTask(int fd);
Prototype void DnsTest(DnsReq *dreq, DnsRes *dres, const char *name, char **ary);

int TestForwardLookup(const char *hostName, struct sockaddr_in *rsin);
void GetAuthUser(struct sockaddr_in *lsin, struct sockaddr_in *rsin, char *ubuf, int ulen);

void
DnsTask(int fd)
{
    /*
     * read remote address to resolve.  Note: fd is not set to non-blocking
     */
    DnsReq dreq;
    int n;

    while ((n = read(fd, &dreq, sizeof(dreq))) == sizeof(dreq)) {
	struct hostent *he;
	DnsRes dres;

	bzero(&dres, sizeof(dres));

	stprintf("reverse auth %s", inet_ntoa(dreq.dr_RSin.sin_addr));

	GetAuthUser(&dreq.dr_LSin, &dreq.dr_RSin, dres.dr_User, sizeof(dres.dr_User));

	stprintf("dns lookup %s", inet_ntoa(dreq.dr_RSin.sin_addr));

	dres.dr_Addr = dreq.dr_RSin.sin_addr;

	he = gethostbyaddr((char *)&dreq.dr_RSin.sin_addr,sizeof(dreq.dr_RSin.sin_addr),AF_INET);
	if (he != NULL) {
	    DnsTest(&dreq, &dres, he->h_name, he->h_aliases);
	    if (dres.dr_Code == 0)  {
		bzero(&dres, sizeof(dres));
		DnsTest(&dreq, &dres, inet_ntoa(dreq.dr_RSin.sin_addr), NULL);
	    }
	} else {
	    DnsTest(&dreq, &dres, inet_ntoa(dreq.dr_RSin.sin_addr), NULL);
	}
	write(fd, &dres, sizeof(dres));
	stprintf("dns idle");
    }
}

/*
 * DnsTest() - test access file entry against host.  If ary == NULL, then
 *		'name' is a dotted quad and we do not try to do a forward
 *		lookup security check with it.
 */

void
DnsTest(DnsReq *dreq, DnsRes *dres, const char *name, char **ary)
{
    FILE *fi;
    int ok = 0;

    /*
     * open dreader.hosts
     */

    if ((fi = xfopen("r", "%s/dreader.hosts", NewsHome)) != NULL) {
	char buf[256];
	char *flags = "";

	while (fgets(buf, sizeof(buf), fi) != NULL) {
	    int i;
	    int needauth = 0;
	    char *host;

	    host = strtok(buf, " \t\n");
	    if (host == NULL || host[0] == '#' || host[0] == '\n')
		continue;
	    flags = strtok(NULL, " \t\n");
	    if (flags == NULL || flags[0] == '#') {
		flags = "";
	    }

	    if (strchr(flags, 'a') != NULL) {
		char *kt;
		char *user = NULL;
		char *pass = NULL;

		while ((kt = strtok(NULL, " \t\n")) != NULL) {
		    if (strncmp(kt, "user=", 5) == 0) {
			user = kt + 5;
		    } else if (strncmp(kt, "pass=", 5) == 0) {
			if (kt[5])
			    pass = kt + 5;
		    }
		}
		if (DebugOpt)
		    printf("USER %s PASS %s DR_USER %s\n", user, pass, dres->dr_User);
		if (user == NULL) {
		    needauth = 1;
		} else if (strcasecmp(dres->dr_User, user) == 0) {
		    /*
		     * reverse-auth auto authorization if password is blank
		     */
		    if (pass && strcmp(dreq->dr_AuthPass, pass) != 0)
			needauth = 1;
		} else if (strcasecmp(dreq->dr_AuthUser, user) == 0) {
		    /*
		     * password required
		     */
		    if (pass == NULL || strcmp(dreq->dr_AuthPass, pass) != 0) {
			needauth = 1;
		    }
		} else {
		    needauth = 1;
		}
	    }

	    if (DebugOpt)
		printf("needauth is %d\n", needauth);

	    /*
	     * note: strtok used by subroutines, invalid after this point.
	     */

	    if (strchr(host, '?') == NULL && strchr(host, '*') == NULL) {
		if (strcasecmp(host, name) == 0) {
		    if (ary == NULL || TestForwardLookup(name, &dreq->dr_RSin) == 0) {
			ok = 1;
			snprintf(dres->dr_Host, sizeof(dres->dr_Host), "%s", name);
		    } else {
			sprintf(dres->dr_Host, "dns fwd/rev mismatch");
		    }
		}
	    } else if (WildCaseCmp(host, name) == 0) {
		ok = 1;
		snprintf(dres->dr_Host, sizeof(dres->dr_Host), "%s", name);
	    }
	    if (DebugOpt)
		printf("OK = %d %s,%s\n", ok, host, name);

	    for (i = 0; ok == 0 && ary && ary[i] != NULL; ++i) {
		if (strchr(host, '?') == NULL && strchr(host, '*') == NULL) {
		    if (strcasecmp(host, ary[i]) == 0) {
			if (TestForwardLookup(ary[i], &dreq->dr_RSin) == 0) {
			    ok = 1;
			    snprintf(dres->dr_Host, sizeof(dres->dr_Host), "%s", ary[i]);
			    break;
			} else {
			    sprintf(dres->dr_Host, "dns fwd/rev mismatch");
			}
		    }
		} else if (WildCaseCmp(host, ary[i]) == 0) {
		    ok = 1;
		    snprintf(dres->dr_Host, sizeof(dres->dr_Host), "%s", ary[i]);
		}
	    }
	    if (ok) {
		if (needauth)
		    dres->dr_Flags |= DF_AUTHREQUIRED;
		break;
	    }
	}
	fclose(fi);

	/*
	 * munge dr_Host if it's illegal
	 */
	SanitizeString(dres->dr_Host);

	/*
	 * deal with flags
	 */

	if (ok) {
	    char c;

	    while ((c = *flags) != 0) {
		switch(c) {
		case 'a':
		    /* already dealt with */
		    break;
		case 'f':
		    dres->dr_Flags |= DF_FEED;
		    break;
		case 'r':
		    dres->dr_Flags |= DF_READ;
		    break;
		case 'p':
		    dres->dr_Flags |= DF_POST;
		    break;
		case 'w':
		    dres->dr_Flags |= DF_POST;
		    break;
		case 's':
		    dres->dr_Flags |= DF_FETCHSERVER;
		    dres->dr_FetchPri = strtol(flags + 1, NULL, 0);
		    break;
		case 'o':
		    dres->dr_Flags |= DF_PUSHSERVER;
		    dres->dr_PushPri = strtol(flags + 1, NULL, 0);
		    break;
		case 'g':
		    dres->dr_Flags |= DF_GROUPID;
		    dres->dr_GroupId = strtol(flags + 1, NULL, 0);
		    break;
		default:
		    /*
		     * number or unknown flag ignored (unless number retrieved
		     * above, in which case we skip it here).
		     */
		    break;
		}
		++flags;
	    }
	}
    } else {
	logit(LOG_NOTICE, "connect: file %s/%s not found", NewsHome, "/dreader.hosts");
    }
    dres->dr_Code = ok;
}

/*
 * TestForwardLookup().  XXX WARNING!  We already called gethostbyaddr(),
 *  	gethostbyname() may overwrite gethostbyaddr()'s structure!!!!
 */

int
TestForwardLookup(const char *hostName, struct sockaddr_in *rsin)
{
    return(0);
#ifdef NOTDEF
    sturct hostent *he;

    if ((he = gethostbyname(hostName)) != NULL) {
    }
#endif
}

/*
 * GetAuthUser() - authenticate the remote username by connecting to port
 *		   113 and requesting the user id of the remote port.  This
 *		   is used by X-Trace:
 */

void
GetAuthUser(struct sockaddr_in *plsin, struct sockaddr_in *prsin, char *ubuf, int ulen)
{
    int cfd;
    int lport;
    int rport;
    int n;
    char buf[256];
    struct sockaddr_in lsin = *plsin;
    struct sockaddr_in rsin = *prsin;

    rport = ntohs(rsin.sin_port);
    lport = ntohs(lsin.sin_port);

    lsin.sin_port = 0;
    lsin.sin_family = AF_INET;

    if ((cfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
        return;
    }
    if (bind(cfd, (struct sockaddr *)&lsin, sizeof(lsin)) < 0) {
	perror("bind");
        close(cfd);
        return;
    }
    rsin.sin_port = htons(113);
    rsin.sin_family = AF_INET;

    /*
     * Do asynchronous connection with 10 second timeout.  The timeout is
     * necessary to deal with broken clients or firewalls.
     */

    fcntl(cfd, F_SETFL, O_NONBLOCK);
    errno = 0;
    if (connect(cfd, (struct sockaddr *)&rsin, sizeof(rsin)) < 0) {
	fd_set wfds;
	struct timeval tv = { 10, 0 };

	if (errno != EINPROGRESS) {
	    close(cfd);
	    return;
	}
	FD_ZERO(&wfds);
	FD_SET(cfd, &wfds);
	if (select(cfd + 1, NULL, &wfds, NULL, &tv) == 0) {
	    close(cfd);
	    cfd = -1;
	}
    }

    /*
     * Send query, interpret response
     *
     * write() may fail if connect() failed to establish a connection.
     */

    if (cfd >= 0) {
	fcntl(cfd, F_SETFL, 0);
	sprintf(buf, "%d, %d\r\n", rport, lport);
	if (DebugOpt)
	    printf("IDENTD COMMAND %s\n", buf);
	if (write(cfd, buf, strlen(buf)) != strlen(buf))
	    n = -1;
	else
	    n = read(cfd, buf, sizeof(buf) - 1);
    } else {
	n = -1;
    }
    if (n > 0) {
	char *uid;

        buf[n] = 0;

	if (DebugOpt)
	    printf("IDENTD RESPONSE %s\n", buf);

        if ((uid = strstr(buf, "USERID")) != NULL) {
            uid = strchr(uid + 1, ':');
            if (uid) {
                uid = strchr(uid + 1, ':');
                if (uid) {
                    uid = strtok(uid + 1, "\r\n: \t");
                    if (uid && strlen(uid) < ulen) {
                        strcpy(ubuf, uid);
			SanitizeString(ubuf);

                    }
                }
            }
        }
    }
    if (cfd >= 0)
	close(cfd);
}

