
/*
 * LIST.C
 *
 *	NNTP LIST commands.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype void NNTPList(Connection *conn, char **pptr);
Prototype void NNTPListActive(Connection *conn, char **pptr);
Prototype void NNTPListActiveTimes(Connection *conn, char **pptr);
Prototype void NNTPListNewsgroups(Connection *conn, char **pptr);
Prototype void NNTPListDistributions(Connection *conn, char **pptr);
Prototype void NNTPListDistribPats(Connection *conn, char **pptr);
Prototype void NNTPListOverviewFmt(Connection *conn, char **pptr);
Prototype void NNTPListSubscriptions(Connection *conn, char **pptr);
Prototype void NNTPListModerators(Connection *conn, char **pptr);
Prototype void NNTPListGroup(Connection *conn, char **pptr);
Prototype void NNTPNewgroups(Connection *conn, char **pptr);
Prototype void NNTPXGTitle(Connection *conn, char **pptr);

void NNStartListActiveScan(Connection *conn, const char *wc, int mode);
void NNListActiveScan(Connection *conn);
void NNListGroup(Connection *conn);

void 
NNTPList(Connection *conn, char **pptr)
{
    /*
     * list active [groupwild]
     * list active.times
     * list newsgroups [groupwild]
     * list distributions
     * list distrib.pats
     * list overview.fmt
     * list subscriptions
     * list moderators
     */
    char *ltype = parseword(pptr, " \t");

    if (ltype) {
	int i;
	for (i = 0; ltype[i]; ++i)
	    ltype[i] = tolower((int)(unsigned char)ltype[i]);
    }
    if (ltype == NULL) {
	NNTPListActive(conn, NULL);
    } else if (strcmp(ltype, "active") == 0) {
	NNTPListActive(conn, pptr);
    } else if (strcmp(ltype, "active.times") == 0) {
	NNTPListActiveTimes(conn, pptr);
    } else if (strcmp(ltype, "newsgroups") == 0) {
	NNTPListNewsgroups(conn, pptr);
    } else if (strcmp(ltype, "distributions") == 0) {
	NNTPListDistributions(conn, pptr);
    } else if (strcmp(ltype, "distrib.pats") == 0) {
	NNTPListDistribPats(conn, pptr);
    } else if (strcmp(ltype, "overview.fmt") == 0) {
	NNTPListOverviewFmt(conn, pptr);
    } else if (strcmp(ltype, "subscriptions") == 0) {
	NNTPListSubscriptions(conn, pptr);
    } else if (strcmp(ltype, "moderators") == 0) {
	NNTPListModerators(conn, pptr);
    } else {
	NNBadCommandUse(conn);
    }
}

void 
NNTPListActive(Connection *conn, char **pptr)
{
    const char *wc = NULL;

    if (pptr)
	wc = parseword(pptr, " \t");	/* may also be NULL */

    MBPrintf(&conn->co_TMBuf, "215 Newsgroups in form \"group high low flags\".\r\n");

    NNStartListActiveScan(conn, wc, COM_ACTIVE);
}

void
NNStartListActiveScan(Connection *conn, const char *wc, int mode)
{
    /*
     * Setup list pattern
     */

    zfreeStr(&conn->co_MemPool, &conn->co_ListPat);
    if (wc)
	conn->co_ListPat = zallocStr(&conn->co_MemPool, wc);

    /*
     * Initiate active file scan, but optimize if specific group requested
     */

    if (wc == NULL || strchr(wc, '*') || strchr(wc, '?')) {
	conn->co_ListRec = KPDBScanFirst(KDBActive, 0, &conn->co_ListRecLen);
	conn->co_Flags |= COF_PATISWILD;
    } else {
	conn->co_ListRec = KPDBReadRecord(KDBActive, wc, 0, &conn->co_ListRecLen);
	conn->co_Flags &= ~COF_PATISWILD;
    }
    conn->co_ArtMode = mode;
    NNListActiveScan(conn);
}

void
NNListActiveScan(Connection *conn)
{
    conn->co_Func = NNListActiveScan;
    conn->co_State = "listac";

    while (conn->co_TMBuf.mh_Bytes < 800 && conn->co_ListRec) {
	int glen;
	const char *group = KPDBGetField(conn->co_ListRec, conn->co_ListRecLen, NULL, &glen, NULL);
	char grpbuf[MAXGNAME];

	if (glen < MAXGNAME) {
	    bcopy(group, grpbuf, glen);
	    grpbuf[glen] = 0;

	    if (conn->co_ListPat == NULL || 
		WildCmp(conn->co_ListPat, grpbuf) == 0
	    ) {
		if (conn->co_ArtMode == COM_GROUPDESC) {
		    int glen;
		    const  char *desc = KPDBGetField(conn->co_ListRec, conn->co_ListRecLen, "GD", &glen, "?");

		    MBPrintf(&conn->co_TMBuf, "%s\t", grpbuf);
		    MBWriteDecode(&conn->co_TMBuf, desc, glen);
		    MBWrite(&conn->co_TMBuf, "\r\n", 2);
		} else if (conn->co_ArtMode == COM_ACTIVE) {
		    int flen;

		    const char *flags = KPDBGetField(conn->co_ListRec, conn->co_ListRecLen, "S", &flen, "n");
		    int ne = strtol(KPDBGetField(conn->co_ListRec, conn->co_ListRecLen, "NE", NULL, "0"), NULL, 10);
		    int nb = strtol(KPDBGetField(conn->co_ListRec, conn->co_ListRecLen, "NB", NULL, "0"), NULL, 10);

		    /*
		     * NOTE: we cannot use *.*s because it's broken on most
		     * platforms... it will strlen() the string, and the string
		     * in this case is the entire size of the active file!
		     */

		    MBPrintf(&conn->co_TMBuf, "%s %010d %010d ",
			grpbuf, ne, nb
		    );
		    MBWrite(&conn->co_TMBuf, flags, flen);
		    MBWrite(&conn->co_TMBuf, "\r\n", 2);
		}
	    }
	}

	/*
	 * If we are looking for a wildcard, continue the scan.  Otherwise
	 * we are done.
	 */

	if (conn->co_Flags & COF_PATISWILD) {
	    conn->co_ListRec = KPDBScanNext(
		KDBActive, 
		conn->co_ListRec, 
		0, 
		&conn->co_ListRecLen
	    );
	} else {
	    conn->co_ListRec = NULL;
	}
    }
    if (conn->co_ListRec == NULL) {
	MBPrintf(&conn->co_TMBuf, ".\r\n");
	NNCommand(conn);
	zfreeStr(&conn->co_MemPool, &conn->co_ListPat);
	printf("done\n");
    }
}

void 
NNTPListActiveTimes(Connection *conn, char **pptr)
{
    MBPrintf(&conn->co_TMBuf, "500 list active.times not supported yet\r\n");
    NNCommand(conn);
}

/*
 * list newsgroups grouppat
 */

void 
NNTPListNewsgroups(Connection *conn, char **pptr)
{
    const char *wc = NULL;

    if (pptr)
	wc = parseword(pptr, " \t");	/* may also be NULL */

    MBPrintf(&conn->co_TMBuf, "215 descriptions in form \"group description\"\r\n");

    NNStartListActiveScan(conn, wc, COM_GROUPDESC);
}

void
NNTPXGTitle(Connection *conn, char **pptr)
{
    const char *wc = NULL;

    if (pptr)
	wc = parseword(pptr, " \t");	/* may also be NULL */

    MBPrintf(&conn->co_TMBuf, "282 list follows\r\n");

    NNStartListActiveScan(conn, wc, COM_GROUPDESC);
}

void 
NNTPListDistributions(Connection *conn, char **pptr)
{
    MBPrintf(&conn->co_TMBuf, "215 Distributions in form \"area description\".\r\n");
    MBPrintf(&conn->co_TMBuf, ".\r\n");
    NNCommand(conn);
}

void 
NNTPListDistribPats(Connection *conn, char **pptr)
{
    MBPrintf(&conn->co_TMBuf, "215 Default istributions in form \"weight:pattern:value\".\r\n");
    MBPrintf(&conn->co_TMBuf, ".\r\n");
    NNCommand(conn);
}

void 
NNTPListOverviewFmt(Connection *conn, char **pptr)
{
    MBPrintf(&conn->co_TMBuf, "215 Order of fields in overview database.\r\n");
    MBPrintf(&conn->co_TMBuf, "%s.\r\n", OverViewFmt);
    NNCommand(conn);
}

void 
NNTPListSubscriptions(Connection *conn, char **pptr)
{
    MBPrintf(&conn->co_TMBuf, "215 Subscriptions in form \"group\"\r\n");
    MBPrintf(&conn->co_TMBuf, ".\r\n");
    NNCommand(conn);
}

void
NNTPListModerators(Connection *conn, char **pptr)
{
    FILE *fi;

    MBPrintf(&conn->co_TMBuf, "215 Newsgroup moderators in form \"group-pattern:mail-address-pattern\".\r\n");
    if ((fi = xfopen("r", "%s/moderators", NewsHome)) != NULL) {
	char buf[256];
	while (fgets(buf, sizeof(buf), fi) != NULL) {
	    int l = strlen(buf);
	    if (l == 0 || buf[0] == '#' || buf[0] == '\n')
		continue;
	    if (buf[l-1] == '\n')
		buf[--l] = 0;
	    if (buf[0] == '.')
		MBWrite(&conn->co_TMBuf, ".", 1);
	    MBWrite(&conn->co_TMBuf, buf, l);
	    MBWrite(&conn->co_TMBuf, "\r\n", 2);
	}
	fclose(fi);
    }
    MBPrintf(&conn->co_TMBuf, ".\r\n");
    NNCommand(conn);
}

void 
NNTPListGroup(Connection *conn, char **pptr)
{
    char *group;
    const char *rec;
    int recLen;

    if ((group = parseword(pptr, " \t")) != NULL && strlen(group) < MAXGNAME)
	SetCurrentGroup(conn, group);
    if (conn->co_GroupName) {
	if ((rec = KPDBReadRecord(KDBActive, conn->co_GroupName, 0, &recLen)) != NULL) {
	    conn->co_ListBegNo = strtol(KPDBGetField(rec, recLen, "NB", NULL, "0"), NULL, 10);
	    conn->co_ListEndNo = strtol(KPDBGetField(rec, recLen, "NE", NULL, "0"), NULL, 10);
	    if (conn->co_ListEndNo - conn->co_ListBegNo > MAXARTSINGROUP)
		conn->co_ListBegNo = conn->co_ListEndNo - MAXARTSINGROUP;
	    MBPrintf(&conn->co_TMBuf, "211 Article list follows\n");
	    NNListGroup(conn);
	} else {
	    MBPrintf(&conn->co_TMBuf, "411 No such group %s\n", conn->co_GroupName);
	    NNCommand(conn);
	}
    } else {
	MBPrintf(&conn->co_TMBuf, "481 No group specified\n");
	NNCommand(conn);
    }
}

void
NNListGroup(Connection *conn)
{
    conn->co_Func = NNListGroup;
    conn->co_State = "listgr";

    while (conn->co_TMBuf.mh_Bytes < 800 && conn->co_ListBegNo < conn->co_ListEndNo) {
	MBPrintf(&conn->co_TMBuf, "%d\r\n", conn->co_ListBegNo);
	++conn->co_ListBegNo;
    }
    if (conn->co_ListBegNo < conn->co_ListEndNo) {
    } else {
	MBPrintf(&conn->co_TMBuf, ".\r\n");
	NNCommand(conn);
    }
}

void
NNTPNewgroups(Connection *conn, char **pptr)
{
    MBPrintf(&conn->co_TMBuf, "231 New newsgroups follow. (not implemented)\r\n");
    MBPrintf(&conn->co_TMBuf, ".\r\n");
    NNCommand(conn);
}   
	    

