/*********************************************************************
 *
 *      Copyright (C) 1999-2002 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Abstract Data Types
 * MODULE:      Pancake Stack
 * FILE:        PancakeStack.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      08/04/97        Initial version
 *      nf      01/20/01        Removed some synchronization
 *      nf      01/21/01        Implemented Collection interface and
 *                              renamed to PancakeStack
 *      nf      01/05/02        Removed all synchronization
 *
 * DESCRIPTION:
 *      This is an implementation of a pancake stack. See the
 *      class documentation for more information on this ADT.
 *
 * $Id: PancakeStack.java,v 1.6 2002/01/06 09:33:25 nfiedler Exp $
 *
 ********************************************************************/

package com.bluemarsh.adt;

import java.util.Collection;
import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * This class implements a pancake stack. A pancake stack is very
 * similar to a normal LIFO data structure except for one additional
 * feature - this stack allows you to flip a portion of the stack.
 * You may think of this stack like a stack of pancakes. You can insert
 * a spatula at any position within the stack and flip all the pancakes
 * between the spatula and the top of the stack. Those pancakes that
 * were on the bottom of the pile are now on the top and vice versa.
 *
 * <p>This collection may contain null items.</p>
 *
 * <p>The implementation is basically that of a singly-linked list.</p>
 *
 * <p><b>Note that this implementation is not synchronized.</b> If multiple
 * threads access a set concurrently, and at least one of the threads modifies
 * the set, it <i>must</i> be synchronized externally.  This is typically
 * accomplished by synchronizing on some object that naturally encapsulates
 * the set.  If no such object exists, the set should be "wrapped" using the
 * <code>Collections.synchronizedCollection</code> method.  This is best done
 * at creation time, to prevent accidental unsynchronized access to the set:
 * </p>
 * <pre>
 *     Collection c = Collections.synchronizedCollection(new PancakeStack(...));
 * </pre>
 *
 * <p>The Iterators returned by this class's <tt>iterator</tt> method are
 * <i>fail-fast</i>: if the set is modified at any time after the iterator is
 * created, in any way except through the iterator's own <tt>remove</tt>
 * method, the iterator will throw a <tt>ConcurrentModificationException</tt>.
 * Thus, in the face of concurrent modification, the iterator fails quickly
 * and cleanly, rather than risking arbitrary, non-deterministic behavior at
 * an undetermined time in the future.</p>
 *
 * @author  Nathan Fiedler
 */
public class PancakeStack implements Collection, Cloneable {
    /** Head of the list (top of the stack). */
    protected Entry head;
    /** Number of elements in the stack. */
    protected int numElements;
    /** Keeps track of the modifications to the stack. */
    protected int modCount;

    /**
     * Ensures that this collection contains the specified element.
     *
     * @param  o  element whose presence in this collection is to be ensured.
     * @return  always returns true
     */
    public boolean add(Object o) {
        push(o);
        modCount++;
        return true;
    } // add

    /**
     * Adds all of the elements in the specified collection to this
     * collection.
     *
     * @param  c  elements to be inserted into this collection.
     * @return  always returns true
     */
    public boolean addAll(Collection c) {
        Iterator iter = c.iterator();
        while (iter.hasNext()) {
            push(iter.next());
        }
        modCount++;
        return true;
    } // addAll

    /**
     * Removes all of the elements from this collection.
     */
    public void clear() {
        numElements = 0;
        head = null;
        modCount++;
    } // clear

    /**
     * Creates and returns a copy of this object.
     *
     * @return  a clone of this instance.
     * @exception  CloneNotSupportedException
     *             Never thrown.
     * @exception  OutOfMemoryError
     *             Thrown if there is not enough memory.
     */
    public Object clone() throws CloneNotSupportedException {
        PancakeStack copy = new PancakeStack();
        copy.numElements = numElements;
        if (head != null) {
            // clone the head entry
            Entry entryCopy = (Entry) head.clone();
            copy.head = entryCopy;
            while (entryCopy != null) {
                if (entryCopy.next != null) {
                    // clone the next entry
                    entryCopy.next = (Entry) entryCopy.next.clone();
                }
                entryCopy = entryCopy.next;
            }
        }
        return copy;
    } // clone

    /**
     * Returns true if this collection contains the specified element.
     *
     * @param  o  element whose presence in this collection is to be tested.
     * @return  true if this collection contains the specified element
     */
    public boolean contains(Object o) {
        return indexOf(o) > -1;
    } // contains

    /**
     * Returns true if this collection contains all of the elements in
     * the specified collection.
     *
     * @param  c  collection to be checked for containment in this collection.
     * @return  true if this collection contains all of the elements in the
     *          specified collection
     */
    public boolean containsAll(Collection c) {
        Iterator iter = c.iterator();
        while (iter.hasNext()) {
            if (!contains(iter.next())) {
                return false;
            }
        }
        return true;
    } // containsAll

    /**
     * Compares the specified object with this collection for equality.
     *
     * @param  o  Object to be compared for equality with this collection.
     * @return  true if the specified object is equal to this collection
     */
    public boolean equals(Object o) {
        return o == this;
    } // equals

    /**
     * Flips the stack starting from the "flip point" and moving
     * towards the top of the stack. This is equivalent to flipping
     * a portion of a stack of pancakes - those that were on the
     * bottom are now on the top, and vice versa.
     *
     * @param  point  index from top of stack where to flip
     */
    public void flip(int point) {
        if (isEmpty() || (point <= 0)) {
            return;
        }
        Entry prev = null;
        Entry curr = head;
        Entry next = head.next;
        int index = 0;
        // Run through the list and reverse all the links.
        while ((index < point) && (next != null)) {
            curr.next = prev;
            prev = curr;
            curr = next;
            next = next.next;
            index++;
        }
        // Fix up the pointers for new head of list.
        curr.next = prev;
        head.next = next;
        head = curr;
        modCount++;
    } // flip

    /**
     * Returns the hash code value for this collection.
     *
     * @return  the hash code value for this collection
     */
    public int hashCode() {
        return System.identityHashCode(this);
    } // hashCode

    /**
     * Returns where an object is on the stack. It begins at the top
     * of the stack and moves downward, comparing objects using the
     * <code>equals</code> method. It will stop after finding the
     * first matching object.
     *
     * @param  data  object to find in the stack
     * @return  index from top of stack where data is (zero-based);
     *          -1 if the object was not found in the stack
     */
    public int indexOf(Object o) {
        Entry current = head;
        int index = 0;
        while (current != null) {
            if (current.data == null) {
                if (o == null) {
                    return index;
                }
            } else if (current.data.equals(o)) {
                return index;
            }
            current = current.next;
            index++;
        }
        return -1;
    } // indexOf

    /**
     * Returns true if this collection contains no elements.
     *
     * @return  true if this collection contains no elements
     */
    public boolean isEmpty() {
        return head == null;
    } // isEmpty

    /**
     * Returns an iterator over the elements in this collection.
     *
     * @return  an Iterator over the elements in this collection
     */
    public Iterator iterator() {
        return new Iter(head, modCount);
    } // iterator

    /**
     * Return the data object stored at the top of the stack.
     *
     * @return  data object at top of stack; null if empty stack
     */
    public Object peek() {
        return isEmpty() ? null : head.data;
    } // peek

    /**
     * Remove and return the data object stored at the top of the stack.
     *
     * @return  data object at the top of the stack; null if empty stack
     */
    public Object pop() {
        if (isEmpty()) {
            return null;
        } else {
            Entry entry = head;
            head = head.next;
            numElements--;
            modCount++;
            return entry.data;
        }
    } // pop

    /**
     * Pushes a new data object onto the stack, putting it at the top.
     * A subsequent peek or pop will return this object.
     *
     * @param  data  object to push onto the stack
     */
    public void push(Object data) {
        Entry prevHead = head;
        head = new Entry(data);
        head.next = prevHead;
        numElements++;
        modCount++;
    } // push

    /**
     * Removes the given entry from the stack. Does not modify the
     * modification count. This method is used by the iterator
     * remove() method.
     *
     * @param  n  Entry to be removed.
     */
    protected void remove(Entry n) {
        Entry current = head;
        Entry previous = null;
        while (current != null) {
            if (current == n) {
                if (previous == null) {
                    head = current.next;
                } else {
                    previous.next = current.next;
                }
            }
            previous = current;
            current = current.next;
        }
    } // remove

    /**
     * Removes a single instance of the specified element from this
     * collection, if it is present.
     *
     * @param  o  element to be removed from this collection, if present.
     * @return  true if this collection changed as a result of the call
     */
    public boolean remove(Object o) {
        Entry current = head;
        Entry previous = null;
        while (current != null) {
            if (current.data == null) {
                if (o == null) {
                    if (previous == null) {
                        head = current.next;
                    } else {
                        previous.next = current.next;
                    }
                    return true;
                }
            } else if (current.data.equals(o)) {
                if (previous == null) {
                    head = current.next;
                } else {
                    previous.next = current.next;
                }
                return true;
            }
            previous = current;
            current = current.next;
        }
        return false;
    } // remove

    /**
     * Removes all this collection's elements that are also contained
     * in the specified collection.
     *
     * @param  c  elements to be removed from this collection.
     * @return  true if this collection changed as a result of the call
     */
    public boolean removeAll(Collection c) {
        Iterator iter = c.iterator();
        boolean changed = false;
        while (iter.hasNext()) {
            if (remove(iter.next())) {
                changed = true;
            }
        }
        return changed;
    } // removeAll

    /**
     * Retains only the elements in this collection that are contained
     * in the specified collection.
     *
     * @param  c  elements to be retained in this collection.
     * @return  true if this collection changed as a result of the call
     */
    public boolean retainAll(Collection c) {
	boolean changed = false;
	Iterator iter = iterator();
	while (iter.hasNext()) {
	    if (!c.contains(iter.next())) {
		iter.remove();
		changed = true;
	    }
	}
	return changed;
    } // retainAll

    /**
     * Returns the number of elements in this collection.
     *
     * @return  the number of elements in this collection
     */
    public int size() {
        return numElements;
    } // size

    /**
     * Returns an array containing all of the elements in this collection.
     *
     * @return  an array containing all of the elements in this collection
     */
    public Object[] toArray() {
        return toArray(new Object[numElements]);
    } // toArray

    /**
     * Returns an array with a runtime type is that of the specified array and
     * that contains all of the elements in this collection. If the
     * collection fits in the specified array, it is returned therein.
     * Otherwise, a new array is allocated with the runtime type of the
     * specified array and the size of this collection.
     *<p>
     * If the collection fits in the specified array with room to spare (i.e.,
     * the array has more elements than the collection), the element in the
     * array immediately following the end of the collection is set to
     * <code>null</code>. This is useful in determining the length of the
     * collection <i>only</i> if the caller knows that the collection does
     * not contain any <code>null</code> elements.)
     *
     * @param  a  the array into which the elements of the collection are to
     * 	          be stored, if it is big enough; otherwise, a new array of
     * 	          the same runtime type is allocated for this purpose.
     * @return  an array containing the elements of the collection.
     * @exception  NullPointerException
     *             Thrown if the specified array is <code>null</code>.
     * @exception  ArrayStoreException
     *             Thrown if the runtime type of the specified array
     *             is not a supertype of the runtime type of every element
     *             in this collection.
     */
    public Object[] toArray(Object a[]) {
        int size = size();
        if (a.length < size) {
            a = (Object[])java.lang.reflect.Array.newInstance
                (a.getClass().getComponentType(), size);
        }
        Iterator iter = iterator();
        for (int i = 0; i < size; i++) {
            a[i] = iter.next();
        }
        if (a.length > size) {
            a[size] = null;
        }
        return a;
    } // toArray

    /**
     * Implements a entry of the pancake stack. This class is used
     * internally to the Pancake stack class, and is not to be used
     * outside of that class.
     *
     * @author  Nathan Fiedler
     */
    protected class Entry implements Cloneable {
        /** The data stored in this entry. */
        public Object data;
        /** The next entry in the list. */
        public Entry next;

        /**
         * One-arg constructor for Entry.
         *
         * @param  data  data object to store in this entry
         */
        public Entry(Object data) {
            this.data = data;
        } // Entry

        /**
         * Creates and returns a copy of this object.
         *
         * @return  a clone of this instance.
         * @exception  CloneNotSupportedException
         *             Never thrown.
         * @exception  OutOfMemoryError
         *             Thrown if there is not enough memory.
         */
        public Object clone() throws CloneNotSupportedException {
            return super.clone();
        } // clone
    } // Entry

    /**
     * An iterator over this collection.
     *
     * @author  Nathan Fiedler
     */
    protected class Iter implements Iterator {
        /** modification count */
        protected int modCount;
        /** current entry */
        protected Entry currentEntry;
        /** previous entry */
        protected Entry previousEntry;

        /**
         * Creates a Iter.
         *
         * @param  head  Head of stack to iterate.
         */
        public Iter(Entry head, int modCount) {
            this.currentEntry = head;
            this.modCount = modCount;
        } // Iter

        /**
         * Returns true if the iteration has more elements.
         *
         * @return  true if the iterator has more elements.
         */
        public boolean hasNext() {
            return currentEntry != null;
        } // hasNext

        /**
         * Returns the next element in the interation.
         *
         * @return  the next element in the iteration.
         */
        public Object next() {
            if (PancakeStack.this.modCount != modCount) {
                throw new ConcurrentModificationException("stack changed");
            }
            if (!hasNext()) {
                throw new NoSuchElementException("no elements left");
            }
            previousEntry = currentEntry;
            currentEntry = currentEntry.next;
            return previousEntry.data;
        } // next

        /**
         * Removes from the underlying collection the last element
         * returned by the iterator.
         */
        public void remove() {
            if (PancakeStack.this.modCount != modCount) {
                throw new ConcurrentModificationException("stack changed");
            }
            PancakeStack.this.remove(previousEntry);
        } // remove
    } // Iter
} // PancakeStack
