/*********************************************************************
 *
 *	Copyright (C) 1999 Nathan Fiedler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * The program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Template Place, Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * PROJECT:      Utilities
 * MODULE:       SimpleLock Test
 * FILE:         TestSimpleLock.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name     Date           Description
 *      ----     ----           -----------
 *      NF       07/03/00       Initial version
 *
 * DESCRIPTION:
 *      Implements test-wrapper for the SimpleLock class.
 *
 ********************************************************************/

package com.bluemarsh.util.test;

import com.bluemarsh.util.Lock;
import com.bluemarsh.util.LockException;
import com.bluemarsh.util.SimpleLock;

/**
 * Test-wrapper for the SimpleLock class.
 *
 * @author  Nathan Fiedler
 * @version 1.0  7/3/00
 */
public class TestSimpleLock implements Runnable {
    /** My lock. */
    private Lock mylock;
    /** My key. */
    private Object mykey;

    /**
     * Constructs a new TestSimpleLock with the given lock and key.
     *
     * @param  lock  Lock to use.
     * @param  key   Key to use on lock.
     */
    public TestSimpleLock(Lock lock, Object key) {
        mylock = lock;
        mykey = key;
    } // TestSimpleLock

    /**
     * Test the SimpleLock.
     */
    public void run() {
        if (mylock.hasKey(mykey)) {
            System.out.println("ERROR: already have lock for " + mykey);
        } else {
            sleepRandom();
            System.out.println("acquiring lock for " + mykey);
            try {
                mylock.acquire(mykey);
            } catch (LockException le) {
                le.printStackTrace();
                return;
            }
            if (mylock.hasKey(mykey)) {
                System.out.println("lock acquired for " + mykey);
            } else {
                System.out.println("ERROR: do not have lock for " + mykey);
                return;
            }
            sleepRandom();
        }
        System.out.println("releasing lock for " + mykey);
        mylock.release(mykey);
        if (mylock.hasKey(mykey)) {
            System.out.println("failed to release lock for " + mykey);
        } else {
            System.out.println("lock released for " + mykey);
        }
    } // run

    /**
     * Sleep for some random time between 1 and 10 seconds.
     */
    protected void sleepRandom() {
        int rand = (int)(Math.random() * 10) + 1;
        System.out.println(mykey + " waiting for " + rand + " seconds...");
        try {
            Thread.sleep(rand * 1000);
        } catch (InterruptedException ie) {
        }
    } // sleepRandom

    /**
     * Main method, tests the SimpleLock class.
     *
     * @param  args  command-line string arguments
     */
    public static void main(String[] args) {
        // Create a single lock and several keys and workers.
        // Fire them all off at about the same time.
        SimpleLock alock = new SimpleLock();
        for (int i = 1; i <= 10; i++) {
            String key = new String("number " + i);
            TestSimpleLock worker = new TestSimpleLock(alock, key);
            Thread th = new Thread(worker);
            System.out.println("starting worker " + i);
            th.start();
        }
    } // main
} // TestSimpleLock
