# Part of the A-A-P recipe executive: Setup using MSVC
# Last Change: 2004 May 08

# Copyright (c) 2002-2003 stichting NLnet Labs
# Permission to copy and use this file is specified in the file COPYING.
# If this file is missing you can find it here: http://www.a-a-p.org/COPYING

#
# This module sets up variables and actions for using the Microsoft Visual C
# compiler tools.
#

from RecPython import *
import Global
from Action import action_add
from Dictlist import str2dictlist
from RecPos import RecPos


def exists():
    """
    Return TRUE when the MSVC toolchain can be found.
    """
    return program_path("cl") or program_path("vcvars32")


def define_actions():
    """
    Define the actions that MSVC can accomplish.
    """
    rd = Global.globals

    cmd = ("defi = $?DEFINE\n"
            + "@defi = re.sub('-D', '/D', defi)\n"
            + "incl = $?INCLUDE\n"
            + "@incl = re.sub('-I', '/I', incl)\n")

    # TODO: use DEBUG and OPTIMIZE
    rpstack = [ RecPos("compile_msvc c action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "compile_msvc object,libobject,dllobject c"),
            cmd + ":sys $MSVC $CPPFLAGS $defi $incl $CFLAGS /c $source /Fo$target")

    rpstack = [ RecPos("compile_msvc cpp action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "compile_msvc object,libobject,dllobject cpp"),
            cmd + ":sys $MSVC $CPPFLAGS $defi $incl $CXXFLAGS /TP /c $source /Fo$target")

    if not rd.get("MSVC"):
        rd["MSVC"] = "cl"

    # Build action.
    # Change $LIBS from "-lname" to "name.lib".
    rpstack = [ RecPos("build_msvc action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "build_msvc object"),
            "libs = $?LIBS\n"
            "libs = `re.sub('(\\\\A| )-l', ' ', libs)`\n"
            "libs = `sufadd('.lib', libs)`\n"
            ":sys $MSLINK $LINKFLAGS /out:$target $source $libs")
    rpstack = [ RecPos("build_msvc_dll action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "build_msvc_dll object,libobject,dllobject"),
            "libs = $?LIBS\n"
            "libs = `re.sub('(\\\\A| )-l', ' ', libs)`\n"
            "libs = `sufadd('.lib', libs)`\n"
            ":sys $MSLINK $LINKFLAGS /DLL /out:$target $libs $source")
    if not rd.get("MSLINK"):
        rd["MSLINK"] = "link"
    if not rd.get("LINKFLAGS"):
        rd["LINKFLAGS"] = "/nologo"

    rpstack = [ RecPos("build_msvc_lib action") ]
    action_add(rpstack, rd, str2dictlist(rpstack, "build_msvc_lib object,libobject,dllobject"),
            "libs = $?LIBLIBS\n"
            "libs = `re.sub('(\\\\A| )-l', ' ', libs)`\n"
            "libs = `sufadd('.lib', libs)`\n"
            ":sys $MSLIB $LIBFLAGS $source /out:$target $libs")
    # XXX: We're using a variable LIBLIBS here, instead of the usual LIBS.
    # Static libraries typically don't want to link with the common libraries
    # that executables want to link to (in fact, doing so will lead to linker
    # errors), but we still need a way to link other libraries to a library.
    if not rd.get("MSLIB"):
        rd["MSLIB"] = "lib"
    if not rd.get("LIBFLAGS"):
        rd["LIBFLAGS"] = "/nologo"
    # XXX: Should we use ARFLAGS here? The problem with that is that ARFLAGS
    # contains "r" when we get here, so it won't be reverted to /nologo.

def use_actions(scope):
    """
    Setup variables so that the default actions use the MSVC actions.
    """
    scope["C_COMPILE_ACTION"] = "compile_msvc"
    scope["CXX_COMPILE_ACTION"] = "compile_msvc"
    scope["C_BUILD_ACTION"] = "build_msvc"
    scope["CXX_BUILD_ACTION"] = "build_msvc"
    scope["BUILDLIB_ACTION"] = "build_msvc_lib"
    scope["BUILDDLL_ACTION"] = "build_msvc_dll"

    # Avoid using gcc for dependency checks.  It may exist but won't work with
    # the MSVC compiler flags.
    scope["HASGCC"] = "no"
    scope["HASGCCXX"] = "no"

# vim: set sw=4 et sts=4 tw=79 fo+=l:
