!{\src2tex{textfont=tt}}
!!****f* ABINIT/PSolver_hartree
!! NAME
!! PSolver_hartree
!!
!! FUNCTION
!! Given rho(G), compute Hartree potential considering the system as
!! an isolated one. This potential is obtained from the convolution
!! of 1/r and rho(r), treated in Fourier space. A kernel is built from
!! the Fourier transform of 1/r, using a gaussian decomposition of 1/r
!! and this kernel is applied on rho(g). This method is a wrapper around
!! PSolver_Kernel developped for BigDFT.
!!
!! COPYRIGHT
!! Copyright (C) 1998-2005 ABINIT group (DCA, XG, GMR).
!! This file is distributed under the terms of the
!! GNU General Public License, see ~abinit/COPYING
!! or http://www.gnu.org/copyleft/gpl.txt .
!! For the initials of contributors, see ~abinit/doc/developers/contributors.txt .
!!
!! INPUTS
!!  dtset <type(dataset_type)>=all input variables in this dataset
!!  iaction=0 to free the kernel allocated array, 1 to compute the hartree
!!          potential from the density (computing the kernel also, if required).
!!  nfft=(effective) number of FFT grid points (for this processor)
!!  ngfft(18)=contain all needed information about 3D FFT, see ~abinit/doc/input_variables/vargs.htm#ngfft
!!  rhor(dimfft)=electron density in real space.
!!  rprimd(3,3)=dimensional primitive translations in real space (bohr)
!!
!! OUTPUT
!!  vhartr(dimfft)=Hartree potential in real space.
!!
!! PARENTS
!!      fresid,mklocl_wavelets,rhohxc_coll,scfcv
!!
!! CHILDREN
!!      build_kernel,calculate_pardimensions,dimensions_fft,leave_new
!!      mpi_barrier,parbuild_kernel,parpsolver_kernel,psolver_kernel,wrtout
!!
!! SOURCE

#if defined HAVE_CONFIG_H
#include "config.h"
#endif

subroutine PSolver_hartree(dtset, iaction, nfft, ngfft, rhor, rprimd, vhartr)

 use defs_basis
 use defs_datatypes

!This section has been created automatically by the script Abilint (TD). Do not modify these by hand.
#ifdef HAVE_FORTRAN_INTERFACES
 use interfaces_01manage_mpi
 use interfaces_12poisson, except_this_one => PSolver_hartree
#endif
!End of the abilint section

 implicit none

!Arguments ------------------------------------
!scalars
 integer,intent(in) :: nfft, iaction
 type(dataset_type),intent(in) :: dtset
!arrays
 integer,intent(in) :: ngfft(18)
 real(dp),intent(in) :: rprimd(3,3)
 real(dp),intent(in) :: rhor(nfft)
 real(dp),intent(out) :: vhartr(nfft)

!Local saved variables-------------------------
!scalars
!arrays
 integer, save :: kernel_scfOrder ! Order of scaling functions used for the kernel.
 integer, save :: kernel_size(3)  ! size of the grid for the stored
                                  ! kernel.
 real(dp), save :: kernel_hgrid   ! The size of the grid step used when creating the kernel.
 real(dp), allocatable, save :: kernel_array(:, :, :) ! contains the values
                                                      ! of the kernel on the grid.

!Local variables-------------------------------
!scalars
 integer :: i1, i2, i3, i23, i33, ndata
 real(dp) :: hgrid
 real(dp) :: ehartree, exc, evxc, mean_v
 character(len=500) :: message
!arrays
 integer               :: dimfft(3), dimData(3)
#if defined MPI
 integer, save :: ngfft_reversed(3) ! same values as ngfft, 
                                    ! but exchanged x2/x3 and x3/x2
 integer, save :: fft_size(3) ! doubled FFT dimensions.
 integer, save :: ngfft_half(3) ! half size of ngfft
#endif
 real(dp), allocatable :: rhopot(:, :, :) ! Stores the density in n1xn2xn3 when in
                                          ! and the potential when out.
 real(dp), allocatable :: dummy_ion_pot(:, :, :)

 ! If iaction == 0, we free the kernel.
 if (iaction == 0) then
  if (allocated(kernel_array)) then
    write(message, "(A)") "Psolver_hartree() : deallocating kernel..."
   call wrtout(06, message,'COLL')
   deallocate(kernel_array)
  end if
  return
 end if

 ! Can't work in parallel because of zero-padding, the given rho
 ! must be the complete rho. Work should be done to overcome this limitation,
 ! since PSolver_Kernel has its own parallelisation.
 if (ngfft(10) /= 1) then
  write(message, '(a,a,a,a,I0,a)' ) ch10,&
&  ' PSolver_hartree: BUG -',ch10,&
&  '  parallelism is on (ngfft(10) /= 1 : ', ngfft(10), &
&  '  .'
  call wrtout(06, message, 'COLL')
  call leave_new('COLL')
 end if

  ! Get the size depending on wavelets calculations or not
 if (dtset%usewvl == 0) then
   dimData(:) = ngfft(1:3)
   ndata = nfft
   hgrid = rprimd(1, 1) / ngfft(1)
 else
   dimData(:) = dtset%wvl_internal%dpSize(:)
   ndata = dtset%wvl_internal%nDpPoints
   hgrid = 0.5d0 * dtset%wvl_hgrid
 end if

 ! Check if the box is cubic. Can't work if the grid size is not the same
 ! in x, y and z directions.
 if (rprimd(1, 2) /= 0.d0 .or. rprimd(1, 3) /= 0.d0 .or. rprimd(2, 3) /= 0.d0) then
  write(message, '(a,a,a,a,9F6.2,a)' ) ch10,&
&  ' PSolver_hartree: BUG -',ch10,&
&  '  box geometry is not a parallelepipede', rprimd, &
&  '  .'
  call wrtout(06, message, 'COLL')
  call leave_new('COLL')
 end if
 if (dtset%usewvl == 0 .and. &
   & (rprimd(2, 2) / ngfft(2) /= hgrid .or. rprimd(3, 3) / ngfft(3) /= hgrid)) then
  write(message, '(a,a,a,a,3F12.8)' ) ch10,&
&  ' PSolver_hartree: BUG -',ch10,&
&  '  grid is not regular', hgrid, rprimd(2, 2) / ngfft(2), rprimd(3, 3) / ngfft(3)
  call wrtout(06, message, 'COLL')
  call leave_new('COLL')
 end if
 
! allocate(rhopot(ngfft(1), ngfft(2), ngfft(3)))

 ! Compute a new kernel if grid size has changed or if the kernel
 ! has never been computed.
#if defined MPI
 call calculate_pardimensions(ngfft(1), ngfft(2), ngfft(3), &
      & ngfft_reversed(1), ngfft_reversed(2), ngfft_reversed(3), &
      & fft_size(1), fft_size(2), fft_size(3), &
      & ngfft_half(1), ngfft_half(2), ngfft_half(3), &
      & dimfft(1), dimfft(2), dimfft(3), mpi_renreg%nproc)
#else
 call Dimensions_FFT(ngfft(1), ngfft(2), ngfft(3), dimfft(1), dimfft(2), dimfft(3))
#endif
 if (.not. allocated(kernel_array) .or. &
   & kernel_size(1) /= dimfft(1) .or. &
   & kernel_size(2) /= dimfft(2) .or. &
   & kernel_size(3) /= dimfft(3) .or. &
   & kernel_hgrid /= hgrid .or. &
   & kernel_scfOrder /= dtset%nscforder) then
  if (allocated(kernel_array)) then
   deallocate(kernel_array)
  end if
  write(message, "(A,A,A,3I6)") "Psolver_hartree() : building kernel...", ch10, &
                            & " | dimension:", dimfft
  call wrtout(06, message, 'COLL')
#if defined MPI
  allocate(kernel_array(dimfft(1), dimfft(2), dimfft(3) / nproc))
  call MPI_BARRIER(MPI_COMM_WORLD,ierr)
  call ParBuild_Kernel(ngfft(1), nggft(2), ngfft(3), &
       & fft_size(1), fft_size(2), fft_size(3), dimfft(1), dimfft(2), dimfft(3), &
       & hgrid, dtset%nscforder, mpi_enreg%me, mpi_enreg%nproc, kernel_array)
#else
  allocate(kernel_array(dimfft(1) / 2 + 1, dimfft(2) / 2 + 1, dimfft(3) / 2 + 1))
  call Build_Kernel(dimData(1), dimData(2), dimData(3), &
       & dimfft(1), dimfft(2), dimfft(3), &
       & hgrid, dtset%nscforder, kernel_array)
#endif
  ! Storing variables which were used to make the kernel
  kernel_size(:)  = dimfft(:)
  kernel_hgrid    = hgrid
  kernel_scfOrder = dtset%nscforder

  !write(*,*) "Testing kernel..."
  !call test_kernel(ngfft(1), ngfft(2), ngfft(3), kernel_size(1), kernel_size(2), kernel_size(3), &
  !                 hgrid,kernel_array,dummy_ion_pot,rhopot)
 end if

 write(message, "(A,A,A,3I6)") "Psolver_hartree() : compute potential...", ch10, &
      & " | dimension:", dimData
 call wrtout(06, message,'COLL')

 ! Only the 1:ndata part is relevant, other part is buffering for FFT transformations.
 vhartr(1:ndata) = rhor(1:ndata)

#if defined MPI
 call ParPSolver_Kernel(dimData(1), dimData(2), dimData(3), &
      & kernel_size(1), kernel_size(2), kernel_size(3), &
      & hgrid, kernel_array, dtset%ixc, dummy_ion_pot, vhartr, &
      & ehartree, exc, evxc, mpi_enreg%me, mpi_enreg%nproc)
#else
 call PSolver_Kernel(dimData(1), dimData(2), dimData(3), &
                   & kernel_size(1), kernel_size(2), kernel_size(3), &
                   & hgrid, kernel_array, .false., dummy_ion_pot, &
                   & vhartr, ehartree, exc, evxc)
#endif

 if (ndata /= nfft) then
  ! Pad with zeros in the case consistent values are
  ! less than nfft
  vhartr(ndata + 1:nfft) = real(0., dp)
 end if
 
end subroutine PSolver_hartree
!!***
