/*
------------------------------------------------------------------
File sparse_jac_hess.cpp of ADOL-C version 1.10.0 as of July/01/05
------------------------------------------------------------------
 
   Example: Drivers for sparse Jacobians and Hessians 
  
   Last changed:
    050701 andrea: new example
  
------------------------------------------------------------------
*/
#include "math.h"
#include "stdlib.h"
#include "stdio.h"

#include "../../../adolc/adolc.h"

#define tag_f 1
#define tag_c 2

double  feval(double *x);
adouble feval_ad(adouble *x);

void   ceval_ad(adouble *x, adouble *c);
void   ceval(double *x, double *c);

void printmat(char* kette, int n, int m, double** M);
void printmatint(char* kette, int n, int m, int** M);

int main() {
    unsigned int n=6, m=3;
    double f, x[6],c[3];
    adouble fad, xad[6], cad[3];

    unsigned int i, j;

    /* full matrices */
    double **J, **H;
    double **Seed_J, **Seed_H;
    int p_J, p_H;
    J = myalloc2(m,n);
    H = myalloc2(n,n);

    /* sparsity pattern of Jacobian */
    unsigned int  *rb=NULL;          /* dependent variables          */
    unsigned int  *cb=NULL;          /* independent variables        */
    unsigned int  **JP=NULL;         /* compressed block row storage */
    int ctrl[2];

    JP = (unsigned int **) malloc(n*sizeof(unsigned int*));
    ctrl[0] = 0;
    ctrl[1] = 0;

    /* sparsity pattern of Hessian */
    unsigned int  **HP=NULL;         /* compressed block row storage */
    HP = (unsigned int **) malloc(n*sizeof(unsigned int*));

    /* compressed row format for Jacobian */
    unsigned int *jac_rind  = NULL;
    unsigned int *jac_cind  = NULL;
    double       *jac_values = NULL;
    int jac_nnz;

    /* compressed row format for Hessian */
    unsigned int *hess_rind  = NULL;
    unsigned int *hess_cind  = NULL;
    double       *hess_values = NULL;
    int hess_nnz;

    for(i=0;i<n;i++)
        x[i] = log(1.0+i);

    f = feval(x);
    ceval(x,c);

    printf("\n f = %e\n c =  ",f);
    for(j=0;j<m;j++)
        printf(" %e ",c[j]);
    printf("\n");

    /* Tracing of function f(x) */
    trace_on(tag_f);
    for(i=0;i<n;i++)
        xad[i] <<= x[i];

    fad = feval_ad(xad);

    fad >>= f;
    trace_off();


    /* Tracing of function c(x) */
    trace_on(tag_c);
    for(i=0;i<n;i++)
        xad[i] <<= x[i];

    ceval_ad(xad,cad);

    for(i=0;i<m;i++)
        cad[i] >>= c[i];
    trace_off();

    /* Full Jacobian: */
    jacobian(tag_c,m,n,x,J);

    printmat(" J",m,n,J);
    printf("\n");

    /* Sparsity pattern of Jacobian: */

    jac_pat(tag_c, m, n, x, rb, cb, JP, ctrl);

    printf("\n");
    printf("Sparsity pattern of Jacobian: \n");
    for (i=0;i<m;i++) {
        printf(" %d: ",i);
        for (j=1;j<=JP[i][0];j++)
            printf(" %d ",JP[i][j]);
        printf("\n");
    }
    printf("\n");

    /* Entries in row-compressed format using sparse_jac: */

    sparse_jac(tag_c, m, n, 0, x, &jac_nnz, &jac_rind, &jac_cind, &jac_values);

    printf("In sparse format:\n");
    for (i=0;i<(unsigned int)jac_nnz;i++)
        printf("%2d %2d %10.6f\n\n",jac_rind[i],jac_cind[i],jac_values[i]);

    /* Entries in row-compressed format using own seed matrix: */

    generate_seed_jac(m, n, JP, &Seed_J, &p_J);
    fov_forward(tag_c,m,n,p_J,x,Seed_J,c,J);
    printmat("compressed J:",m,p_J,J);

    /* Full Hessian: */
    printf("\n");
    hessian(tag_f,n,x,H);

    printf("\n");
    printmat("H:",n,n,H);

    /* Sparsity pattern of Hessian: */

    hess_pat(tag_f,n,x,HP,0);

    printf("\n");
    printf("Sparsity pattern of Hessian: \n");
    for (i=0;i<n;i++) {
        printf(" %d: ",i);
        for (j=1;j<=HP[i][0];j++)
            printf(" %d ",HP[i][j]);
        printf("\n");
    }
    printf("\n");

    /* Entries in row-compressed format using sparse_hess: */

    sparse_hess(1, n,0,x,&hess_nnz,&hess_rind,&hess_cind,&hess_values);

    printf("In sparse format:\n");
    for (i=0;i<(unsigned int)hess_nnz;i++)
        printf("%2d %2d %10.6f\n\n",hess_rind[i],hess_cind[i],hess_values[i]);

    /* Entries in row-compressed format using own seed matrix: */

    generate_seed_hess(n, HP, &Seed_H, &p_H);
    hess_mat(tag_f,n,p_H,x,Seed_H,H);
    printmat("compressed H:",p_H,n,H);
}


/***************************************************************************/

double feval(double *x) {
    double res;

    res = (x[0] - 1)*(x[0] -1) + (x[1] - 2)*(x[1] -2)  + (x[2] - 3)*(x[2] -3);
    res += x[0]*x[1];
    res += cos(x[3]);
    res += sin(x[4]);
    res += exp(x[5]);
    res += sin(x[4]*x[5]);

    return res;
}

void ceval(double *x, double *c) {
    c[0] = 2*x[0]+x[1]-2.0;
    c[1] = x[2]*x[2]+x[3]*x[3]-2.0;
    c[2] = 3*x[4]*x[5] - 3.0;
}

/***************************************************************************/

adouble feval_ad(adouble *x) {
    adouble res;

    res = (x[0] - 1)*(x[0] -1) + (x[1] - 2)*(x[1] -2)  + (x[2] - 2)*(x[2] -2);
    res += x[0]*x[1];
    res += cos(x[3]);
    res += sin(x[4]);
    res += exp(x[5]);
    res += sin(x[4]*x[5]);

    return res;
}

void ceval_ad(adouble *x, adouble *c) {
    c[0] = 2*x[0]+x[1]-2.0;
    c[1] = x[2]*x[2]+x[3]*x[3]-2.0;
    c[2] = 3*x[4]*x[5] - 3.0;
}

/***************************************************************************/

void printmat(char* kette, int n, int m, double** M) {
    int i,j;

    printf("%s \n",kette);
    for(i=0; i<n ;i++) {
        printf("\n %d: ",i);
        for(j=0;j<m ;j++)
            printf(" %10.4f ", M[i][j]);
    }
    printf("\n");
}

void printmatint(char* kette, int n, int m, int** M) {
    int i,j;

    printf("%s \n",kette);
    for(i=0; i<n ;i++) {
        printf("\n %d: ",i);
        for(j=0;j<m ;j++)
            printf(" %d ", M[i][j]);
    }
    printf("\n");
}
