/*
   Project: UL

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston 

   Created: 2005-12-09 14:47:28 +0100 by michael johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/

#ifndef _ULANALYSISMANAGER_H_
#define _ULANALYSISMANAGER_H_

#include <Foundation/Foundation.h>
#include <AdunKernel/AdunDataSet.h>
#include "ULFramework/ULAnalysisPlugin.h"
#include "ULFramework/ULDatabaseInterface.h"
#include "ULFramework/ULSimulation.h"

/**
This class manages the analysis of a set of input object using a chosen plugin.
The available plugins that can be used are dependant on the types and number of input
objects. The results of analysis is usually a one or more AdDataSet instances.
*/

@interface ULAnalysisManager : NSObject
{
	@private
	id energies;
	id currentPlugin;
	id currentPluginName;
	id results;				//!< The last analysis results
	NSMutableArray* outputObjectsReferences;	//!< Array of the objects that generated the current output set.
	NSString* pluginDir;			//!< Directory containing the analyis plugin bundles
	NSMutableArray* availablePlugins;	//!< Array of plugin bundle names
	NSMutableArray* inputObjects;		//!< Array of input objects
	NSMutableDictionary* objectsCountDict;	//!< Count of each type of input object
	NSMutableDictionary* pluginInfoDict;	//!< Contains information about the inputs needed by each plugin
}
/**
Adds \e object to the analysis input. Does nothing if object
is nil
*/
- (void) addInputObject: (id) object;
/**
Removes \e object from the analysis input. Does nothing if
\e object is not among the inputs.
*/
- (void) removeInputObject: (id) object;
/**
Clears the analysis inputs
*/
- (void) removeAllInputObjects;
/**
Returns an array of containing the input objects.
If their are none the array will be empty
*/
- (NSArray*) inputObjects;
/** Return YES if the analysis manager contain any input objects.
No otherwise
*/
- (BOOL) containsInputObjects;
/**
Returns an array containing the name of plugins that can be applied
to the set of input objects i.e. the plugins who can take the current
input objects as their input
*/
- (NSArray*) pluginsForCurrentInputs;
/** 
Returns the number of input objects of \e class 
*/
- (int) countOfInputObjectsOfClass: (NSString*) className;
/** Returns the last array of data sets produced by applying a plugin to a set of inputs
*/
- (NSArray*) outputDataSets;
/**
Returns the analysis string (if any) which provides information on the last analysis 
performed
*/
- (NSString*) outputString;
/**
Saves the output data set \e dataSet to the file system database adding all necessary
references. The object passed must be one of the data sets returned by the outputDataSets method.
The ability of this method to correctly assign output references
is not affected if input objects are removed or added before it is called i.e.
ULAnalysisManager remembers who generated \e dataSet.
*/
- (void) saveOutputDataSet: (AdDataSet*) dataSet;
/**
Applies the plugin \e name to the current input objects using \e options
and returns the plugin output (either nil or an array of AdDataSet objects)
If \e name is not in the array returned by pluginsForCurrentInputs an
NSInvalidArgumentException is raised.
\todo Change so returns void - user currentOutputObject to get results.
*/
- (id) applyPlugin: (NSString*) name withOptions: (NSMutableDictionary*) options;
/**
Returns the options for plugin \e name. Note a plugin may have no options.
If \e name is not in the array returned by pluginsForCurrentInputs an
NSInvalidArgumentException is raised.
*/
- (id) optionsForPlugin: (NSString*) name;
@end

#endif // _ULANALYSISMANAGER_H_

