/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <qfileinfo.h>

#include <orb.h>
#include <magellan.h>
#include <mailclasses.h>
#include <indexclass.h>
#include <transactionserver.h>
#include <attributemarshall.h>
#include <conversions.h>
#include <messagedescriptor.h>
#include <messagedevice.h>
#include <messagefactory.h>
#include <servernotifier.h>
#include <accounts.h>
#include <accountmanager.h>
#include <mimecodec.h>

bool ObjectRequestBroker::setObjectData(const QString &url, const QByteArray &data)
{
	// debug
	// printf("orb: setObjectData called on %s, data size %d\n", (const char *)url, data.size());
	
  if(!exists(url))
  {
		// debug
		printf("orb: warning: object %s does not exist\n", (const char *)url);
		
    err="Non-existent object or unknown protocol";
    return false;    
  }
  else if(isFolder(url))
  {
		// debug
		printf("orb: warning: setObjectData can't be applied on folders (%s)\n", (const char *)url);
		
    err="Can't apply setObjectData() on a folder";
    return false;
  }
	else if(url.contains(".attr") || url.contains(".fattr"))
	{
		// debug
		printf("orb: this is an attribute object\n");
			
		if(subtype(url)==Magellan::PartAttribute)
			return setPartAttributeData(url, data);
		
		err="Cannot modify a read-only attribute object";
		return false;
	}
  else if(type(url)==Magellan::Message)
  {
		if(subtype(url)==Magellan::Part)
		{
			QString object=url.left(url.find("."));
			int part=url.mid(url.find(".")+1).toInt();
			return setPartData(object, part, data);
		}
		else
		{
	    return setMailData(url, data);
		}
  }
	else
	{
		return setCollectionObjectData(url, data);
	}
}

bool ObjectRequestBroker::setCollectionObjectData(const QString &url, const QByteArray &data)
{
  ObjectReference *objectRef=objectReference(url);
  if(!objectRef)
  {
    printf("Warning: objectReference() returned null when a valid pointer was expected, bailing out\n");
    err="Internal error in URLManager::objectReference()";
    return false;
  }
  
	// debug
	// printf("orb: setting data on collection object %s(%p), data size %d...\n", (const char *)url, objectRef, data.size());
	
  objectRef->rawData=data;
  objectRef->dirty=true;
  TransactionServer::thisInstance()->save(objectRef);
	    
  return true;
}

bool ObjectRequestBroker::setMailData(const QString &url, const QByteArray &data)
{
	IndexClass *index=indexReference(url);
	if(!index)
	{
		printf("Warning: indexReference() returned null when a valid pointer was expected, bailing out\n");
		err="Internal error in URLManager::indexReference()";
		return false;
	}
	
	if(index->isLocked())
	{
		printf("Warning: message is locked, cannot write\n");
		err="Message was locked by the mail spooler and cannot be modified";
		return false;
	}
		
	MessageDevice *dev=MessageFactory::thisInstance()->getMessage(index);
	
	if(!dev)
	{
		printf("Warning: MessageFactory::getMessage() returned null when a valid pointer was expected, bailing out\n");
		err="Internal error in MessageFactory::getMessage()";
		return false;
	}
	
	// debug
	// printf("orb: setting data on mail object %s(index:%p, device:%p), data size %d...\n", (const char *)url, index, dev, data.size());
	
	QDataStream stream(data, IO_ReadOnly);
	MessageDescriptor &descriptor=dev->getDescriptor();
	stream>>descriptor;
	
	bool triggerHierarchyUpdate=false;
	
	// set index unread mark
	bool oldUnreadMark=index->getUnreadMark();
	index->setUnreadMark(descriptor.status=="New");
	
	// set index parent id
	// ... not sure about implications
	// update descriptor relatedIndexIDList;
	// ... not sure about implications
	
	// set default account if needed
	if(url.find("sysmail_outbox")==0 || url.find("sysmail_drafts")==0)
	{
		Account *acc;
		if(descriptor.account.isEmpty())
		{
			acc=AccountManager::ref()->getDefaultAccount();
			if(acc) descriptor.account=acc->accname;
		}
		else
		{
			acc=AccountManager::ref()->getAccount(descriptor.account);
		}

		// set from address, reply-to and messageID based on the account
		if(acc)
		{
			descriptor.from="\""+acc->name+"\" <"+acc->from+">";
			if(!acc->reply.isEmpty()) descriptor.replyTo="\""+acc->name+"\" <"+acc->reply+">";
			descriptor.messageID= MIMECodec::encodeMessageId(descriptor.indexID.right(5), acc->name)
			                      +"@"+acc->smtp;
		}
	}

	// set index message id
	index->setMessageID(descriptor.messageID);
	
	if(oldUnreadMark!=index->getUnreadMark())
	{
		if(index->getUnreadMark()) index->getParentFolder()->incrementUnread();
		else index->getParentFolder()->decrementUnread();
		
		ServerNotifier::thisInstance()->objectChanged(index->getParentFolder()->name());
	}
	if(triggerHierarchyUpdate)
		ServerNotifier::thisInstance()->hierarchyChanged(index->getParentFolder()->name()+"/"+index->getID());
	
	dev->saveDescriptor();
	
	return true;
}

bool ObjectRequestBroker::setMailPropertyData(const QString &url, const QString &property, const QByteArray &data)
{
	return false;
}

bool ObjectRequestBroker::setPartData(const QString &url, int part, const QByteArray &data)
{
	IndexClass *idx=indexReference(url);

	if(!idx)
	{
		// debug
		printf("orb: could not find index %s\n", (const char *)url);
		
		err="Internal error in URLManager::indexReference()";
		return false;
	}
	
	// debug
	// printf("orb: setting part data (%s.%d)\n", (const char *)url, part);
	
	err="Could not set part data: unknown error";
	MessageDevice *dev=new MessageDevice(idx);
	bool res=dev->setPartData(part, data);
	delete dev;
	
	return res;
}

bool ObjectRequestBroker::setPartAttributeData(const QString &url, const QByteArray &data)
{
	QString objectUrl=url.left(url.find('.'));  
  _PartAttribute attr=_narrowPartAttribute(data);
  IndexClass *idx=indexReference(objectUrl);
	
	if(!idx)
	{
		// debug
		printf("orb: warning, setPartAttributeData could not find index %s\n", (const char *)objectUrl);
		
		err="Internal error in URLManager::indexReference()";
		return false;
	}

  int pBegin=url.find('.')+1, pEnd=url.findRev('.');
  MimePart *part=idx->getPartAt(url.mid(pBegin, pEnd-pBegin).toInt());
	
	// debug
	// printf("orb: setting properties for part %d, index %s\n", url.mid(pBegin, pEnd-pBegin).toInt(), (const char *)idx->getID());

	part->mimetype=attr.mimetype;
	part->name=attr.name;
	part->cid=attr.cid;
	part->encoding=attr.encoding;
	part->charset=attr.charset;
	part->type=attr.mtype;
	
	idx->getParentFolder()->saveIndex(idx);
	
  return true;
}
