/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#ifndef HAS_VALARM
	#define HAS_VALARM

#include <qobject.h>
#include <qstring.h>
#include <qdatetime.h>
#include <metaobject.h>

class VAlarm : public QObject
{
	Q_OBJECT
public:
	enum AlarmType { Unknown=0, Audio, Email, Display, Procedure };
	enum BufferType { Empty=0, URI, EmailAddresses, BinaryData, Text };
	enum AlarmStyle { AbsoluteTime=0, RelativeTime=1};
private:
	bool hasErrors;
	MetaObject *object;		// this is a pointer to the real object
	QDateTime absoluteTrigger;
	long relativeTrigger;
	long duration;
	int  repeat;
	AlarmType alarmType;	// this represents the alarm type
	QString *buffer;			// this represents data function of which 
												// type is on.
	BufferType bufferType;// the type of buffer so I should know what
												// to do.
	QString *summary;  		// the subject or the exec's description
	QString attendees;		// the attendees
	
	
	bool setBuffers();
	bool setBuffersAudio();
	bool setBuffersEmail();
	bool setBuffersDisplay();
	bool setBuffersProcedure();
	bool setAlarmType();
	bool setDuration();
	bool setRepeat();
	bool setTime();
public slots:
	/**
	 *	This method is called by the "act" if alarm type is AUDIO
	 */
	bool actAudio();
	/**
	 *	This method is called by the "act" if alarm type is EMAIL
	 */
	bool actEmail();
	/**
	 *	This method is called by the "act" if alarm type is DISPLAY
	 */
	bool actDisplay();
	/**
	 *	This method is called by the "act" if alarm type is PROCEDURE
	 */
	bool actProcedure();
public:
	VAlarm(MetaObject *mo=NULL);
	virtual ~VAlarm() {};
	/**
	 *	This method is used to regenerate the alarm's code from the metaobject.
	 * @return a string containing the alarm's code.
	 */
	QString recompose(AlarmStyle style);
	/**
	 *	This method is used to check if the object has been initialised without
	 * errors.
	 * @return true if the object has no errors.
	 */
	bool objectOK() { return !hasErrors; }
	/**
	 *	This method is used to check if the alarm needs the parent's datetime
	 * @return true if i need the parent's time, false if not.
	 */
	bool needParentTime() { return relativeTrigger!=0; }
	/**
	 *	This method is used to set an absolute time for an alarm. It should be 
	 * used only when the alarm has a relativeTrigger set (just use
	 * @ref needParentTime)
	 * @param absdate the reference date from the parent.
	 */
	void setAbsoluteTime(QDateTime absdate);
	/**
	 *	This method should be called just before executing the "act" from
	 * the alarm.<br>
	 *	The topology works like this:<br>
	 *	In the qtimer function, if CurrentDateTime>absoluteTrigger then
	 *		updateTrigger<br>
	 *		act
	 */
	bool updateTrigger();
	/**
	 * This method is used for initializing the trigger to the current date
	 * Should be called the first time the qtimer calls the function.
	 * @param present is a QDateTime value (usually QDateTime::currentDateTime
	 * @return true if the alarm is still active, false if the alarm is no
	 * longer active.
	 */
	bool initTrigger(QDateTime present);
	/**
	 *	This method is designed to be called by the qtimer when necessary;
	 *	It will call act*, depending on what type of alarm is.
	 *	NOTE: considering the fact that only M$Window$ programs are using
	 * the "PROCEDURE" type, I have decided to support in "ATTACH" only
	 * URIs and the method is designed for running local programs (if
	 * there is a field (X-...)  X-LOCATION:local). This is recognised by
	 * magellan programs.
	 *
	 *	For example: <br>
	 *		<code>ATTACH;FMTTYPE=application/binary:fortune</code><br>
	 *		<code>X-LOCATION:local</code><br>
	 *		<code>X-CAPTURE:stdio</code><br>
	 *	This should use ProcessRun object to run fortune and display the result
	 * in a nice dialog box (see kprocwrap.h).
	 *	@return true if the action has been succesfully accomplished (always)
	 */
	virtual bool act();
	/**
	 *	In qtimer call this to test against the current date/time.
	 *	@return the date/time when the trigger is supposed to be activated
	 */
	QDateTime getTrigger() { return absoluteTrigger; }
};

#endif








