// ---------------------------------------------------------------------------
// - Hasher.hpp                                                              -
// - afnix cryptography - base message hasher class definition               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_HASHER_HPP
#define  AFNIX_HASHER_HPP

#ifndef  AFNIX_INPUT_HPP
#include "Input.hpp"
#endif

namespace afnix {

  /// The Hasher class is a base class that is used to build a message
  /// hash. The hash result is stored in an array of bytes and can be
  /// retreived byte by byte or as a formatted printable string
  /// @author amaury darsch

  class Hasher : public virtual Object {
  protected:
    /// the hasher name
    String d_name;
    /// requested buffer size
    long    d_size;
    /// the message low length
    t_octa  d_llen;
    /// the message upper lenght
    t_octa  d_ulen;
    /// the buffer count
    long    d_count;
    /// the buffer data
    t_byte* p_data;
    /// the hash size
    long    d_hlen;
    /// the hash result
    t_byte* p_hash;

  public:
    /// create a hasher object by name and size
    /// @param name the hasher name
    /// @param size the block size
    /// @param hlen the hash length
    Hasher (const String& name, const long size, const long hlen);

    /// destroy the hasher
    ~Hasher (void);

    /// @return the class name
    String repr (void) const;

    /// reset this hasher
    virtual void reset (void);

    /// @return the hasher name
    virtual String getname (void) const;

    /// @return the hasher length
    virtual long gethlen (void) const;

    /// @return the hash value by index
    virtual t_byte gethash (const long index) const;

    /// compute a message hasher from a string
    /// @param msg the string message to compute
    virtual String compute (const String& msg);

    /// compute a message hasher from an input stream
    /// @param is the input stream
    virtual String compute (Input& is); 

    /// @return the formatted message hash
    virtual String format (void) const =0;

  protected:
    /// clear this hasher counter
    virtual void clear (void);

    /// @return the buffer count
    virtual long getcount (void) const;

    /// @return the buffer low length
    virtual t_octa getllen (void) const;

    /// @return the buffer upper length
    virtual t_octa getulen (void) const;

    /// copy some data in the buffer by size and return the copied length
    /// @param data the data to copy in the buffer
    /// @param size the data size
    virtual long copy (const t_byte* data, const long size);

    /// copy some data in the buffer from a stream
    /// @param is the input stream used as input
    virtual long copy (Input& is);

    /// update the message state with the buffer data
    virtual void update (void) =0;

    /// process a message by data
    /// @param data the data to process
    /// @param size the data size
    virtual void process (const t_byte* data, const long size) =0;
 
    /// process a message with an input stream
    /// @param is the input stream to process
    virtual void process (Input& is) =0;

    /// finish the message by padding the data and return the hasher
    virtual void finish (void) =0;

  private:
    // make the copy constructor private
    Hasher (const Hasher&);
    // make the assignment operator private
    Hasher& operator = (const Hasher&);

  public:
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
