// ---------------------------------------------------------------------------
// - Datagram.cpp                                                            -
// - afnix:net module - datagram class implementation                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Ascii.hpp"
#include "Vector.hpp"
#include "Integer.hpp"
#include "Datagram.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"
#include "cnet.hpp"
#include "cerr.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default datagram
  
  Datagram::Datagram (const int sid, t_word dport, const t_byte* daddr,
		      const char* buf, const long size) {
    d_sid   = sid;
    p_daddr = c_cpaddr (daddr);
    d_dport = dport;
    d_buffer.add (buf, size);
  }

  // destroy this datagram
  
  Datagram::~Datagram (void) {
    close ();
    delete [] p_daddr;
  }

  // return the class name

  String Datagram::repr (void) const {
    return "Datagram";
  }

  // close this datagram

  bool Datagram::close (void) {
    wrlock ();
    if (Object::uref (this) == true) d_sid = -1;
    unlock ();
    return true;
  }

  // read a character from the datagram

  char Datagram::read (void) {
    wrlock ();
    try {
      char result = d_buffer.read ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write one character to the datagram sender
  
  void Datagram::write (const char value) {
    wrlock ();
    long count = c_ipsendto (d_sid, d_dport, p_daddr, &value, 1);
    unlock ();
    if (count < 0) throw Exception ("write-error", c_errmsg (count));
  }

  // write a buffer to the datagram sender
  
  void Datagram::write (const char* value) {
    wrlock ();
    long size  = Ascii::strlen (value);
    long count = c_ipsendto (d_sid, d_dport, p_daddr, value, size);
    unlock ();
    if (count < 0) throw Exception ("write-error", c_errmsg (count));
  }

  // return true if the datagram buffer is empty

  bool Datagram::iseof (void) const {
    rdlock ();
    bool result = (d_buffer.length () == 0);
    unlock ();
    return result;
  }

  // return true if the datagram buffer is not empty

  bool Datagram::valid (const long) const {
    rdlock ();
    bool result = (d_buffer.length () != 0);
    unlock ();
    return result;
  }

  // return the peer address

  Address* Datagram::getpeeraddr (void) const {
    rdlock ();
    try {
      Address* result = new Address (p_daddr);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the peer port

  t_word Datagram::getpeerport (void) const {
    rdlock ();
    t_word result = d_dport;
    unlock ();
    return result;
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 1;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_WRITE = zone.intern ("write");

  // return true if the given quark is defined

  bool Datagram::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark

  Object* Datagram::apply (Runnable* robj, Nameset* nset, const long quark,
			   Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_WRITE) {
	write ((char*) nilp);
	return nilp;
      }
    }
    // call the socket method
    return Socket::apply (robj, nset, quark, argv);
  }
}
