// ---------------------------------------------------------------------------
// - Ico.cpp                                                                 -
// - afnix:csm service - intercom cloud object class implementation          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2016 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Ico.hpp"
#include "Csmsid.hxx"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // this procedure returns a new rco for deserialization
  static Serial* mksob (void) {
    return new Ico;
  }
  // register this ico serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_ICO_ID, mksob);

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a nil object

  Ico::Ico (void) {
    d_ixo = "";
  }

  // create an intercom object by command and rco

  Ico::Ico (const String& ixo, const Rco& rco) : Rco (rco) {
    d_ixo = ixo;
  }

  // copy construct this intercom object

  Ico::Ico (const Ico& that) {
    that.rdlock ();
    try {
      // set the base object
      Rco::operator = (that);
      // set locally
      d_ixo  = that.d_ixo;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // return the ico class name

  String Ico::repr (void) const {
    return "Ico";
  }

  // return a clone of this object

  Object* Ico::clone (void) const {
    return new Ico (*this);
  }

  // return the ico serial code

  t_byte Ico::serialid (void) const {
    return SERIAL_ID;
  }

  // serialize this ico

  void Ico::wrstream (OutputStream& os) const {
    rdlock ();
    try {
      // write the base object
      Rco::wrstream   (os);
      // save the intercom command
      d_ixo.wrstream  (os);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // deserialize this ico

  void Ico::rdstream (InputStream& is) {
    wrlock ();
    try {
      // read the base object
      Rco::rdstream (is);
      // read the command
      d_ixo.rdstream (is);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the intercom command

  void Ico::setixo (const String& ixo) {
    wrlock ();
    try {
      d_ixo = ixo;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the intercom command

  String Ico::getixo (void) const {
    rdlock ();
    try {
      String result = d_ixo;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // assign a object to this one

  Ico& Ico::operator = (const Ico& that) {
    // check for self assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      // copy the base object
      Rco::operator = (that);
      // copy locally
      d_ixo = that.d_ixo;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }
  
  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 2;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GETIXO = zone.intern ("get-intercom-command");
  static const long QUARK_SETIXO = zone.intern ("set-intercom-command");

  // create a new object in a generic way

  Object* Ico::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // create a default registered object
    if (argc == 0) return new Ico;
    // check for 2 arguments
    if (argc == 2) {
      String  ixo = argv->getstring (0);
      Object* obj = argv->get (1);
      Rco*    rco = dynamic_cast <Rco*> (obj);
      if (rco == nilp) {
	throw Exception ("type-error", "invalid object with ico constructor",
			 Object::repr (obj));
      }
      return new Ico (ixo, *rco);
    }
    throw Exception ("argument-error",
                     "too many argument with ico constructor");
  }

  // return true if the given quark is defined

  bool Ico::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Rco::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* Ico::apply (Runnable* robj, Nameset* nset, const long quark,
		      Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETIXO) return new String (getixo  ());
    }
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETIXO) {
	String ixo = argv->getstring (0);
	setixo (ixo);
	return nilp;
      }
    }
    // call the rco method
    return Rco::apply (robj, nset, quark, argv);
  }
}
