/*******************************************************************************
 *  PROJECT: Agave
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2006 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/
#include <string>
#include <fstream>
#include <list>

#include <glib/gstdio.h>

#include "palette-selector.h"
#include "paletteparser/palette.h"
#include "gcs-conf.h"
#include "core/log-stream.h"

namespace gcs
{
    namespace Widgets
    {
        PaletteSelector::PaletteSelector() :
            m_refListStore(Gtk::ListStore::create(m_columns))
        {
            set_model(m_refListStore);
            pack_start(m_columns.m_columnName); // only display name, not path

            LOG("Looking for installed palette files...");
            // read the directory to see what palette files are there
            Glib::Dir dir(AGAVE_PALETTEDIR);
            typedef std::list<Glib::Dir::iterator::value_type> filelist_t;
            filelist_t files(dir.begin(), dir.end());
            // there is no path information -- we need to prepend the path
            // information to each file
            for (filelist_t::iterator i = files.begin(); i != files.end(); ++i)
            {
                std::string path_prefix(AGAVE_PALETTEDIR "/");
                (*i).insert(0, path_prefix);
            }

            // now add files from the user's palette directory
            try
            {
                Glib::Dir userdir(Conf::USER_PALETTE_DIR);
                for (Glib::Dir::iterator i = userdir.begin(); i != userdir.end(); ++i)
                {
                    files.push_back(Conf::USER_PALETTE_DIR + "/" + *i);
                }
            }
            catch (Glib::Error& e)
            {
                // directory doesn't exist, so let's create it
                g_mkdir(Conf::USER_PALETTE_DIR.c_str(), 0755);
            }
            // TODO: also check the standard "CREATE" resource directory when
            // spec is somewhat finished (see
            // http://create.freedesktop.org/wiki/index.php/Shared_resources )

            // now check to see if the files are valid palette files
            for (filelist_t::iterator iter = files.begin(); iter != files.end(); ++iter)
            {
                std::string filename = *iter;
                std::ifstream infile(filename.c_str());
                if (infile.is_open())
                {
                    try
                    {
                        // FIXME: we don't really want to parse the full palette
                        // just to get the name here.  Maybe need another
                        // interface to palette parser to just get the header
                        // info
                        pp::Palette palette;
                        palette.parse(infile);
                        // add valid files to the ComboBox
                        Gtk::TreeModel::iterator treeiter = m_refListStore->append();
                        (*treeiter)[m_columns.m_columnName] = palette.name();
                        (*treeiter)[m_columns.m_columnFile] = filename;
                        LOG("Found palette file: " << filename);
                    }
                    catch(pp::ParseError& e)
                    {
                        //it wasn't a valid palette, move on
                    }
                }
            }
        }

        PaletteSelector::~PaletteSelector()
        {
        }


        std::string PaletteSelector::get_palette_file(void) const
        {
            return get_active()->get_value(m_columns.m_columnFile);
        }


        void PaletteSelector::set_palette_file(const std::string& filename)
        {
            for (Gtk::TreeModel::iterator iter =
                    m_refListStore->children().begin(); iter !=
                    m_refListStore->children().end(); ++iter)
            {
                if (iter->get_value(m_columns.m_columnFile) == filename)
                {
                    set_active(iter);
                    break;
                }
            }
        }

    } // namespace Widgets
} // namespace gcs
