/**********************************************************************
 ** Configs class: Stores all configurations, based on what it reads in
 **                from the configuration file and provides access to
 **                those configurations.
 **
 **
 **
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef CONFIGS_C
#define CONFIGS_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "global.h"
#include "newfuncts.h"
#include "memchk.h"
#include "configs.h"
#include "utils.h"
#include "lexer.h"

/***********************************************************************
 ** Configs (constructor) - Opens the config file and loads it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Configs::Configs()
{
   conf_flags = new_Flags(1);
   init_configs();
}


/***********************************************************************
 ** ~Configs (destructor) - destroys the config object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Configs::~Configs()
{
	if (conf_flags != NULL)
		delete_Flags(conf_flags);
}


/***********************************************************************
 ** init_configs - initializes the configs to use default if no entry is
 **                found.
 **
 ***********************************************************************/

void Configs::init_configs()
{
   basedir = "/usr/local/aime";
   mudname = "AIME Codebase";
   gameport = 9993;
   buildport = 9992;
   maxpasswdtries = 3;
   minpasswdlen = 6;
   pollspersec = 5;
   maxreadsperpoll = 5;
   secs_in_day = 36000;
   days_in_year = 365;
   first_day_summer = 70;
   first_day_winter = 240;

   // Here you would set any flag defaults that are not set to false
   conf_flags->set_flag(CF_DAEMONIZE);
   conf_flags->set_flag(CF_LOGDEATH);
   conf_flags->set_flag(CF_LOGFLAGS);
   conf_flags->set_flag(CF_LOGFORCE);
   conf_flags->set_flag(CF_LOGCLONE);
   conf_flags->set_flag(CF_LOGBHOSTS);
   conf_flags->set_flag(CF_LOGBANHOSTS);
   conf_flags->set_flag(CF_LOGDELETE);
   conf_flags->set_flag(CF_GAMEALLOW);
   conf_flags->set_flag(CF_LOGBOOT);
   conf_flags->set_flag(CF_LOOKUPHOSTS);

   timezone = "U.S. Central";
   buildprompt = "&+cB&+C>&* ";
   masteruser = "xxxxxxxx";
   masterpasswd = "xxxxxxxx";

   secs_player_heal = 5;
   secs_mobile_heal = 30;
   secs_player_magic = 10;
   secs_player_endur = 10;
   secs_level_check = 10;
   weight_str_mult = 25;
   max_global_exp = 500000;
   max_num_attacking = 3;
   secs_till_timeout = 600;
   max_line_len = 70;
   default_pager_lines = 24;
   users_level_chain = "none";
   hardcoded_areaname = "limbo";
   backup_locname = "void";
   backup_loctitle = "The Infinite Void";
   backup_locdesc = "\nThis is a vacant void where players wait for a world to b\n"
					"created or a start location to be assigned.\n";
   entry_locname = "entry";
   entry_loctitle = "Vacant Entryway";
   entry_locdesc = "\nThis is a vacant void where players wait for a world to be created\n"
				   "or a start location to be assigned.\n";
   ghost_name = "ghost";
   ghost_title = "Ghost";
   ghost_altnames = "spirit";
   ghost_brief = "The spirit of a fallen soul hovers here.\n";
   ghost_desc = "\nA thin glowing wisp of smoke make up this etheral form of some \n"
				"unfortunate adventurer.\n";

   default_speed = 0;
   default_strength = 8;
   default_dexterity = 8;
   default_constitution = 20;
   default_intel = 5;
   default_maxendur = 80;
   default_maxmagic = 80;
   default_indweight = 180;
   default_indsize = 180;
   default_wimpy = 30;
   default_wisdom = 8;
   default_charisma = 8;
   secs_to_rot = 100;
   endurance_divisor = 15;
   talent_funds = 16;

}


/***********************************************************************
 ** load_configs - loads the config file into memory
 **
 ** Parameters: ??
 **
 ** Returns: 
 **
 ***********************************************************************/

int Configs::load_configs()
{
   FILE         *conf_file = NULL;
   token_record *the_token;
   int          i = 0;
   char         user_input[101];

   char *poss_conf_locs[] = {
      ".." DSEP "data" DSEP "aime.conf",
      "." DSEP "aime.conf",
      "." DSEP "data" DSEP "aime.conf",
      DSEP "etc" DSEP "aime.conf",
      DSEP "etc" DSEP "aime" DSEP "aime.conf",
      NULL
   };

   /* Find the config file */
   if (the_config.conffile_loc.str_show() != NULL)
   {
      if ((conf_file = xfopen(the_config.conffile_loc.str_show(), "r", 
                                                   "config_file")) == NULL)
      {
         printf("Could not find aime.conf file at specified location: %s\n", 
                                       the_config.conffile_loc.str_show());
         exit(0);
      } 
   }
   else
   {
      while ((conf_file == NULL) && (poss_conf_locs[i] != NULL))
      {
         conf_file = xfopen(poss_conf_locs[i], "r", "config_file");
         i++;
      }
   }

   if (conf_file == NULL)
   {
      printf("Could not find aime.conf configuration file.\n");
      printf("Please supply the file location: ");
      fflush(stdout);

      fgets(user_input, 100, stdin);
      for (i=0; i<100; i++)
      {
         if (user_input[i] == '\n')
	 {
            user_input[i] = '\0';
            break;
	 }
      }
      if ((conf_file = xfopen(user_input, "r", "config_file")) == NULL)
      {
         printf("Could not find config file at location '%s'.\n", 
                                                              user_input);
         return -1;
      }
      filename = user_input;
   }
   else
   {
      filename = poss_conf_locs[i-1];
   }

   the_token = get_token(conf_file, '\0');
   while (the_token->token_type > 0)
   {
      switch(the_token->token_type)
      {
         case T_POUND:
            next_line(conf_file);
            break;

         case T_IDENTIFIER:
            if (read_config(the_token->the_string, conf_file) <= 0)
               return -1;
            break;

         default:
            printf("Invalid token '%s' found in config file.\n", 
                                         the_token->the_string);
            xfclose(conf_file, "config_file");
            return -1;
      }
      the_token = get_token(conf_file, '\0');
   }

   xfclose(conf_file, "config_file");
   return 1;
}


/***********************************************************************
 ** read_config - reads in a configuration entry in the config file
 **
 ** Parameters: the_ident - the config entry name to read in
 **             the_file - the file to read from
 **             
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Configs::read_config(char *the_ident, FILE *the_file)
{
   token_record *the_token;
   char         *temp_desc;

   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_EQUALS)
   {
      printf("Invalid format in config entry '%s'\n", the_ident);
      return -1;
   }
   
   if (!STRCASECMP(the_ident, "basedir"))
   {
      the_token = get_token(the_file, '\n');
      basedir = the_token->the_string;
      basedir.remove_head_spaces();
      basedir.remove_tail_spaces();
      basedir.remove_newline();
   }   
   else if (!STRCASECMP(the_ident, "mudname"))
   {
      the_token = get_token(the_file, '\n');
      mudname = the_token->the_string;
      mudname.remove_head_spaces();
      mudname.remove_tail_spaces();
   }   
   else if (!STRCASECMP(the_ident, "gameport"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      gameport = (unsigned short int) atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "buildport"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      buildport = (unsigned short int) atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "maxpasswdtries"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      maxpasswdtries = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "minpasswdlen"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      minpasswdlen = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "pollspersec"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      pollspersec = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "maxreadsperpoll"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      maxreadsperpoll = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "secsinday"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      secs_in_day = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "daysinyear"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      days_in_year = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "firstdaysummer"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      first_day_summer = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "firstdaywinter"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      first_day_winter = atoi(the_token->the_string);
   }   

   else if (!STRCASECMP(the_ident, "secsplayerheal"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      secs_player_heal = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "secsmobileheal"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      secs_mobile_heal = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "secsplayermagic"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      secs_player_magic = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "secsplayerendur"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      secs_player_endur = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "secslevelcheck"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      secs_level_check = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "weightstrmult"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      weight_str_mult = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "maxglobalexp"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      max_global_exp = atol(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "maxnumattacking"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      max_num_attacking = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "secstilltimeout"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      secs_till_timeout = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "maxlinelen"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      max_line_len = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultpagerlines"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_pager_lines = atoi(the_token->the_string);
   }   
 
   else if (!STRCASECMP(the_ident, "daemonize"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_DAEMONIZE);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_DAEMONIZE);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "timezone"))
   {
      the_token = get_token(the_file, '\n');
      timezone = the_token->the_string;
      timezone.remove_head_spaces();
      timezone.remove_tail_spaces();
   }   
   else if (!STRCASECMP(the_ident, "buildprompt"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_QUOTE)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      the_token = get_token(the_file, '\"');
      buildprompt = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "masteruser"))
   {
      the_token = get_token(the_file, '\0');
      masteruser = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "masterpasswd"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_QUOTE)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      the_token = get_token(the_file, '\"');
      masterpasswd = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "autowiz"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_AUTOWIZ);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_AUTOWIZ);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "distancetell"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_DISTANCETELL);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_DISTANCETELL);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "usequests"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_USEQUESTS);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_USEQUESTS);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "usemailboxes"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_USEMAILBOXES);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_USEMAILBOXES);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "logdeath"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOGDEATH);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOGDEATH);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "logflags"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOGFLAGS);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOGFLAGS);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "logforce"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOGFORCE);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOGFORCE);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "logclone"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOGCLONE);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOGCLONE);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "logbhosts"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOGBHOSTS);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOGBHOSTS);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "logbanhosts"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOGBANHOSTS);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOGBANHOSTS);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "logdelete"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOGDELETE);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOGDELETE);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "logslay"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOGSLAY);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOGSLAY);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "logboot"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOGBOOT);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOGBOOT);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "gameallowdefault"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_GAMEALLOW);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_GAMEALLOW);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "bldrallowdefault"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_BLDRALLOW);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_BLDRALLOW);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "lookuphosts"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_LOOKUPHOSTS);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_LOOKUPHOSTS);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }
   else if (!STRCASECMP(the_ident, "userslevelchain"))
   {
      the_token = get_token(the_file, '\0');
      users_level_chain = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "hardcodedareaname"))
   {
      the_token = get_token(the_file, '\0');
      hardcoded_areaname = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "backuplocname"))
   {
      the_token = get_token(the_file, '\0');
      backup_locname = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "backuploctitle"))
   {
      the_token = get_token(the_file, '\n');
      backup_loctitle = the_token->the_string;
      backup_loctitle.remove_head_spaces();
      backup_loctitle.remove_tail_spaces();
   }   
   else if (!STRCASECMP(the_ident, "backuplocdesc"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_CARROT)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      the_token = get_token(the_file, '^');
      backup_locdesc = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "entrylocname"))
   {
      the_token = get_token(the_file, '\0');
      entry_locname = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "ghostname"))
   {
      the_token = get_token(the_file, '\0');
      ghost_name = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "ghosttitle"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_CARROT)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      the_token = get_token(the_file, '^');
      ghost_title = the_token->the_string;
   }  
   else if (!STRCASECMP(the_ident, "ghostaltnames"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_CARROT)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      the_token = get_token(the_file, '^');
      ghost_altnames = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "ghostbrief"))
   {
      temp_desc = read_desc_type(the_file, NULL, NULL);
   
      if (temp_desc == NULL)
         return -1;
      ghost_brief = temp_desc;
      delete temp_desc;
   }
   else if (!STRCASECMP(the_ident, "ghostdesc"))
   {
      temp_desc = read_desc_type(the_file, NULL, NULL);
   
      if (temp_desc == NULL)
         return -1;
      ghost_desc = temp_desc;
      delete temp_desc;
   }
   else if (!STRCASECMP(the_ident, "entryloctitle"))
   {
      the_token = get_token(the_file, '\n');
      entry_loctitle = the_token->the_string;
      entry_loctitle.remove_head_spaces();
      entry_loctitle.remove_tail_spaces();
   }   
   else if (!STRCASECMP(the_ident, "entrylocdesc"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_CARROT)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      the_token = get_token(the_file, '^');
      entry_locdesc = the_token->the_string;
   }
   else if (!STRCASECMP(the_ident, "defaultspeed"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_speed = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultstrength"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_strength = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultdexterity"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_dexterity = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultconstitution"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_constitution = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultintel"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_intel = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultwisdom"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_wisdom = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultcharisma"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_charisma = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultmaxendur"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_maxendur = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultmaxmagic"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_maxmagic = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultindweight"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_indweight = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultindsize"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_indsize = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultwimpy"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_wimpy = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultwisdom"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_wisdom = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "defaultcharisma"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      default_charisma = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "secstorot"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      secs_to_rot = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "endurancedivisor"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      endurance_divisor = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "talentfunds"))
   {
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_NUMERICAL)
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      }

      talent_funds = atoi(the_token->the_string);
   }   
   else if (!STRCASECMP(the_ident, "nopromptconvert"))
   {
      the_token = get_token(the_file, '\0');
      if (!STRCASECMP(the_token->the_string, "yes"))
         conf_flags->set_flag(CF_NOPROMPTCONV);
      else if (!STRCASECMP(the_token->the_string, "no"))
         conf_flags->clr_flag(CF_NOPROMPTCONV);
      else
      {
         printf("Invalid format in config entry '%s'\n", the_ident);
         return -1;
      } 
   }

   else
   {
      printf("Unrecognized entry '%s' in config file.\n", the_ident);
      return -1;
   }

   return 1;
}

#endif


