/**********************************************************************
 ** Utils functions: A non-OO list of useful utility functions. I didn't
 **                  see a need for these to be an object since they didnt
 **                  really fit into any object I could think of
 **
 **
 **
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the
 **   License, or any later version.
 **
 **   This program is distributed in the hope that it will be useful, but
 **   WITHOUT ANY WARRANTY; without even the implied warranty of
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 **   General Public License for more details.
 **
 **   You should have received a copy of the GNU General Public License
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 **
 **********************************************************************/

#ifndef UTILS_C
#define UTILS_C

#ifndef CONVERTER
#include "config.h"
#include "sysdep.h"
#include "global.h"
#include "mudtypes.h"
#include "strings.h"
#include "lexer.h"
#include "errlog.h"
#include "mudobject.h"
#include "data_log.h"
#include "specials.h"
#include "code.h"
#include "inp_funct.h"
#include "newfuncts.h"
#include "comflags.h"
#include "adminflags.h"
#include "gameflags.h"
#include "utils.h"
#include "inp_handler.h"
#include "specials.h"
#include "md5.h"
#include "race.h"
#include "verb_list.h"
#else
#include "../../include/strings.h"
#include "../../include/sysdep.h"
#include "../../include/utils.h"
#include "convconfig.h"
#endif

#ifdef WIN32
#include "../win32/resource.h"
#include "../win32/Aime_w32Dlg.h"

extern CAime_w32Dlg *main_dlg;
#endif

char *ignore_words[] = {"a", "and", "or", "is", "it", "the", "of", "in",
                        NULL};

#ifndef CONVERTER  // Functions used by converter should go after the endif
/***********************************************************************
 ** get_time_str - gets the current time and date in string format
 **
 ** Parameters: None
 **
 ** Returns: A pointer to the time string
 **
 ***********************************************************************/

char *get_time_str(void)
{
   time_t cur_time;  /* holds the current time in long int format */
   char *time_str;   /* holds the time string */
   char *temp_ptr;   /* used to parse out the \n in the time string */

   cur_time = time(0);
   time_str = ctime(&cur_time);

   /* ignore the day of the week */
   while (*time_str != ' ')
      time_str++;
   time_str++;

   temp_ptr = time_str;
   while ((*temp_ptr) && (*temp_ptr != '\n'))
      temp_ptr++;
   *temp_ptr = '\0';

   return time_str;
}


/***********************************************************************
 ** get_date_str - gets the date based on the time passed in
 **
 ** Parameters: None
 **
 ** Returns: A pointer to the time string
 **
 ***********************************************************************/

void get_date_str(Strings *time_str, time_t the_time)
{
   char *temp_str;   /* holds the time string */
   char *temp_ptr;   /* used to parse out the \n in the time string */
   int  i;

   if (time_str == NULL)
      return;

   time_str->truncate(0);
   temp_str = ctime(&the_time);

   /* ignore the day of the week */
   while (*temp_str != ' ')
      temp_str++;

   temp_str++;
   temp_ptr = temp_str;

   for (i=0; i<2; i++)
   {
      while (*temp_ptr != ' ')
         temp_ptr++;
      *temp_ptr = '\0';
      temp_ptr++;

      time_str->str_cat(temp_str);
      time_str->str_cat(" ");

      while (*temp_ptr == ' ')
         temp_ptr++;
      temp_str = temp_ptr;
   }

   *temp_ptr = '\0';

   time_str->str_cat(temp_str);

   temp_str = temp_ptr + 1;
   while (*temp_str == ' ')
      temp_str++;

   while (*temp_str != ' ')
      temp_str++;

   temp_str++;
   time_str->str_cat(temp_str);
}


/***********************************************************************
 ** num_ats - returns the number of ats (@) in the string
 **
 ** Parameters: the_str - the string to check
 **
 ** Returns: The number of ampersands in the string
 **
 ***********************************************************************/

int num_ats(char *the_str)
{
   char *str_ptr;     /* used to move along the string */
   int count = 0;     /* number of ampersands */

   if (the_str == NULL)
      return -1;

   str_ptr = the_str;
   while (*str_ptr)
   {
      if (*str_ptr == '@')
         count++;
      str_ptr++;
   }
   return count;
}

/***********************************************************************
 ** make_full_area - makes this a full area string if it is not already
 **
 ** Parameters: the_string - the string to change
 **             the_area - the area to add to the string
 **
 ** Returns: 1 if success, -1 if failed
 **
 ***********************************************************************/

int make_full_area(Strings *the_string, char *the_area)
{
   Strings tmp_str;

   if (the_area == NULL)
      return -1;

   if (!the_string->num_char('@'))
   {
      tmp_str = the_string->str_show();
      the_string->sprintf("%s@%s", tmp_str.str_show(), the_area);
      return 1;
   }
   return 1;
}

/***********************************************************************
 ** encrypt_passwd - encrypts the string passed in
 **
 ** Parameters: plain_text - the plain text password
 **             encrypted_text - the encrypted password
 **
 ** Returns: 1 if success, -1 if failed
 **
 ***********************************************************************/

int encrypt_passwd(Strings *plain_text, Strings *encrypted_text)
{
   unsigned long aulDigest[ 4 ];
   struct MD5Context stMD5;

   MD5Init( &stMD5 );
   MD5Update( &stMD5, ( unsigned char * )plain_text->str_show( ),
              plain_text->str_len( ) );
   MD5Final( ( unsigned char * )aulDigest, &stMD5 );
   encrypted_text->sprintf( "%08lx%08lx%08lx%08lx",
                            aulDigest[ 0 ], aulDigest[ 1 ],
                            aulDigest[ 2 ], aulDigest[ 3 ] );

   return 1;
}


/***********************************************************************
 ** greater_than - if the first string is greater than the second string
 **
 ** Parameters: first_str - the string we are testing
 **             second_str - the string we are testing against
 **
 ** Returns: 1 if greater than, 0 if less than, -1 if equal
 **
 ***********************************************************************/

int greater_than(char *first_str, char *second_str)
{
   char *str1_ptr;
   char *str2_ptr;
   char str1_holder;
   char str2_holder;

   str1_ptr = first_str;
   str2_ptr = second_str;

   while (*str1_ptr)
   {
      if (!*str2_ptr)
         return 1;

      str1_holder = tolower(*str1_ptr);
      str2_holder = tolower(*str2_ptr);
      if (str1_holder > str2_holder)
         return 1;
      else if (str1_holder < str2_holder)
         return 0;
      else
      {
         str1_ptr++;
         str2_ptr++;
      }
   }
   return -1;
}



int test_malloc()
{
   char *teststr;
   int  i;

   for (i=0; i<1000; i++)
   {
      teststr = new_char(100);
      delete_char(teststr);
   }
   return 1;

}


/***********************************************************************
 ** data_log_on_pop - when a data_log editor input handler is popped
 **
 ** Parameters: player_name - the name of the player submitting
 **             data_log_name - the name of the data_log to add to
 **                current choices:  suggest, bug
 **             the_string - the string to add to the data log
 **
 **
 ** Returns:  1 if success
 **          -1 if error
 **
 ***********************************************************************/

int data_log_on_pop(Inp_Handler *the_handler, char *player_name,
          char *player_loc, char *data_log_name, Strings *the_string)
{
   Data_Log *the_log;

   if ((data_log_name == NULL) || (the_string == NULL))
      return -1;

   if (!STRCASECMP(data_log_name, "suggest"))
      the_log = mainstruct->get_suggestlog();
   else if (!STRCASECMP(data_log_name, "bug"))
      the_log = mainstruct->get_buglog();
   else
   {
      printf("blabla!\n");
      return -1;
   }

   the_log->log_data(player_name, player_loc, the_string->str_show());
   return 1;

   the_handler = NULL;
}


/***********************************************************************
 ** check_specials - checks list of specials to see if we run any
 **
 ** Parameters: trig_name - the trigger name to look for
 **             the_obj - the object the special is attached to
 **             the_primary - the obj that fills the primary slot passed in
 **             the_secondary - the obj filling the secondary slot passed in
 **             this_obj - the obj filling the 'this' slot passed in
 **             the_player - the player causing the special to be run
 **
 **
 ** Returns: -1 if failed,
 **           1 if good,
 **           2 if we should delete the object
 **           3 if conditions were not met to complete the special execution
 **           4 used with abilities in attempt trigger, if the attempt
 **             should fail
 **           5 used with abilities in attempt trigger, if the attempt
 **             should succeed
 **             Note: 4 and 5 override all output
 **
 ***********************************************************************/

int check_specials(char *trig_name, MudObject *the_obj,
                         MudObject *the_primary, MudObject *the_secondary,
                         MudObject *this_obj, Player *the_player)
{
   return check_specials(trig_name, the_obj, the_primary, the_secondary,
                                               this_obj, the_player, NULL);
}


/***********************************************************************
 ** check_specials - checks list of specials to see if we run any
 **
 ** Parameters: trig_name - the trigger name to look for
 **             the_obj - the object the special is attached to
 **             the_primary - the obj that fills the primary slot passed in
 **             the_secondary - the obj filling the secondary slot passed in
 **             this_obj - the obj filling the 'this' slot passed in
 **             the_player - the player causing the special to be run
 **             target_text - the player-specified target name in text form
 **
 **
 ** Returns: -1 if failed,
 **           1 if good,
 **           2 if we should delete the object
 **           3 if conditions were not met to complete the special execution
 **           4 used with abilities in attempt trigger, if the attempt
 **             should fail
 **           5 used with abilities in attempt trigger, if the attempt
 **             should succeed
 **             Note: 4 and 5 override all output
 **
 ***********************************************************************/

int check_specials(char *trig_name, MudObject *the_obj,
                         MudObject *the_primary, MudObject *the_secondary,
                         MudObject *this_obj, Player *the_player,
                                                    char *target_text)
{
   special_holder *the_special;
   in_params      fill_param;

   if (the_obj == NULL)
      return -1;

   if ((the_special = the_obj->find_special(trig_name)) != NULL)
   {
      int results;

      fill_param.primary_obj = the_primary;
      fill_param.secondary_obj = the_secondary;
      fill_param.this_obj = this_obj;

      the_special->environment.trig_used = trig_name;
      the_special->environment.target_str = target_text;
      if ((results = (the_special->the_special)->
        run_special(the_player, &fill_param, &the_special->environment)) == 2)
      {
         MudObject *container;

         container = the_obj->get_contained_by();
         if (container != NULL)
         {
            container->remove_contained(the_obj);
         }
         mainstruct->delete_object(the_obj);
      }
      else if (results == 6)
      {
         the_obj->remove_special(the_special->the_special);
      }
      return results;
   }
   return 0;
}


/***********************************************************************
 ** run_command - mimics a user's input and runs a command for the user
 **
 ** Parameters: the_user - the user who is running the command
 **             command_str - the string with the input to run
 **
 **
 ** Returns:  1 if success
 **          -1 if error
 **
 ***********************************************************************/

int run_command(MudObject *the_user, char *command_str)
{
   return main_plr_handler(the_user, command_str);
}


/***********************************************************************
 ** enter_new_player - sends a new player into the game
 **
 ** Parameters: player_name - the name of the player entering the game
 **
 ***********************************************************************/

int enter_new_player(Inp_Handler *the_handler, char *player_name,
               char *not_used1, char *not_used2, Strings *not_used3)
{
   Player   *the_player;
   Strings  holder;
   Flags    *tmp_comflags;
   Flags    *tmp_admflags;
   Flags    *tmp_gameflags;
   Location *loc_holder;
   Race     *plr_race;
   Verb_List *verbs;

   if ((the_player = mainstruct->get_player(player_name)) == NULL)
   {
      holder.sprintf("Tried to place player %s in game, was not on",
                                                          player_name);
      mainstruct->log_error(holder.str_show(), "enter_new_player");
      return -1;
   }

   //   the_player->reset_player( );

   tmp_comflags = the_player->get_comflags();
   tmp_gameflags = the_player->get_gameflags();
   tmp_admflags = the_player->get_admflags();

   // Make sure the new character isn't deleted
   tmp_gameflags->clr_flag( GAMEFLAG_DELETED );

   if ((the_config.conf_flags->get_flag(CF_AUTOWIZ)) ||
       ((!STRCASECMP(the_player->get_title(), the_config.masteruser.str_show())) &&
       (!strcmp(the_player->get_passwd(), the_config.masterpasswd.str_show()))))
   {

      tmp_comflags->set_all(comflagnames);
      tmp_admflags->set_all(admflagnames);
   }
   else
   {
      Object_List *the_dbase;
      Mask        *plr_mask;
      int         i;
      int         num_of[2];
      Flags       *mask_comflags;
      Flags       *mask_admflags;

      /* first we look for the players mask */
      the_dbase = mainstruct->get_dbase();

      if ((plr_mask = the_dbase->get_mask_obj("players")) == NULL)
      {
         mainstruct->log_error("Could not find players mask in mask file.\n",
                                       "enter_new_player");
         tmp_comflags->set_by_table(comflagnewbiemask);

      }
      else
      {
         mask_comflags = plr_mask->get_comflags();
         mask_admflags = plr_mask->get_admflags();

         i=0;
         while (comflagnames[i] != NULL)
            i++;
         num_of[0] = i;

         i=0;
         while (admflagnames[i] != NULL)
            i++;
         num_of[1] = i;

         for (i=0; i<num_of[0]; i++)
    {
            if (mask_comflags->get_flag(i))
               tmp_comflags->set_flag(i);
    }

         for (i=0; i<num_of[0]; i++)
    {
            if (mask_admflags->get_flag(i))
               tmp_admflags->set_flag(i);
    }

      }
   }

   /* if we can't get a start location, assign them to the default
      hard-coded location.  If that won't work, die */

   plr_race = mainstruct->get_race(the_player->get_race());

   if (plr_race == NULL)
   {
      the_player->set_location(mainstruct->get_def_start_loc());   
   }
   else
      the_player->set_location(plr_race->get_init_location());

   if (the_player->reset_curr_loc() == -1)
   {
      mainstruct->log_error("Default start location not a valid location.",
                            "enter_new_player");
      holder.sprintf("%s@%s", the_config.entry_locname.str_show(),
                                    the_config.hardcoded_areaname.str_show());
      mainstruct->set_def_start_loc(holder.str_show());

      the_player->set_location(mainstruct->get_def_start_loc());
      if (the_player->reset_curr_loc() == -1)
      {
         printf("Fatal Error!!! "
                "Could not get harcoded start location from database!!!\n");
         exit(0);
      }
   }

   if (tmp_gameflags->get_flag(GAMEFLAG_COLOR))
                     (the_player->get_connection())->set_color(1);

   the_player->set_pname(the_player->get_title());

   loc_holder = the_player->get_loc();

   //holder.sprintf("&+b%s&+W>&* ", the_player->get_title());
   holder.sprintf("&+c%%h/%%Hh %%m/%%Mm %%e/%%Ee&+W>&* "); 
   the_player->set_prompt(holder.str_show());

   mainstruct->save_player(the_player);

   holder.sprintf("Log &+Gon&* by new player: %s", the_player->get_title());
   mainstruct->log_event(holder.str_show());

   loc_holder = the_player->get_loc();
   holder.sprintf("&+Y[&+WNew Player &+M%s&+W has entered "
                  "the game at &+c%s&+Y]&*\n", the_player->get_title(),
                                    loc_holder->get_title());
   mainstruct->send_all_players(holder.str_show(), the_player,
                                                 ADMINFLAG_SEELOGIN);

   holder.sprintf("%s enters the game.\n", the_player->get_title());
   tmp_gameflags->set_flag(GAMEFLAG_ONGAME);
   loc_holder->send_location(holder.str_show(), the_player);

   the_player->send_plr("\n&+CWelcome new player &+M%s&+C!&*\n\n",
                                               the_player->get_name());

   //   the_player->add_initial_obj();

   loc_holder->show_location(the_player, 0);

   the_player->send_plr(
            "You have been given tutorial scrolls to help you.\n"
            "Type '&+cread intro&*' to start the tutorial\n\n");

   mainstruct->sort_playerlist("exp");

   verbs = mainstruct->get_player_verb_list();

   verbs->add_init_chatlines(the_player);

   /* get rid of compiler warnings */
   not_used1 = not_used2 = NULL;
   not_used3 = NULL;

   return 1;

   the_handler = NULL;
}


/***********************************************************************
 ** get_indicated_param - returns the appropriate parameter based on
 **                       what the user indicates, whether it be
 **                       this, primary, secondary, or actor
 **
 ** Parameters: player_name - the name of the player entering the game
 **
 ***********************************************************************/

MudObject *get_indicated_param(char *param_name, Specials *the_special,
                               char *function_name, Player *the_player,
                               in_params *player_params)
{
   MudObject *tmp_obj;
   Strings   holder;

   if (!STRCASECMP(param_name, "this"))
   {
      tmp_obj = player_params->this_obj;
   }
   else if (!STRCASECMP(param_name, "actor"))
   {
      tmp_obj = the_player;
   }
   else if (!STRCASECMP(param_name, "primary"))
   {
      tmp_obj = player_params->primary_obj;
   }
   else if (!STRCASECMP(param_name, "secondary"))
   {
      tmp_obj = player_params->secondary_obj;
   }
   else if (!STRCASECMP(param_name, "none"))
   {
      return NULL;
   }
   else
   {
      holder.sprintf(
       "%s is not a valid second parameter for function %s in special %s.  "
       "Must be either 'primary', 'secondary', 'this' or 'actor'\n",
        param_name, function_name, the_special->get_name());
      mainstruct->log_error(holder.str_show(), "get_indicated_param");
      return NULL;
   }

   if (tmp_obj == NULL)
   {
      holder.sprintf("%s is NULL for function %s in special %s.",
                     param_name, function_name, the_special->get_name());
      mainstruct->log_error(holder.str_show(), "get_indicated_param");
      return NULL;
   }

   return tmp_obj;
}


/***********************************************************************
 ** get_stat_bonus - gets the stats bonus based on the current table
 **
 ** Parameters: the_stat - the stat number to use the table with (Ex. str)
 **
 ***********************************************************************/

int get_stat_bonus(int the_stat)
{
   if (the_stat <= 20)
      return the_stat;
   if (the_stat <= 80)
      return ((((the_stat / 10) + 2) * 2)+20);
   return 20;
}


/***********************************************************************
 ** get_rand_word - gets a random word from a list of space-delineated word
 **
 ** Parameters: the_list - the string with the list of words
 **
 ** Returns: the random word selected
 **
 ***********************************************************************/

char *get_rand_word(char *the_list)
{
   int     i = 0;
   int     rand_results;
   Strings word_holder;

   word_holder.assign_word(the_list, i+1);
   while (word_holder.str_show() != NULL)
   {
      i++;
      word_holder.assign_word(the_list, i+1);
   }
   rand_results = 1+ (long) (((float) (i+1)) *(rand()/(RAND_MAX+1.0)));
   word_holder.assign_word(the_list, rand_results);
   return word_holder.str_show();
}


/***********************************************************************
 ** get_var - gets the variable in the variable list
 **
 ** Parameters: environment - the environment with the var stack
 **
 ** Returns: pointer to the var if found, NULL if not
 **
 ***********************************************************************/

vars *get_var(special_env *environment, char *var_name)
{
   vars *tmp_var;

   tmp_var = environment->exec_vars;
   while ((tmp_var != NULL) && STRCASECMP(tmp_var->varname.str_show(),
                                           var_name))
   {
      tmp_var = tmp_var->next_var;
   }

   return tmp_var;
}


/***********************************************************************
 ** find_var - finds a variable in the variable list and if not found,
 **            creates another one
 **
 ** Parameters: environment - the environment with the var stack
 **
 ** Returns: pointer to the var found
 **
 ***********************************************************************/

vars *find_var(special_env *environment, char *var_name)
{
   vars *tmp_var;
   vars **add_var;

   tmp_var = environment->exec_vars;
   add_var = &(environment->exec_vars);
   while ((tmp_var != NULL) && STRCASECMP(tmp_var->varname.str_show(),
                                           var_name))
   {
      add_var = &(tmp_var->next_var);
      tmp_var = tmp_var->next_var;
   }

   if (tmp_var != NULL)
      return tmp_var;

   tmp_var = new_vars();

   tmp_var->varname = var_name;
   tmp_var->vartype = 0;
   tmp_var->next_var = NULL;
   *add_var = tmp_var;

   return tmp_var;
}


/***********************************************************************
 ** delete_vars - delete the linked list of vars
 **
 ** Parameters: environment - the environment with the var stack
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int delete_var_list(special_env *environment)
{
   vars *tmp_var;

   while (environment->exec_vars != NULL)
   {
      tmp_var = environment->exec_vars->next_var;
      switch(environment->exec_vars->vartype)
      {
         case VAR_TYPE_STRING:
         case VAR_TYPE_OBJNAME:
         case VAR_TYPE_VAR:
            delete_Strings(environment->exec_vars->the_ptr.a_string);
            break;

         case VAR_TYPE_INT:
            delete_long(environment->exec_vars->the_ptr.an_int);
            break;

         default:
            break;
      }
      delete_vars(environment->exec_vars);
      environment->exec_vars = tmp_var;
   }
   return 1;
}



/***********************************************************************
 ** is_ignore_word - is this a word that is meaningless, that we should
 **                  ignore?
 **
 ** Parameters: the_str - the string we are checking against
 **
 ** Returns: 1 for it is an ignore word, 0 it is not, -1 error
 **
 ***********************************************************************/

int is_ignore_word(char *the_str)
{
   int i;


   if (the_str == NULL)
      return -1;

   i = 0;
   while (ignore_words[i] != NULL)
   {
      if (!STRCASECMP(ignore_words[i], the_str))
         return 1;
      i++;
   }
   return 0;
}


/***********************************************************************
 ** find_pos_in_list - takes a string and finds its identical string in
 **                    a list, returning the numerical position indicator.
 **           Warning: the list must contain a NULL as the last entry
 **
 ** Parameters: the_list - the strings list to search
 **             find_str -  the string to find
 **
 ** Returns: position if found, -1 if not found
 **
 ***********************************************************************/

int find_pos_in_list(char *the_list[], char *find_str)
{
   int i = 0;

   while (the_list[i] != NULL)
   {
      if (!STRCASECMP(the_list[i], find_str))
         return i;
      i++;
   }
   return -1;
}


/***********************************************************************
 ** str_sep: aime implimentation of strsep for platforms that don't
 **          include it
 **
 ** Parameters: acStr - pointer to the token string
 **             acDelim - the delim string to parse tokens on
 **
 ** Returns: a pointer to the next token or NULL
 **
 ***********************************************************************/

char *str_sep( char **acStr, char *acDelim )
{
   char *pcNext;
   char *pcRetVal;

   if(    acStr == NULL
       || *acStr == NULL
       || acDelim == NULL )
      return NULL;

   pcNext = strstr( *acStr, acDelim );
   if( pcNext == NULL )
   {
      pcRetVal = *acStr;
      *acStr = NULL;
   }
   else
   {
      pcRetVal = *acStr;
      *pcNext = '\0';
      *acStr = pcNext + 1;
   }

   return pcRetVal;
}


/***********************************************************************
 ** sysmessage - sends a string to the system screen
 **
 ** Parameters: the_str - the string to be sent
 **
 ** Returns: the num of chars sent if successful, -1 if failed
 **
 ***********************************************************************/

int sysmessage(char *new_str, ...)
{
   int xlen;       /* holds the total characters resulting from vsprinf */
   va_list args;   /* holds the list of arguments */
   char *tmp_str;  /* used to move along the string */
   char *arg_str;  /* used to display the argument if we want to, testing */
   int  arg_int;   /* used to hold an integer argument, for testing */
   int  arg_long;  /* used to hold a long argument, for testing */
   char arg_char;  /* used to hold a char argument, for testing */
   int  count = 0;
   char *dig_ptr;
   int  format = 0;


   /* we have to first estimate the size of a string we need to allocate
      to put this into. This is not an easy task, so we have to search
      along the va_list structs and count the size of each. So far it is
      only set up for strings, integers, and characters. Anything else and
      youre SOL */

   va_start( args, new_str );
   tmp_str = new_str;

   while (*tmp_str)
   {

      if ((*tmp_str == '%') && (*(tmp_str+1) && (*(tmp_str+1) != '%')))
      {
         format = 0;
         tmp_str++;
         if ((*tmp_str) && (*tmp_str == '-'))
    {
       /* advance beyond it */
            tmp_str++;
    }

         if (*tmp_str && (isdigit(*tmp_str)))
    {
            dig_ptr = tmp_str;
            while (*tmp_str && (isdigit(*tmp_str)))
               tmp_str++;
            format = atoi(dig_ptr);
            count += format;
         }

         if (*tmp_str && (*tmp_str == 's'))
    {
            arg_str = va_arg(args, char *);
            if (arg_str != NULL)
               count += (strlen(arg_str));
            else
          count += 6;
         }
         else if (*tmp_str && (*tmp_str == 'd'))
    {
            arg_int = va_arg(args, int);

            count += 8;
         }
         else if (*tmp_str && (*tmp_str == 'l'))
    {
            if (*(tmp_str+1) && (*(tmp_str+1) == 'd'))
               arg_long = va_arg(args, long);

            count += 14;
         }

         else if (*tmp_str && (*tmp_str == 'c'))
    {
            arg_char = (char) va_arg(args, int);
            count += 3;
         }
         else if (*tmp_str && !isalpha(*tmp_str))
    {
            arg_str = (char *) va_arg(args, void *);
            count += 100;
         }
         else
    {
            arg_str = (char *) va_arg(args, void *);
       count += 10;
         }
      }
      tmp_str++;
      count++;
   }
   /* increment two more just for safety */
   count += 2;

   /* now we reset the varg list and create the array, so we can load them
      all into the string */
   va_start( args, new_str );
   tmp_str = new_char(count);
   BZERO(tmp_str, count);
   xlen = VSNPRINTF( tmp_str, count-1, new_str, args );

   /* Here is where we do the machine-dependant sysmessage send */
#ifndef WIN32
   printf("%s", tmp_str);
#else
   main_dlg->sysmessage(tmp_str);
#endif

   /* now clean up after ourselves */
   va_end( args );

   delete_char(tmp_str);

   return 1;
}


/***********************************************************************
 ** prompt_user_convert - prompts the user to convert this file, if
 **                       applicable
 **
 ** Parameters: filename - the file name (without directory) of the file
 **                        that needs converting.  It should include the
 **                        extension
 **             directory - the directory the file exists in, without a
 **                         / at the end
 **
 ** Returns: 1 for successful, 0 for failed
 **
 ***********************************************************************/

int prompt_user_convert(char *filename, char *directory)
{
   char user_input[11];
   Strings holder;

   if (!the_config.conf_flags->get_flag(CF_NOPROMPTCONV))
   {
      sysmessage(_("The file '%s' is an old version and needs a convert.\n"
                   "Would you like to convert it? (Y/n) "), filename);

      fflush(stdout);

      fgets(user_input, 10, stdin);
      if (user_input[0] == 'n')
      {
         sysmessage(
              _("WARNING: File will not load properly without a convert.\n"));
         return 0;
      }
   }

   holder.sprintf("%s/%s/convert %s/%s %s/%s", the_config.basedir.str_show(),
		  BINDIR, directory, filename, directory, filename);
   
   return system(holder.str_show());
}


#endif // Functions for converter should go after this endif


/***********************************************************************
 ** read_desc_type - reads in a description type from a file
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **             the_obj - the object that we are getting from
 **
 ** Returns: a pointer to the created string
 **
 ***********************************************************************/

char *read_desc_type(FILE *read_file, ErrLog *error_log,
                                                 Entity *the_obj)
{
   token_record *the_token;
   token_record *temp_tok;
   int size = 0;
   char *temp_desc;
   Strings holder;
   int first = 1;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
#ifndef CONVERTER
      if (the_obj != NULL)
         holder.sprintf("Invalid format in mudobject %s",
                                                    the_obj->get_name());
      else
#endif
         holder = "Invalid format in multiple line text.\n";

#ifndef CONVERTER
      if (error_log != NULL)
         error_log->log_err(holder.str_show(), "read_desc_type");
#endif
      return NULL;
   }

   /* get the location description and get its size, how many characters */
   the_token = get_token(read_file, '^');
   temp_tok = the_token;
   while (temp_tok != NULL)
   {
      size = size + strlen(temp_tok->the_string) + 1;
      temp_tok = temp_tok->more;
   }


   /* create the description array given the size we need to store, efficient
      use of memory by dynamically allocating it based on size. We use this
      instead of the method provided by strings since this is more efficient
      to do one malloc instead of a malloc and destroy for each additional
      line in the desc. */
   temp_desc = new_char(size + 2);
   BZERO(temp_desc, size + 2);

   /* load the desc strings into the array */
   temp_tok = the_token;
   while (temp_tok != NULL)
   {
      /* remove the preceeding newline */
      if (!first)
      {
         strcat(temp_desc, temp_tok->the_string);
         strcat(temp_desc, "\n");
      }
      else
      {
         char *tmp_str;

         tmp_str = temp_tok->the_string;
         while ((*tmp_str) && (*tmp_str == ' '))
       tmp_str++;

         if ((*tmp_str) && (*tmp_str != '\n') &&
             (*tmp_str != '\r') && (*tmp_str != '\0'))
    {
            strcat(temp_desc, temp_tok->the_string);
            strcat(temp_desc, "\n");
         }
         first = 0;
      }
      temp_tok = temp_tok->more;
   }

   return temp_desc;
}



#endif // UTILS_C





