/**********************************************************************
 ** Code class: Controls a line of code, running that code and handling
 **             the parameters of that code
 **
 **   
 ** Reviewed through: version 0.14
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef CODE_C
#define CODE_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "mudobject.h"
#include "objtype.h"
#include "utils.h"
#include "specials.h"
#include "code.h"
#include "spec_func_table.h"
#include "lexer.h"
#include "newfuncts.h"
#include "global.h"

/***********************************************************************
 ** ~Code (destructor) - destroys the code
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Code::~Code()
{
   delete_param_list(pass_list);
   delete_param_list(param_list);
}


/***********************************************************************
 ** Code (constructor) - Creates the code 
 **
 ** Parameters: the_funct - the name of the function this one is
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Code::Code(char *the_funct)
{
   int i = 0;

   valid = 1;
   the_mudcode = NULL;
   param_list = NULL;
   pass_list = NULL;
   next_code = NULL;

   /* find the appropriate function */
   while (specfuncts[i].functname != NULL)
   {
      if (!STRCASECMP(the_funct, specfuncts[i].functname))
      {
         the_mudcode = specfuncts[i].specfunctptr;
         break;
      }
      i++;
   }

   if (the_mudcode == NULL)
   {
      valid = 0;
      return;
   }
}


/***********************************************************************
 ** Code (constructor) - Creates the code
 **
 ** Parameters: the_funct - the pointer to the function
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Code::Code(Spec_Func the_funct)
{
   valid = 1;
   the_mudcode = NULL;
   param_list = NULL;
   pass_list = NULL;
   next_code = NULL;

   
   the_mudcode = the_funct;

   if (the_mudcode == NULL)
   {
      valid = 0;
      return;
   }

}


/***********************************************************************
 ** check_valid - checks if the code had no errors
 **
 ** Parameters: None
 **
 ** Returns: 1 for valid, 0 for invalid
 **
 ***********************************************************************/

int Code::check_valid(void)
{ 
   return valid;
}


/***********************************************************************
 ** get_next_code - gets the next code in the list
 **
 ** Parameters: None
 **
 ** Returns: NULL for end of list, pointer to the next code if not
 **
 ***********************************************************************/

Code *Code::get_next_code(void)
{ 
   return next_code;
}


/***********************************************************************
 ** set_next_code - sets the next code in the list
 **
 ** Parameters: the_code - the code to set
 **
 ** Returns: -1 if failed, 1 if good
 **
 ***********************************************************************/

int Code::set_next_code(Code *the_code)
{ 
   next_code = the_code;
   return 1;
}


/***********************************************************************
 ** add_param - adds a parameter to the end of the list
 **
 ** Parameters: the_param - the parameter to add to the list
 **
 ** Returns: -1 if failed, 1 if good
 **
 ***********************************************************************/

int Code::add_param(params *the_param)
{
   params *tmp_param;

   if (the_param == NULL)
      return -1;
   
   the_param->next_param = NULL;
   if (param_list == NULL)
   {
      param_list = the_param;
      return 1;
   }   
 
   tmp_param = param_list;
   while (tmp_param->next_param != NULL)
   {
      tmp_param = tmp_param->next_param;
   }

   tmp_param->next_param = the_param;
   return 1;
}


/***********************************************************************
 ** add_pass_param - adds a parameter to the passed in parameter list
 **
 ** Parameters: the_param - the parameter to add to the list
 **
 ** Returns: -1 if failed, 1 if good
 **
 ***********************************************************************/

int Code::add_pass_param(params *the_param)
{
   params *tmp_param;

   if (the_param == NULL)
      return -1;
   
   the_param->next_param = NULL;
   if (pass_list == NULL)
   {
      pass_list = the_param;
      return 1;
   }   
 
   tmp_param = pass_list;
   while (tmp_param->next_param != NULL)
   {
      tmp_param = tmp_param->next_param;
   }

   tmp_param->next_param = the_param;
   return 1;
}


/***********************************************************************
 ** delete_param_list - deletes all the parameters from the list
 **
 ** Parameters: the_list - the list to delete
 **
 ** Returns: -1 if failed, 1 if good
 **
 ***********************************************************************/

int Code::delete_param_list(params *the_list)
{
   params *tmp_list;

   tmp_list = the_list;
   while (tmp_list != NULL)
   {
      switch(tmp_list->param_type)
      {
         case VAR_TYPE_STRING:
         case VAR_TYPE_OBJNAME:
         case VAR_TYPE_VAR:
            delete_Strings(tmp_list->the_param.a_string);
            break;

         case VAR_TYPE_INT:
            delete_long(tmp_list->the_param.an_int);
            break;

         case VAR_TYPE_CODE:
            delete_Code(tmp_list->the_param.a_code);
            break;

         default:
            printf("error, unrecognized param type in code, not freed.\n");
            break;
      }
      the_list = tmp_list->next_param;
      delete_params(tmp_list);
      tmp_list = the_list;
   }
   return 1;   
}


/***********************************************************************
 ** get_param - gets a parameter from the list based on the number it is
 **             down the list
 **
 ** Parameters: the_num - the number it is down the list
 **
 ** Returns: -1 if failed, 1 if good
 **
 ***********************************************************************/

params *Code::get_param(int the_num)
{
   int count = 1;
   params *tmp_list;

   tmp_list = param_list;

   while (tmp_list != NULL)
   {
      if (count == the_num)
      {
         return tmp_list;
      }
      tmp_list = tmp_list->next_param;
      count++;
   }
   return NULL;
}


/***********************************************************************
 ** execute_command - executes this command and returns the parameter
 **                   that we got from the command
 **
 ** Parameters: Nothing
 **
 ** Returns: the parameter that the command returns
 **
 ***********************************************************************/
   
params *Code::execute_command(Specials *the_special, Player *the_player, 
	in_params *player_params, int *the_results, special_env *environment)
{
   Code   *code_holder;
   params *cur_param;
   params *new_param;
   Strings holder;

   if (!valid)
   {
      return NULL;
   }

   delete_param_list(pass_list);
   pass_list = NULL;
   /* go through the parameter list executing them and getting what they
      return so we can replace the code parameters in the parameter list
      with the code's resulting parameters */
   cur_param = param_list;

   while (cur_param != NULL)
   {

      if (cur_param->param_type == VAR_TYPE_CODE)
      {
         params *new_param;

         code_holder = cur_param->the_param.a_code;
         new_param = code_holder->execute_command(the_special, the_player, 
                          player_params, the_results, environment);

         add_pass_param(new_param);

      }
      else if (cur_param->param_type == VAR_TYPE_VAR)
      {
         params *new_param;
         vars *tmp_param;

         code_holder = cur_param->the_param.a_code;
         tmp_param = 
             get_var(environment, cur_param->the_param.a_string->str_show());
         if ((tmp_param == NULL) || (tmp_param->vartype == 0))
	 {
            holder.sprintf("Variable '%s' used as parameter before assigned "
                "anything in special '%s'.\n", 
             cur_param->the_param.a_string->str_show(), the_special->get_name());
               
            mainstruct->log_error(holder.str_show(), "execute_command");
            *the_results = 0;
            return NULL;
	 }

         new_param = new_params();
         new_param->param_type = tmp_param->vartype;
         switch(tmp_param->vartype)
	 {
	    case VAR_TYPE_STRING:
            case VAR_TYPE_OBJNAME:
               new_param->the_param.a_string = 
                     new_Strings(tmp_param->the_ptr.a_string->str_show());
               break;

	    case VAR_TYPE_INT:
               new_param->the_param.an_int = new_long();
               *(new_param->the_param.an_int) = 
                                        *(tmp_param->the_ptr.an_int);
               break;

            default:
               break;
         }
    
         add_pass_param(new_param);

      }
      else
      {
         new_param = new_params();
         switch(cur_param->param_type)
	 {
	    case VAR_TYPE_STRING:
            case VAR_TYPE_OBJNAME:
	    case VAR_TYPE_VAR:
               new_param->the_param.a_string = 
                     new_Strings(cur_param->the_param.a_string->str_show());
               break;

	    case VAR_TYPE_INT:
               new_param->the_param.an_int = new_long();
               *(new_param->the_param.an_int) = 
                                        *(cur_param->the_param.an_int);
               break;

            default:
               break;
         }
         new_param->param_type = cur_param->param_type;
         add_pass_param(new_param);
      }
      cur_param = cur_param->next_param;
   }

   /* execute the code, returning the parameter */
   return the_mudcode(pass_list, the_special, the_player, player_params, 
                                                  environment, the_results);
}


/***********************************************************************
 ** load_from_file - loads the code from a file and puts it into
 **                  memory, specifically a Code object
 **
 ** Parameters: the_file - the file to load the specials from
 **             the_log - where to record the errors we encounter
 **             is_param - is this code a part of a parameter?
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Code::load_from_file(FILE *the_file, ErrLog *the_log)
{
   return load_from_file(the_file, the_log, FORMAT_FUNC);
}


/***********************************************************************
 ** load_from_file - loads the code from a file and puts it into
 **                  memory, specifically a Code object
 **
 ** Parameters: the_file - the file to load the specials from
 **             the_log - where to record the errors we encounter
 **             the_format - the format we should read this in
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/

int Code::load_from_file(FILE *the_file, ErrLog *the_log, int the_format)
{
   token_record  *the_token;
   Strings       holder;
   Strings       ident_name;
   Code          *code_holder;
   params        *param_holder = NULL;
   long          int_holder;
   char          *write_ptr;
   char          *read_ptr;


   if ((the_file == NULL) || (the_log == NULL))
      return -1;

   the_token = get_token(the_file, '\0');

   do
   {
      switch(the_token->token_type)
      {
         case T_QUOTE:
            the_token = get_token(the_file, '\"');

            if (strlen(the_token->the_string) == 0)
	    {
               the_log->log_err("Null string passed into code as parameter.",
                                            "load_from_file"); 
               return -1;
            }
            param_holder = new_params();
            param_holder->param_type = VAR_TYPE_STRING;
            param_holder->the_param.a_string = 
                           new_Strings(the_token->the_string);

            read_ptr = param_holder->the_param.a_string->str_show();
            write_ptr = read_ptr;

            while (*read_ptr)
	    {
               if ((*read_ptr == '\\') && 
                   ((*(read_ptr+1)) && (*(read_ptr + 1) == 'n')))
	       {
                  *write_ptr = '\n';
                  write_ptr++;
                  read_ptr += 2;
               }
               else
	       {
                  *write_ptr = *read_ptr;
                  write_ptr++;
                  read_ptr++;
               }
            }
            *write_ptr = '\0';
            the_token = get_token(the_file, '\0');
            break;

         case T_IDENTIFIER:
            ident_name = the_token->the_string;
            the_token = get_token(the_file, '\0');

            if (ident_name.num_char('@') > 0)
	    {
               param_holder = new_params();
               param_holder->param_type = VAR_TYPE_OBJNAME;
               param_holder->the_param.a_string = 
                                      new_Strings(ident_name.str_show());
            }
            else if (the_token->token_type == T_LPAREN)
	    {
               code_holder = new_Code(ident_name.str_show());

               if (!code_holder->check_valid())
               {
                  holder.sprintf("Unrecognized function '%s' as parameter", 
                                                    ident_name.str_show());
                  the_log->log_err(holder.str_show(), "load_from_file");
                  delete_Code(code_holder);
                  return -1;
               }

               if (code_holder->load_from_file(the_file, the_log) <= 0)
               {
                  holder.sprintf("Error in function '%s' as parameter", 
                                                     ident_name.str_show());
                  the_log->log_err(holder.str_show(), "load_from_file");
                  delete_Code(code_holder);
                  return -1;
               }
               param_holder = new_params();
               param_holder->param_type = VAR_TYPE_CODE;
               int_holder = atoi(the_token->the_string);
               param_holder->the_param.a_code = code_holder;
               the_token = get_token(the_file, '\0');

            }
            else if ((the_token->token_type == T_COMMA) ||
                     (the_token->token_type == T_RPAREN))
	    {
               param_holder = new_params();
               param_holder->param_type = VAR_TYPE_VAR;
               param_holder->the_param.a_string = new_Strings(
                                                       ident_name.str_show());    
	    }
            break;

	 case T_NUMERICAL:
            param_holder = new_params();
            param_holder->param_type = VAR_TYPE_INT;
            int_holder = atoi(the_token->the_string);
            param_holder->the_param.an_int = new_long();
            *(param_holder->the_param.an_int) = int_holder;
            the_token = get_token(the_file, '\0');
            break;

         case T_RPAREN:
            break;

	 default:
            holder.sprintf("Unexpected '%s' found", the_token->the_string); 
            the_log->log_err(holder.str_show(), "load_from_file (code)");
            return -1;
      }
      if (((the_token->token_type != T_COMMA) ||
           ((the_token->token_type == T_COMMA) &&
            (the_format == FORMAT_EQ))) &&
          (the_token->token_type != T_RPAREN) &&
          ((the_format != FORMAT_EQ) ||
           (the_token->token_type != T_SEMICOLON)))
      {
         holder.sprintf("Unexpected '%s' found", the_token->the_string); 
         the_log->log_err(holder.str_show(), "load_from_file (code)");
         delete_params(param_holder);
         return -1;
      }
      if (the_token->token_type == T_COMMA)
         the_token = get_token(the_file, '\0');

      add_param(param_holder);

   } while (((the_format != FORMAT_EQ) && (the_token->token_type != T_RPAREN)) ||
        ((the_format == FORMAT_EQ) && (the_token->token_type != T_SEMICOLON)));

   return 1;

}


/***********************************************************************
 ** get_next_ptr - returns the memory address of the next pointer
 **
 ** Parameters: None
 **
 ** Returns: the pointer address
 **
 ***********************************************************************/

Code **Code::get_next_ptr(void)
{ 
   return &next_code;
}


/***********************************************************************
 ** get_funct_ptr - returns a pointer to the function that this code uses
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the function, or NULL for failed
 **
 ***********************************************************************/

Spec_Func Code::get_funct_ptr(void)
{ 
   return the_mudcode;
}


/***********************************************************************
 ** copy_code - copies the info from the copy_from object to here
 **
 ** Parameters: copy_from - the code we copy this from
 **
 ** Returns: 1 for success, -1 for failed
 **
 ***********************************************************************/

int Code::copy_code(Code *copy_from)
{
   param_list = copy_from->copy_param_list();
   return 1;
}


/***********************************************************************
 ** copy_param_list - copies the param list and passes the new list out
 **
 ** Parameters: None
 **
 ** Returns: pointer to the newly created list if success
 **
 ***********************************************************************/

params *Code::copy_param_list()
{
   params *tmp_list;
   params *new_list = NULL;
   params *last_added = NULL;
   params *new_param;

   tmp_list = param_list;
   while (tmp_list != NULL)
   {
      new_param = new_params();
      new_param->next_param = NULL;
      new_param->param_type = tmp_list->param_type;
      switch(tmp_list->param_type)
      {
         case VAR_TYPE_STRING:
         case VAR_TYPE_OBJNAME:
         case VAR_TYPE_VAR:
            new_param->the_param.a_string = 
                     new_Strings(tmp_list->the_param.a_string->str_show());
            break;
         case VAR_TYPE_INT:
            new_param->the_param.an_int = new_long();
            *(new_param->the_param.an_int) = *(tmp_list->the_param.an_int);
            break;

         case VAR_TYPE_CODE:
            new_param->the_param.a_code = 
                new_Code(tmp_list->the_param.a_code->get_funct_ptr());
            new_param->the_param.a_code->copy_code(tmp_list->the_param.a_code);
            break;

         case VAR_TYPE_GOTO:
            new_param->the_param.goto_code = tmp_list->the_param.goto_code;
            break;

         default: return NULL;
      }
      if (last_added == NULL)
      {
         new_list = new_param;
         last_added = new_param;
      }
      else
      {
         last_added->next_param = new_param;
         last_added = new_param;
      }

      tmp_list = tmp_list->next_param;
   }
   return new_list;
}



/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Code::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Code::get_mem_size_dynamic()
{
   int  size = 0;
   params *tmp_param;

   tmp_param = param_list;
   while (tmp_param != NULL)
   {
      size += sizeof(tmp_param);
      tmp_param = tmp_param->next_param;
   }

   tmp_param = pass_list;
   while (tmp_param != NULL)
   {
      size += sizeof(tmp_param);
      tmp_param = tmp_param->next_param;
   }
   return size;   
}


/***********************************************************************
 ** set_var_name - the variable name in an assignment.  Not used during
 **                normal functions
 **
 ** Parameters: new_name - the new name to assign to it
 **
 ***********************************************************************/

void Code::set_var_name(char *new_name)
{ 
   var_name = new_name;
}


/***********************************************************************
 ** get_var_name - gets the variable name in an assignment.  Not used during
 **                normal functions
 **
 ***********************************************************************/

char *Code::get_var_name()
{ 
   return var_name.str_show();
}
#endif


