/**********************************************************************
 ** Mudobject class: These functions manipulate all objects in the mud.
 **                  It is the highest superclass in the object class
 **                  heirarchy
 **
 **
 **
 **    Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef MUDOBJECT_C
#define MUDOBJECT_C

#include <signal.h>
#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "mudobject.h"
#include "objtype.h"
#include "specials.h"
#include "builder.h"
#include "moveable.h"
#include "mobile.h"
#include "book.h"
#include "marker.h"
#include "location.h"
#include "door.h"
#include "key.h"
#include "merger.h"
#include "money.h"
#include "editor.h"
#include "global.h"
#include "weapon.h"
#include "individual.h"
#include "wearable.h"
#include "food.h"
#include "utils.h"
#include "object_list.h"
#include "errlog.h"
#include "newfuncts.h"
#include "code.h"
#include "rope.h"
#include "boat.h"
#include "indflags.h"
#include "gameflags.h"

/***********************************************************************
 ** MudObject (constructor) - creates the mudobject
 **
 ** Parameters: the_name - the name of the object
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

MudObject::MudObject()
{
   special_list = NULL;
   next_cloned = 0;
   contained_by = NULL;
   capacity = 0;
}

/***********************************************************************
 ** ~MudObject (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

MudObject::~MudObject()
{
   special_holder *tmp_special;

   /* delete the special holders */
   tmp_special = special_list;
   while (tmp_special != NULL)
   {
      special_list = tmp_special->next_special;
      delete_special_holder(tmp_special);
      tmp_special = special_list;
   }

   /* a little bit of code to prevent deleting wielded/worn objects, need
      to remove this after the mud seems more stable */
   if ((get_type() != OBJ_TYPE_PLAYER) && (mainstruct != NULL) &&
       ((get_type() == OBJ_TYPE_WEARABLE) || (get_type() == OBJ_TYPE_WEAPON)))
   {
      Player *the_player;

      the_player = mainstruct->get_first_player();
      while (the_player != NULL)
      {
         if ((((MudObject *) the_player->get_wielded(Left)) == this) ||
             (((MudObject *) the_player->get_wielded(Right)) == this))
	    RAISE(11);

         if (get_type() == OBJ_TYPE_WEARABLE)
	 {
            if (the_player->is_worn((Wearable *) this))
               RAISE(11);
         }

         the_player = the_player->get_next_player();
      }
   }
}


/***********************************************************************
 ** find_contained - finds a mudobject the mudobject contains
 **
 ** Parameters: the_object - the name of the object to find
 **             find_num - the number passed in of the amount to valid
 **                        names to pass up before picking the right one
 **
 ** Returns:  the object if found
 **           NULL if not
 **
 ***********************************************************************/

MudObject *MudObject::find_contained(char *the_object, int *find_num)
{
   char *str_ptr;
   char *num_ptr;
   MudObject *result_obj;

   if (*find_num == 0)
   {
      /* first break out the number from the end, if there is one */
      if (the_object == NULL)
         return NULL;

      str_ptr = num_ptr = the_object;
      while ((*str_ptr) && (*str_ptr != '\0'))
      {
         str_ptr++;      
      }
      num_ptr = str_ptr - 1;
      while ((*num_ptr) && (isdigit(*num_ptr)))
         num_ptr--;

      if ((!*num_ptr) || (num_ptr == str_ptr))
         *find_num = 1;
      else
      {
         *find_num = atoi(num_ptr+1);
         *(num_ptr+1) = '\0';
      }
   }

   if ((result_obj = inventory.find(the_object, find_num)) == NULL)
   {
      result_obj = inventory.find_altname(the_object);
   }
   return result_obj;
}


/***********************************************************************
 ** find_contained - finds if a mudobject is contained
 **
 ** Parameters: the_object - the name of the object to find
 **             find_num - the number passed in of the amount to valid
 **                        names to pass up before picking the right one
 **
 ** Returns:  the object if found
 **           NULL if not
 **
 ***********************************************************************/

int MudObject::find_contained(MudObject *the_obj)
{
   MudObject *tmp_obj;
   
   inventory.reset_current();
   tmp_obj = inventory.get_next();  

   while ((tmp_obj != NULL) && (tmp_obj != the_obj))
      tmp_obj = inventory.get_next();

   if (tmp_obj != NULL)
      return 1;

   return 0;
}


/***********************************************************************
 ** remove_contained - finds a mudobject the mudobject contains and removes
 **                    it
 **
 ** Parameters: the_object - the name of the object to find
 **
 ** Returns:  1 if success, 0 if not
 **
 ***********************************************************************/

int MudObject::remove_contained(MudObject *the_object)
{

   if ((get_type() == OBJ_TYPE_PLAYER) || (get_type() == OBJ_TYPE_MOBILE))
   {
      Individual *the_ind;

      the_ind = (Individual *) this;

      if (the_object == the_ind->get_wielded(Left))
         the_ind->unwield_moveable(Left);   
      else if (the_object == the_ind->get_wielded(Right))
         the_ind->unwield_moveable(Right);

   }

   if ((the_object->get_type() == OBJ_TYPE_PLAYER) || 
       (the_object->get_type() == OBJ_TYPE_MOBILE))
      ((Individual *)the_object)->clr_curr_loc();


   the_object->set_contained_by(NULL);

   return inventory.del_entry(the_object);
}

/***********************************************************************
 ** adds_contained - adds a mudobject to the inventory list
 **
 ** Parameters: the_object - the object to add
 **
 ** Returns:  1 if success, 0 if not
 **
 ***********************************************************************/

int MudObject::add_contained(MudObject *the_object)
{
   Merger *old_merger;
   Merger *add_merger;
   int    old_value = 0;

   if (!STRNCASECMP(the_object->get_name(), "bridge", 6))
      raise(11);

   if (the_object->is_merger())
   {
      add_merger = (Merger *) the_object;
      if ((old_merger = inventory.
                        get_same_merger(add_merger->get_parent())) == NULL)
      {
         the_object->set_contained_by(this);
         return inventory.add_entry(the_object);
      }

      old_value = old_merger->get_number_of();
      old_merger->set_number_of(old_value + add_merger->get_number_of());
      return 2;
   }

   the_object->set_contained_by(this);
   return inventory.add_entry(the_object);
}


/***********************************************************************
 ** get_inventory - gets an inventory from a mudobject.
 **
 ** Parameters: the_player - the player where to send the text to.
 **
 ** Returns:  1 if success, 0 if not
 **
 ***********************************************************************/
LinkedList *MudObject::get_inventory()
{   return &inventory;
}

/***********************************************************************
 ** get_special_list - returns the pointer to the main specials list
 **
 **
 ***********************************************************************/
special_holder *MudObject::get_special_list()
{   return special_list;
}

/***********************************************************************
 ** get_special_list - returns the pointer to the main specials list
 **
 **
 ***********************************************************************/

void MudObject::set_special_list(special_holder *the_holder)
{
   special_list = the_holder;
}

/***********************************************************************
 ** add_special_holder - adds a special holder to the list of specials
 **
 **
 ***********************************************************************/

int MudObject::add_special_holder(special_holder *new_holder)
{
   special_holder *tmp_holder;

   tmp_holder = special_list;
   special_list = new_holder;
   special_list->next_special = tmp_holder;
   return 1;
}

/***********************************************************************
 ** set_location - sets the location for an object
 **
 ** Parameters: new_loc - the new location for the object
 **             prev_loc - the old location, so we can remove it
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int MudObject::set_location(MudObject *new_loc, MudObject *prev_loc)
{
   if ((prev_loc != NULL) && (!prev_loc->remove_contained(this)))
   {  
      printf("Obj %s has previous loc, but was not contained in that loc\n", 
                                                                 get_name());
      return -1;
   }

   if (new_loc != NULL)
   {
      new_loc->add_contained(this);
   }

   if (new_loc != NULL)
      location.sprintf("%s@%s", new_loc->get_name(), new_loc->get_area());  
   else
      location = "none";

   if ((get_type() == OBJ_TYPE_PLAYER) || 
       (get_type() == OBJ_TYPE_MOBILE))
   {
      Individual *the_ind;

      if (new_loc->get_type() != OBJ_TYPE_LOCATION)
      {
         if (new_loc->get_type() == OBJ_TYPE_BOAT)
            new_loc = new_loc->get_contained_by();
         
         if (new_loc->get_type() != OBJ_TYPE_LOCATION)
	 {
            printf("attempt to assign individual to non-location\n");
            return -1;
	 }
      }

      the_ind = (Individual *) this;
      the_ind->set_curr_loc((Location *) new_loc);
   }

   return 1;   
}


/***********************************************************************
 ** set_dual_loc - sets the object to a location, but also sets it as
 **                contained by another object as well.  It is important
 **                for you to remember the second contained loc so you
 **                can uncontain later.  Mainly for use in the rope code,
 **                to place the rope in two locations
 **
 ** Parameters: loc1 - the first location to set to
 **             loc2 - the second location to set to
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int MudObject::set_dual_loc(char *loc1_name, char *loc2_name, 
                                                  Object_List *the_dbase)
{
   MudObject *tmp_obj;
   LinkedList *the_inv;
   MudObject *loc1;
   MudObject *loc2;

   if ((loc1_name == NULL) || (loc2_name == NULL))
      return -1;

   if ((loc1 = the_dbase->get_mudobject(get_area(), loc1_name)) == NULL)
   {
      return -1;
   }

   if ((loc2 = the_dbase->get_mudobject(get_area(), loc2_name)) == NULL)
   {
      return -1;
   }

   the_inv = loc1->get_inventory();
   the_inv->reset_current();
   tmp_obj = the_inv->get_next();
   while ((tmp_obj != NULL) && (tmp_obj != this))
   {
      tmp_obj = the_inv->get_next();
   }

   /* if it is null, it is not already in location 1 */
   if (tmp_obj == NULL)
   {
      loc1->add_contained(this);
      location.sprintf("%s@%s", loc1->get_name(), loc1->get_area());
   }
   
   /* add to location 2, but don't modify the location string since we
      already did with location 1 */
   the_inv = loc2->get_inventory();
   the_inv->reset_current();
   tmp_obj = the_inv->get_next();
   while ((tmp_obj != NULL) && (tmp_obj != this))
   {
      tmp_obj = the_inv->get_next();
   }

   /* if it is null, it is not already in location 1 */
   if (tmp_obj == NULL)
   {
      loc2->add_contained(this);
   }
   
   return 1;   
}

/***********************************************************************
 ** set_location - sets the location for an object. You should only use this
 **                one in the builder port
 **
 ** Parameters: the_loc - the location to set it to
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int MudObject::set_location(char *the_loc)
{
   location.assign_word(the_loc, 1);
   return 1;
}


/***********************************************************************
 ** get_location - gets the location of the object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the location name string
 **
 ***********************************************************************/

char *MudObject::get_location()
{
   return location.str_show();
}

/***********************************************************************
 ** assign_keywords - assigns the keywords from the string passed in
 **
 ** Parameters: the_string - the string to set from
 **
 ** Returns: num words read for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::assign_keywords(char *the_string)
{
   int     i;
   Strings tmp_word;

   keywords.truncate(0);
   i=1;
   tmp_word.assign_word(the_string, i);
   while (tmp_word.str_show() != NULL)
   {
      if (!is_ignore_word(tmp_word.str_show()))
      {
         keywords.str_cat(tmp_word.str_show());
         keywords.str_cat(" ");
      }
      i++;
      tmp_word.assign_word(title.str_show(), i);
   }
   return i;
}

/***********************************************************************
 ** set_title - sets the title of the mudobject
 **
 ** Parameters: the_title - the title string to set it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_title(char *the_title)
{
   if (the_title == NULL)
      return -1;

   title = the_title;

   if (title.str_len() > MAXTITLELEN)
      title.truncate(MAXTITLELEN);

   assign_keywords(title.str_show());

   return 1;
}


/***********************************************************************
 ** get_title - gets the location title
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the title string
 **
 ***********************************************************************/

char *MudObject::get_title()
{
   return title.str_show();
}


/***********************************************************************
 ** set_desc - sets the description of the object to the_desc
 **
 ** Parameters: the_desc - the description to set it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_desc(char *the_desc)
{
   if (the_desc == NULL)
      return -1;

   return desc.str_copy(the_desc);
}


/***********************************************************************
 ** get_desc - gets the location description
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the description string
 **
 ***********************************************************************/

char *MudObject::get_desc()
{
   return desc.str_show();
}


/***********************************************************************
 ** find_altname - see if a word is in the list of altnames
 **
 ** Parameters: the_word - the word to check for
 **
 ** Returns: 1 if found, 0 if not found
 **
 ***********************************************************************/

int MudObject::find_altname(char *the_word)
{
   return altnames.find_in_str(the_word);
}

/***********************************************************************
 ** set_altname - sets the altnames of the mudobject
 **
 ** Parameters: the_names - the altnames to set it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_altname(char *the_names)
{
   if (the_names == NULL)
      return -1;

   altnames = the_names;

   return 1;
}


/***********************************************************************
 ** get_altname - get the altnames of the mudobject
 **
 ** Parameters: None
 **
 ** Returns: Pointer to the altname string
 **
 ***********************************************************************/

char *MudObject::get_altname(void)
{
   return altnames.str_show();
}

/***********************************************************************
 ** add_special - adds the special to the special list
 **
 ** Parameters: the_type - the type of the object
 **
 ** Returns: 1 if it is valid, -1 if not
 **
 ***********************************************************************/

int MudObject::add_special(Specials *the_special)
{
   special_holder *tmp_spec;
   special_holder *new_holder;

   new_holder = new_special_holder();
   new_holder->the_special = the_special;
   new_holder->environment.counter = the_special->get_counter();
   new_holder->environment.exec_vars = NULL;
   new_holder->next_special = NULL;
   new_holder->environment.stack_pointer = NULL;

   if (special_list == NULL)
   {
      special_list = new_holder;
      return 1;
   }

   tmp_spec = special_list;

   while (tmp_spec->next_special != NULL)
   {
      tmp_spec = tmp_spec->next_special;
   }
   
   tmp_spec->next_special = new_holder;
   return 1;
}


/***********************************************************************
 ** del_special - deletes a special from the special list
 **
 ** Parameters: the_type - the type of the object
 **
 ** Returns: 1 if it is valid, -1 if not
 **
 ***********************************************************************/

int MudObject::del_special(int the_num)
{
   special_holder *tmp_spec;
   special_holder *prev_spec = NULL;
   int      count = 1;

   if ((special_list == NULL) || (count <= 0))
      return -1;

   tmp_spec = special_list;

   while (count != the_num)
   {
      if (tmp_spec == NULL)
         return -1;

      prev_spec = tmp_spec;
      count++;
      tmp_spec = tmp_spec->next_special;
   }
   
   if (tmp_spec == NULL)
      return -1;

   if (prev_spec == NULL)
      special_list = tmp_spec->next_special;
   else
      prev_spec->next_special = tmp_spec->next_special;


   delete_special_holder(tmp_spec);

   return 1;
}


/***********************************************************************
 ** remove_special - removes a special from the special list
 **
 ** Parameters: the_type - the type of the object
 **
 ** Returns: 1 if it is valid, -1 if not
 **
 ***********************************************************************/

int MudObject::remove_special(Specials *the_special)
{
   special_holder *tmp_spec;
   special_holder *prev_spec = NULL;

   if (special_list == NULL)
      return -1;

   tmp_spec = special_list;

   while ((tmp_spec != NULL) && (tmp_spec->the_special != the_special))
   {
      if (tmp_spec == NULL)
         return -1;

      prev_spec = tmp_spec;
      tmp_spec = tmp_spec->next_special;
   }
   
   if (tmp_spec == NULL)
      return -1;

   if (prev_spec == NULL)
      special_list = tmp_spec->next_special;
   else
      prev_spec->next_special = tmp_spec->next_special;

   delete_special_holder(tmp_spec);

   return 1;
}


/***********************************************************************
 ** find_special - finds a special given a certain trigger
 **
 ** Parameters: the_text - the text that is the trigger
 **
 ** Returns: pointer to the special if found, NULL if not found
 **
 ***********************************************************************/

special_holder *MudObject::find_special(char *the_text)
{
   special_holder *tmp_special;
   Strings        holder; 
   int            counter;

   if (special_list == NULL)
      return NULL;

   tmp_special = special_list;

   while (tmp_special != NULL)
   {
      counter = 1;
      holder.assign_word((tmp_special->the_special)->get_trigger(), 
                                                                   counter);
      while (holder.str_show() != NULL)
      {
         if (!STRCASECMP((tmp_special->the_special)->get_trigger(), 
                                                                 the_text))
            return tmp_special;
         counter++;
         holder.assign_word((tmp_special->the_special)->get_trigger(), 
                                                                   counter);

      }

      tmp_special = tmp_special->next_special;
   }
   return NULL;
}

/***********************************************************************
 ** copy_specials - copies a linked list of specials and passes it out
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the specials list if it worked, NULL if failed
 **
 ***********************************************************************/

special_holder *MudObject::copy_specials()
{
   special_holder *new_special;
   special_holder *new_list = NULL;
   special_holder *old_list;
   special_holder *last_added = NULL;

   old_list = special_list;
   while (old_list != NULL)
   { 

      new_special = new_special_holder();
      new_special->the_special = old_list->the_special;
      new_special->next_special = NULL;
      new_special->environment.exec_vars = NULL;
      new_special->environment.stack_pointer = NULL;
      new_special->environment.counter = 
                                new_special->the_special->get_counter();
      new_special->environment.store_int = 0;

      if (last_added == NULL)
      {
         new_list = new_special;
         last_added = new_special;
      }
      else
      {
         last_added->next_special = new_special;
         last_added = new_special;
      }

      old_list = old_list->next_special;
   }
   return new_list;
}

/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

MudObject *MudObject::operator = (MudObject *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** copy_object - copies attributes of another object to this one
 **
 ** Parameters: copy_obj - the object we won't copy cause it is an error
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::copy_object(Entity *copy_obj)
{
   copy_obj = NULL;
   printf("Error!  Attempted to copy a mudobject object\n");
   return -1;
}


/***********************************************************************
 ** describe_special - describes the special indicated by specialnr
 **
 ** Parameters: the_builder - the builder to send stuff to
 **             specialnr - which special in order of the list we want
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::describe_special(Builder *the_builder, int specialnr)
{
   special_holder *tmp_special;
   int            count = 1;

   tmp_special = special_list;
   while ((tmp_special != NULL) && (count < specialnr))
   {
      tmp_special = tmp_special->next_special;
      count++;
   }

   if ((count != specialnr) || (tmp_special == NULL))
   {
      the_builder->send_bldr("That special does not exist.\n");
      return -1;
   }
   
   (tmp_special->the_special)->describe(the_builder);
   return 1;
}

/***********************************************************************
 ** set_clones - sets the list of clones to a string
 **
 ** Parameters: the_list - the list to set the mergers to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_clones(char *the_list)
{
   clones = the_list;
   return 1;
}

/***********************************************************************
 ** set_special_str - sets the list of specials to a string
 **
 ** Parameters: the_list - the list to set the specials to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_special_str(char *the_list)
{
   specials = the_list;
   return 1;
}

/***********************************************************************
 ** set_attrib_title - for set attribute command, sets the title attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_attrib_title(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify a title to set to.\n");
      return -1;
   }
   set_title(the_parsed->get_speech());
   the_builder->send_bldr("Title on %s set to: %s\n", get_name(), 
                                                        get_title());
   set_modified(1);
   return 1;
}


/***********************************************************************
 ** set_attrib_altnames - for set attribute command, sets the altname attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_attrib_altnames(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify an altname to set to.\n");
      return -1;
   }
   set_altname(the_parsed->get_speech());
   the_builder->send_bldr("Altnames on %s set to: %s\n", get_name(), 
                                                        get_altname());
   set_modified(1);
   return 1;
}


/***********************************************************************
 ** set_attrib_desc - for set attrib command, sets the desc attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_attrib_desc(Builder *the_builder)
{
   if (the_builder->get_long_input(&desc) < 0)
   {
      the_builder->send_bldr("Error reading in input, failed!\n");
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** set_attrib_loc - for set attrib command, sets the loc attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_attrib_loc(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify a location to set to.\n");
      return -1;
   }
   set_location(the_parsed->get_speech());
   the_builder->send_bldr("Location on %s set to: %s\n", get_name(), 
                                                        get_location());
   set_modified(1);
   return 1;
}


/***********************************************************************
 ** set_attrib_clones - for set_attrib, sets the clones attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_attrib_clones(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify clones to set to.\n");
      return -1;
   }
   set_clones(the_parsed->get_speech());
   the_builder->send_bldr("Clones on %s set to: %s\n", get_name(), 
                                                        get_clones());
   set_modified(1);
   return 1;
}

/***********************************************************************
 ** get_clones - gets the clones string
 **
 **
 ***********************************************************************/

char *MudObject::get_clones()
{
   return clones.str_show();
}


/***********************************************************************
 ** get_special_str - gets the specials string
 **
 **
 ***********************************************************************/

char *MudObject::get_special_str()
{
   return specials.str_show();
}

/***********************************************************************
 ** increment_cloned - increments the number cloned by one
 **
 **
 ***********************************************************************/

void MudObject::increment_cloned() 
{ 
   next_cloned++;
}


/***********************************************************************
 ** get_next_cloned - gets the next number to use for cloning
 **
 **
 ***********************************************************************/

unsigned long MudObject::get_next_cloned() 
{ 
   return next_cloned; 
}


/***********************************************************************
 ** get_contained_by - returns a pointer to the object that contains
 **                    this object
 **
 **
 ***********************************************************************/

MudObject *MudObject::get_contained_by() 
{ 
   return contained_by;
}


/***********************************************************************
 ** set_contained_by - returns a pointer to the object that contains
 **                    this object
 **
 **
 ***********************************************************************/

void MudObject::set_contained_by(MudObject *the_obj) 
{ 
  /*   if ((get_type() == OBJ_TYPE_DOOR) && (the_obj != NULL))
       printf("setting to: %s\n", the_obj->get_name());
   else if (get_type() == OBJ_TYPE_DOOR)
   RAISE(1); */

   contained_by = the_obj;
}


/***********************************************************************
 ** write_mudobject_attrib - writes all mudobject attributes to the file
 **
 ** Parameters: the_file - the file that we are writing to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::write_mudobject_attrib(FILE *the_file)
{
   fprintf(the_file, "^%s^\n", get_title());
   fprintf(the_file, "^%s^\n", (get_altname() == NULL) ? "" : get_altname());
   fprintf(the_file, "^%s^\n", (get_clones() == NULL) ? "" : get_clones());
   fprintf(the_file, "%s\n", (get_location() == NULL) ? "needloc" : 
                                                            get_location());
   fprintf(the_file, "^%s^\n", (get_special_str() == NULL) ? "" : 
                                                     get_special_str());
   fprintf(the_file, "^%s^\n", (desc.str_show() == NULL) ? "" : 
                                                      desc.str_show());
   fprintf(the_file, "%d\n", get_capacity());   

   return 1;
}

/***********************************************************************
 ** read_mudobject_attrib - reads in item attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             build_format - are we reading this from a builder save?
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int MudObject::read_mudobject_attrib(FILE *read_file, char *areaname, 
                                       int build_format, ErrLog *error_log)
{
   token_record *the_token;
   char         *temp_desc;
   Strings      holder;

   /* if this is in builder format, then read in the modified attrib */
   if (build_format)
   {
      the_token = get_token(read_file, '\0');
     
      if (the_token->token_type != T_NUMERICAL)
      {
         holder.sprintf("Invalid format in mudobject %s, area %s", get_name(),
                                                          get_area());
         error_log->log_err(holder.str_show(), "read_mudobject_attrib");
         return -1;
      }
  
      set_modified(atoi(the_token->the_string));
   }


   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in mudobject %s, area %s", get_name(),
		     areaname);
      error_log->log_err(holder.str_show(), "read_mudobject_attrib");
      return -1;
   }

   /* get the title of this location */
   the_token = get_token(read_file, '^');
   set_title(the_token->the_string);   

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in mudobject %s, area %s", get_name(),
		     areaname);
      error_log->log_err(holder.str_show(), "read_mudobject_attrib");
      return -1;
   }

   /* get the altnames */
   the_token = get_token(read_file, '^');
   set_altname(the_token->the_string);

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in mudobject %s", get_name());
      error_log->log_err(holder.str_show(), "read_mudobject_attrib");
      return -1;
   }

   /* get the clones */
   the_token = get_token(read_file, '^');
   set_clones(the_token->the_string);


   /* get the object location */
   the_token = get_token(read_file, '\0');
   set_location(the_token->the_string);


   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in mudobject %s, area %s", get_name(),
                                                            get_area());
      error_log->log_err(holder.str_show(), "read_mudobject_attrib");
      return -1;
      }

   /* get the specials */
   the_token = get_token(read_file, '^');
   set_special_str(the_token->the_string);

   /* get the descriptions */
   temp_desc = read_desc_type(read_file, error_log, this);
   
   if (temp_desc == NULL)
      return -1;
   set_desc(temp_desc);
   delete temp_desc;

   /* Set capacity */
   the_token = get_token(read_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.
      sprintf("Invalid format for attribute capacity in mudobject %s, area %s", 
                                                 get_name(), get_area());

      error_log->log_err(holder.str_show(), "read_mudobject_attrib");
      return -1;
   }
   set_capacity(atoi(the_token->the_string));

   return 1;
}

/***********************************************************************
 ** get_parent - returns the parent string of this object
 **
 **
 ***********************************************************************/

char *MudObject::get_parent()
{
   return parent.str_show();
}

/***********************************************************************
 ** set_parent - sets the parent string of this object
 **
 **
 ***********************************************************************/

void MudObject::set_parent(char *new_obj)
{
   parent = new_obj;
}


/***********************************************************************
 ** get_keywords - returns the list of keywords
 **
 **
 ***********************************************************************/

char *MudObject::get_keywords()
{
   return keywords.str_show();
}


/***********************************************************************
 ** set_keywords - sets the keywords string of this object
 **
 **
 ***********************************************************************/

void MudObject::set_keywords(char *new_str)
{
   keywords = new_str;
}


/***********************************************************************
 ** carried_by_player - is this object carried by a player in some fashion,
 **                     whether in a bag or what have you
 **
 ***********************************************************************/

int MudObject::carried_by_player()
{
   MudObject *container;

   container = get_contained_by();
   while (container != NULL)
   {
      if (container->get_type() == OBJ_TYPE_PLAYER)
         return 1;

      container = container->get_contained_by();
   }
   return 0;
}


/***********************************************************************
 ** write_contents - writes the contents of an object to the file
 **
 ** Parameters: the_file - the file we are writing to
 **
 ** Returns: num contained for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::write_contents(FILE *the_file)
{
   MudObject  *tmp_obj;
   Strings    act_name;
   int        count = 0;

   inventory.reset_current();
   tmp_obj = inventory.get_next();

   /* if they carry nothing, return 0 */
   if (tmp_obj == NULL)
      return 0;

   /* first we mark this with a # */
   fprintf(the_file, "{\n");

   /* write all the objects to file */
   while (tmp_obj != NULL)
   {
      count++;

      if (tmp_obj->get_parent() == NULL)
         act_name = tmp_obj->get_name();
      else
         act_name = tmp_obj->get_parent();

      if (tmp_obj->is_merger())
         fprintf(the_file, "%s@%s(%ld)\n", act_name.str_show(), 
                  tmp_obj->get_area(), ((Merger*)tmp_obj)->get_number_of());
      else
         fprintf(the_file, "%s@%s\n", act_name.str_show(), 
                                                  tmp_obj->get_area());

      count = count + tmp_obj->write_contents(the_file);
      
      tmp_obj = inventory.get_next();
   }
   
   fprintf(the_file, "}\n");
   return count;
}


/***********************************************************************
 ** set_attrib_special - for set_attrib, sets the specials attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_attrib_special(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify specials to set to.\n");
      return -1;
   }
   set_special_str(the_parsed->get_speech());
   the_builder->send_bldr("Specials on %s set to: %s\n", get_name(), 
                                                        get_special_str());
   set_modified(1);
   return 1;
}


/***********************************************************************
 ** add_clones - adds all cloned objects to this object 
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/
int MudObject::add_clones(Object_List *obj_dbase, ErrLog *error_log)
{
   Strings next_word;
   Strings holder;
   MudObject *orig_obj;
   MudObject *cloned_obj;
   char      *the_name;
   char      *the_area;
   int i = 1;

   if (get_clones() == NULL)
      return -1;
  
   next_word.assign_word(get_clones(), i);
   while ((next_word.str_show() != NULL) &&
          (next_word.str_len() > 0))
   {
      if (next_word.num_char('@') > 0)
      {
         the_name = next_word.str_show();
         the_area = the_name;
         while (*the_area != '@')
            the_area++;
         *the_area = '\0';
         the_area++;
      }
      else
      {
         the_name = next_word.str_show();
         the_area = get_area();
      }

      if ((orig_obj = obj_dbase->get_mudobject(the_area, the_name)) == NULL)
      {
         holder.sprintf("Mudobject '%s' clone '%s' doesn't seem to exist.", 
                                                    get_name(), the_name); 
         error_log->log_err(holder.str_show(), "add_clones");
      }
      else
      {
         if (!orig_obj->is_merger())
	 {
            if ((get_type() != OBJ_TYPE_LOCATION) && 
                (!orig_obj->is_a_moveable()) &&
                (orig_obj->get_type() != OBJ_TYPE_MOBILE))
	    {
               holder.sprintf(
                  "Mudobject '%s' clone '%s' not an actual object type.", 
                                                       get_name(), the_name); 
               error_log->log_err(holder.str_show(), "add_clones");
	    }
            else
	    {
               cloned_obj = obj_dbase->clone_object(orig_obj);
               if (orig_obj->get_type() == OBJ_TYPE_MOBILE)
	       {
                  Mobile *cloned_mob;

                  if (get_type() != OBJ_TYPE_LOCATION)
		  {
                     holder.sprintf("Attempt to clone Mobile '%s' "
                       "into non-location '%s'", orig_obj->get_name(), 
                                             get_name());
                     error_log->log_err(holder.str_show(), "add_clones");
                     
		  }
                  else
		  {
                     cloned_mob = (Mobile *) cloned_obj;
                     cloned_mob->add_clones(obj_dbase, error_log);
                     cloned_mob->start_wield(obj_dbase, error_log);
                     cloned_mob->start_wear(obj_dbase, error_log);
                     cloned_mob->set_location(((Location *) this), 
                                                  cloned_mob->get_loc());
		  }
	       }
               else
                  add_contained(cloned_obj);
	    }
         }
         else
	 {
            next_word.assign_word(get_clones(), i+1);
            if ((next_word.str_show() == NULL) || 
                (!isdigit(*(next_word.str_show()))))
	    {
               holder.sprintf("Mudobject '%s' cloned '%s' has no number_of "
                            "specified.\n", get_name(), the_name);
               error_log->log_err(holder.str_show(), "add_clones");
            }
            else
	    {
               cloned_obj = obj_dbase->clone_object(orig_obj);
               add_contained(cloned_obj);
               ((Merger *)cloned_obj)->set_number_of(
                                                 atoi(next_word.str_show()));
	    }
         }
      }
      i += 2;
      next_word.assign_word(get_clones(), i);
   }
   return 1;
}


/***********************************************************************
 ** set_attrib_desc - for set attrib command, sets the desc attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_attrib_desc(Player *the_player)
{
   if (the_player->get_long_input(&desc) < 0)
   {
      the_player->send_plr("Error reading in input, failed!\n");
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** get_long_input - like get_input, but this gets many strings of input
 **                  used for things like 'set desc'
 **
 ** Parameters: the_input - where we are putting the input into
 **
 ** Returns: the start_editor 
 **          -1 if failed
 **
 ***********************************************************************/
   
int MudObject::get_long_input(Strings *the_input)
{ 
   Editor *input_editor;

   if (the_input == NULL)
      return -1;

   input_editor = new Editor(the_input->str_show());

   if (get_type() == OBJ_TYPE_PLAYER)
   {
      Flags *tmp_gameflags;

      tmp_gameflags = ((Player *) this)->get_gameflags();

      if (!tmp_gameflags->get_flag(GAMEFLAG_FULLEDIT))
         input_editor->set_simple_mode();
   }
   return input_editor->start_editor((MudObject *) this, the_input);
}


/***********************************************************************
 ** set_capacity - sets the capacity of this mudobject
 **
 ** Parameters: the_num - the new capacity to set to
 **
 ***********************************************************************/

void MudObject::set_capacity(int the_num)
{
   if (the_num >= 0)
      capacity = the_num;
}


/***********************************************************************
 ** get_capacity - gets the capacity of this object
 **
 ***********************************************************************/

int MudObject::get_capacity(void)
{
   return capacity;
}


/***********************************************************************
 ** get_size_held - gets the sum of all sizes of objects held
 **
 ***********************************************************************/

int MudObject::get_size_held(void)
{
   MudObject *tmp_obj;
   int       total = 0;

   inventory.reset_current();
   tmp_obj = inventory.get_next();

   /* if they carry nothing, return 0 */
   if (tmp_obj == NULL)
      return 0;

   /* adds up the size of all objects the player has and is not wearing */
   while (tmp_obj != NULL)
   {
      if ((tmp_obj->is_a_moveable()) ||
          (tmp_obj->is_a_moveable() && 
          (tmp_obj->get_type() == OBJ_TYPE_WEARABLE) &&
          (is_an_individual()) &&  
          (!((Individual *) this)->is_worn((Wearable *) tmp_obj))))
         total += ((Moveable *) tmp_obj)->get_size();

      if (tmp_obj->is_an_individual())
         total += the_config.default_indsize;

      tmp_obj = inventory.get_next();
   }
   return total;
   
}


/***********************************************************************
 ** get_weight_held - gets the sum of all weights of objects carried
 **
 ***********************************************************************/

int MudObject::get_weight_held(void)
{
   MudObject *tmp_obj;
   int       total = 0;
   Flags     *tmp_itemflags;
   Moveable  *tmp_mov;

   inventory.reset_current();
   tmp_obj = inventory.get_next();

   /* if they carry nothing, return 0 */
   if (tmp_obj == NULL)
      return 0;

   /* adds up the size of all objects the player has and is not wearing */
   while (tmp_obj != NULL)
   {
      if (tmp_obj->is_a_moveable())
      {
         tmp_mov = (Moveable *) tmp_obj;
         tmp_itemflags = tmp_mov->get_itemflags();
         if (tmp_itemflags->get_flag(ITEMFLAG_CONTAINER))
            total += tmp_mov->get_weight_held();
         total += tmp_mov->get_weight();
      }

      tmp_obj = inventory.get_next();
   }
   return total;
   
}


/***********************************************************************
 ** set_attrib_capacity - for set attribute command, sets the capacity
 **                       attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::set_attrib_capacity(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify a number as well.\n");
      return -1;
   }
 
   if (!isdigit(*(the_parsed->get_speech()))) 
   {
      the_builder->send_bldr("You need to specify a number as capacity.\n");
      return -1;
   }

   set_capacity(atoi(the_parsed->get_speech()));
   the_builder->send_bldr("Capacity set to %d on moveable object %s.\n",
                                               get_capacity(), get_name());
   return 1;
}


/***********************************************************************
 ** copy_mudobject_attrib - copies all the mudobject attributes over from 
 **                         another mudobject
 **
 ** Parameters: copy_from - the object we copy attributes from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int MudObject::copy_mudobject_attrib(MudObject *copy_from)
{
   /******* set the mudobject attributes *****/
   set_title(copy_from->get_title());
   set_desc(copy_from->get_desc());
   set_altname(copy_from->get_altname());
   set_location(copy_from->get_location());
   set_clones(copy_from->get_clones());
   set_special_str(copy_from->get_special_str());
   special_list = copy_from->copy_specials();
   set_capacity(copy_from->get_capacity());
   return 1;
}



/***********************************************************************
 ** add_status - adds a status element to the list
 **
 ** Parameters: the_name - the name of the new status element
 **
 ** Returns:  1 if success
 **           0 if already there
 **          -1 for failure
 **
 ***********************************************************************/

int MudObject::add_status(char *the_name)
{
   return the_status.add_status(the_name);
}


/***********************************************************************
 ** add_status - adds a status element to the list
 **
 ** Parameters: the_name - the name of the new status element
 **             the_string - the data string to add
 **
 ** Returns:  1 if success
 **           0 if already there
 **          -1 for failure
 **
 ***********************************************************************/

int MudObject::add_status(char *the_name, char *the_string)
{
   return the_status.add_status(the_name, the_string);
}

/***********************************************************************
 ** has_status - finds a status element in the list
 **
 ** Parameters: the_name - the name of the status element to find
 **
 ** Returns:  1 if it exists
 **           0 if not there
 **          -1 for failure
 **
 ***********************************************************************/

int MudObject::has_status(char *the_name)
{
   return the_status.has_status(the_name);
}

/***********************************************************************
 ** get_status_string - finds a status element in the list and returns the
 **                     status data string if found
 **
 ** Parameters: the_name - the name of the status element to find
 **
 ** Returns:  pointer to the string if found, NULL if failed
 **
 ***********************************************************************/

char *MudObject::get_status_string(char *the_name)
{
   return the_status.get_status_string(the_name);
}


/***********************************************************************
 ** remove_status - removes a status element from the list
 **
 ** Parameters: the_name - the name of the status element to remove
 **
 ** Returns:  1 if found and removed
 **           0 if not found
 **          -1 for failure
 **
 ***********************************************************************/

int MudObject::remove_status(char *the_name)
{
   return the_status.remove_status(the_name);
}


/***********************************************************************
 ** contains_lit - object contains an object that is giving off light
 **
 ** Parameters: Nothing
 **
 ** Returns: 1 for has one, 0 for not
 **
 ***********************************************************************/

int MudObject::contains_lit(void)
{
   LinkedList *loc_list;
   MudObject  *next_obj;
   Item       *an_item;
   Flags      *itemflags;

   loc_list = get_inventory();
   loc_list->reset_current();
   
   next_obj = loc_list->get_next();
   while (next_obj != NULL)
   {
      if (next_obj->is_an_item())
      {
         an_item = (Item *) next_obj;
         itemflags = an_item->get_itemflags();
         if ((itemflags->get_flag(ITEMFLAG_LIT)) || 
             (itemflags->get_flag(ITEMFLAG_GLOWING)))
            return 1;
    
      }
      else if ((next_obj->get_type() == OBJ_TYPE_MOBILE) ||
               (next_obj->get_type() == OBJ_TYPE_PLAYER))
      {
         Individual *the_ind;

         the_ind = (Individual *) next_obj;
         if (the_ind->has_lit())
            return 1;
      }
      next_obj = loc_list->get_next();
   }
   return 0;
}


/***********************************************************************
 ** find_fire - object contains an object that is on fire
 **
 ** Parameters: Nothing
 **
 ** Returns: object pointer if one is found, NULL for not
 **
 ***********************************************************************/

MudObject *MudObject::find_fire(void)
{
   LinkedList *loc_list;
   MudObject  *next_obj;
   Item       *an_item;
   Flags      *itemflags;

   loc_list = get_inventory();
   loc_list->reset_current();
   
   next_obj = loc_list->get_next();
   while (next_obj != NULL)
   {
      if (next_obj->is_an_item())
      {
         an_item = (Item *) next_obj;
         itemflags = an_item->get_itemflags();
         if (itemflags->get_flag(ITEMFLAG_LIT))
            return next_obj;
    
      }
      next_obj = loc_list->get_next();
   }
   return NULL;
}


/***********************************************************************
 ** check_contained_spec - checks the contained objects of a certain type
 **                        if specified and looks for a special with a
 **                        specified trigger, running it if found
 **
 ** Parameters: trig_name - the trigger name to look for
 **             the_primary - the primary object for the special
 **             the_secondary - secondary object to pass into special
 **             the_player - the player using the special
 **             obj_type - type of object to look for (0 for all obj)
 **
 ** Returns: results of last special ran (3 if one said to exit)
 **
 ***********************************************************************/

int MudObject::check_contained_spec(char *trig_name, MudObject *the_primary,
            MudObject *the_secondary, Player *the_player, int obj_type)
{
   MudObject *tmp_obj;
   int       results = 0;

   inventory.reset_current();
   tmp_obj = inventory.get_next();
   while (tmp_obj != NULL)
   {
      if ((!obj_type) || (tmp_obj->get_type() == obj_type))
      {
         if ((results = check_specials(trig_name, tmp_obj, the_primary, 
                               the_secondary, tmp_obj, the_player)) == 3)
               return 3;
      }
      tmp_obj = inventory.get_next();
   }
   return results;
}


/***********************************************************************
 ** list_specials - lists all specials attached to this object by triggers
 **
 ** Parameters: the_player - the player to send the info to
 **
 ** Returns: results of last special ran (3 if one said to exit)
 **
 ***********************************************************************/

int MudObject::list_specials(Player *the_player)
{
   special_holder *tmp_special;
   int spec_num = 1;

   tmp_special = special_list;

   while (tmp_special != NULL)
   {
      the_player->send_plr("&+W\nTrigger Name &+m[&+G%d&+m]&*: \t%s\n", 
                         spec_num, (tmp_special->the_special)->get_trigger());
      tmp_special = tmp_special->next_special;
      spec_num++;
   }   
   return spec_num;
}


/***********************************************************************
 ** write_object - writes a mudobject to a file.  This write raises an
 **                error as we should not be writing just mudobjects ever
 **
 ** Parameters: the_file - the file we would write to
 **             build_format - is it in builder format
 **
 ***********************************************************************/

void MudObject::write_object(FILE *the_file, int build_format)
{
   build_format = 0;
   the_file = NULL;
   mainstruct->log_error("Tried to write mudobject to file.", "write_object");
}


/***********************************************************************
 ** describe - raises an error for trying to describe an entity
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void MudObject::describe(Player *the_player)
{
   the_player = NULL;
   mainstruct->log_error("Tried to describe a MudObject", "describe");
}


/***********************************************************************
 ** describe - raises an error for trying to describe an entity
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void MudObject::describe(Builder *the_builder)
{
   the_builder = NULL;
   mainstruct->log_error("Tried to describe a mudobject", "describe");
}


/***********************************************************************
 ** get_mem_size_mudobj - gets how much memory is being taken up by the
 **                       mudobject part of this object
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int MudObject::get_mem_size_mudobj()
{
   int size = 0;
   special_holder *tmp_spec;

   size += title.get_mem_size_dynamic();
   size += desc.get_mem_size_dynamic();
   size += altnames.get_mem_size_dynamic();
   size += location.get_mem_size_dynamic();
   size += clones.get_mem_size_dynamic();
   size += parent.get_mem_size_dynamic();

   tmp_spec = special_list;
   while (tmp_spec != NULL)
   {
      size += sizeof(tmp_spec);
      tmp_spec = tmp_spec->next_special;
   }

   size += the_status.get_mem_size_dynamic();
   size += inventory.get_mem_size_dynamic();

   return size;
}


/***********************************************************************
 ** get_mem_size - should not ever run this
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int MudObject::get_mem_size()
{
   return 0;
}

/***********************************************************************
 ** get_mem_size_dynamic - should not ever run this
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int MudObject::get_mem_size_dynamic()
{
   return 0;
}

#endif



