/**********************************************************************
 **
 ** Race - contains methods and attributes for a race object which defines
 **        a race through the builder port
 **
 ** Reviewed through:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/


#ifndef RACE_C
#define RACE_C

#include "sysdep.h"
#include "strings.h"
#include "race.h"
#include "lexer.h"
#include "builder.h"
#include "player.h"
#include "objtype.h"
#include "config.h"
#include "newfuncts.h"
#include "utils.h"
#include "global.h"
#include "specials.h"
#include "code.h"

/***********************************************************************
 ** Race (constructor) - loads race name and attributes
 **
 ** Parameters: race_name - this race name 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Race::Race(char *race_name)
{
   Strings tmp_name;

   obj_type = OBJ_TYPE_RACE;
   tmp_name.assign_word(race_name, 1);

   set_name(tmp_name.str_show());

   set_description("No desc.\n");
   set_init_desc("No desc.\n");
   set_init_brief("No brief.\n");
   set_init_location("needloc@needarea");
   set_death_location("needloc@needarea");
   set_death_text("No text.\n");

   init_str = init_dex = init_intel = init_con = init_wis = init_cha = 0;

   tutorial_list = NULL;
   modified = 0;
}


/***********************************************************************
 ** ~Race (destructor) - cleans up the race
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Race::~Race(void)
{
   Tutorial *tmp_tutorial;

   while (tutorial_list != NULL)
   {
      tmp_tutorial = tutorial_list->next_tutorial;
      delete tutorial_list;
      tutorial_list = tmp_tutorial;
   }
}


/***********************************************************************
 ** set_description - sets the description string for this race
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Race::set_description(char *the_string)
{
   if (the_string == NULL)
      return;

   description = the_string;
}

/***********************************************************************
 ** get_description - gets the description string for this race
 **
 ** Parameters: None
 **
 ** Returns: pointer to the description string
 **
 ***********************************************************************/

char *Race::get_description(void)
{
   return description.str_show();
}



/***********************************************************************
 ** set_init_desc - sets the initial description string that sets the
 **                 player's description when they first log on
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Race::set_init_desc(char *the_string)
{
   if (the_string == NULL)
      return;

   init_desc = the_string;
}

/***********************************************************************
 ** get_init_desc - sets the initial description string that sets the
 **                 player's description when they first log on
 **
 ** Parameters: None
 **
 ** Returns: pointer to the description string
 **
 ***********************************************************************/

char *Race::get_init_desc(void)
{
   return init_desc.str_show();
}


/***********************************************************************
 ** set_init_brief - sets the initial brief string that indicates what
 **                  brief the user will start out with
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Race::set_init_brief(char *the_string)
{
   if (the_string == NULL)
      return;

   init_brief = the_string;
}

/***********************************************************************
 ** get_init_brief - gets the initial brief string that sets the
 **                  player's brief when they first log on
 **
 ** Parameters: None
 **
 ** Returns: pointer to the description string
 **
 ***********************************************************************/

char *Race::get_init_brief(void)
{
   return init_brief.str_show();
}

/***********************************************************************
 ** set_init_ability - sets the initial abilities players of this race
 **                    will start out with
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Race::set_init_ability(char *the_string)
{
   if (the_string == NULL)
      return;

   init_ability = the_string;
}

/***********************************************************************
 ** get_init_ability - sets the initial ability players of this race
 **                    will start out with
 **
 ** Parameters: None
 **
 ** Returns: pointer to the ability string
 **
 ***********************************************************************/

char *Race::get_init_ability(void)
{
   return init_ability.str_show();
}

/***********************************************************************
 ** set_init_location - sets the initial starting-location for players
 **                     of this race.
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Race::set_init_location(char *the_string)
{
   if (the_string == NULL)
      return;

   init_location = the_string;
}

/***********************************************************************
 ** get_init_location - gets the initial starting-location for players
 **                     of this race.
 **
 ** Parameters: None
 **
 ** Returns: pointer to the ability string
 **
 ***********************************************************************/

char *Race::get_init_location(void)
{
   return init_location.str_show();
}


/***********************************************************************
 ** set_death_location - sets the location the players will go to when they
 **                      die
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Race::set_death_location(char *the_string)
{
   if (the_string == NULL)
      return;

   death_location = the_string;
}

/***********************************************************************
 ** get_death_location - gets the location the players go to when they
 **                      die
 **
 ** Parameters: None
 **
 ** Returns: pointer to the ability string
 **
 ***********************************************************************/

char *Race::get_death_location(void)
{
   return death_location.str_show();
}


/***********************************************************************
 ** set_death_text - sets the text the players will see when they die
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Race::set_death_text(char *the_string)
{
   if (the_string == NULL)
      return;

   death_text = the_string;
}

/***********************************************************************
 ** get_death_text - gets the text the players will see when they die
 **
 ** Parameters: None
 **
 ** Returns: pointer to the ability string
 **
 ***********************************************************************/

char *Race::get_death_text(void)
{
   return death_text.str_show();
}


/***********************************************************************
 ** set_allow_incl - sets the inclinations this race can use
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Race::set_allow_incl(char *the_string)
{
   if (the_string == NULL)
      return;

   allow_incl = the_string;
}

/***********************************************************************
 ** get_allow_incl - gets the inclinations this race can use
 **
 ** Parameters: None
 **
 ** Returns: pointer to the string
 **
 ***********************************************************************/

char *Race::get_allow_incl(void)
{
   return allow_incl.str_show();
}

/***********************************************************************
 ** set_allow_talent - sets the talents this race can use
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

void Race::set_allow_talent(char *the_string)
{
   if (the_string == NULL)
      return;

   allow_talent = the_string;
}

/***********************************************************************
 ** get_allow_talent - gets the talent this race can use
 **
 ** Parameters: None
 **
 ** Returns: pointer to the string
 **
 ***********************************************************************/

char *Race::get_allow_talent(void)
{
   return allow_talent.str_show();
}


/***********************************************************************
 ** set_init_str - sets the initial strength players of this race start
 **                out with
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Race::set_init_str(int the_num)
{
   if (the_num < 0)
      return;

   init_str = the_num;
}

/***********************************************************************
 ** get_init_str - sets the initial strength players of this race
 **                will start out with
 **
 ** Returns: the initial strength
 **
 ***********************************************************************/

int Race::get_init_str(void)
{
   return init_str;
}


/***********************************************************************
 ** set_init_dex - sets the initial dexterity players of this race start
 **                out with
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Race::set_init_dex(int the_num)
{
   if (the_num < 0)
      return;

   init_dex = the_num;
}

/***********************************************************************
 ** get_init_dex - sets the initial dexterity players of this race
 **                will start out with
 **
 ** Returns: the initial dexterity
 **
 ***********************************************************************/

int Race::get_init_dex(void)
{
   return init_dex;
}

/***********************************************************************
 ** set_init_intel - sets the initial intelligence players of this race 
 **                  start out with
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Race::set_init_intel(int the_num)
{
   if (the_num < 0)
      return;

   init_intel = the_num;
}

/***********************************************************************
 ** get_init_intel - sets the initial intelligence players of this race
 **                  will start out with
 **
 ** Returns: the initial intel
 **
 ***********************************************************************/

int Race::get_init_intel(void)
{
   return init_intel;
}


/***********************************************************************
 ** set_init_con - sets the initial constitution players of this race 
 **                start out with
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Race::set_init_con(int the_num)
{
   if (the_num < 0)
      return;

   init_con = the_num;
}

/***********************************************************************
 ** get_init_con - sets the initial constitution players of this race
 **                will start out with
 **
 ** Returns: the initial con
 **
 ***********************************************************************/

int Race::get_init_con(void)
{
   return init_con;
}


/***********************************************************************
 ** set_init_wis - sets the initial wisdom players of this race 
 **                start out with
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Race::set_init_wis(int the_num)
{
   if (the_num < 0)
      return;

   init_wis = the_num;
}

/***********************************************************************
 ** get_init_wis - sets the initial wisdom players of this race
 **                will start out with
 **
 ** Returns: the initial wis
 **
 ***********************************************************************/

int Race::get_init_wis(void)
{
   return init_wis;
}

/***********************************************************************
 ** set_init_cha - sets the initial charisma players of this race 
 **                start out with
 **
 ** Parameters: the_num - the number we are setting it to
 **
 ***********************************************************************/

void Race::set_init_cha(int the_num)
{
   if (the_num < 0)
      return;

   init_cha = the_num;
}

/***********************************************************************
 ** get_init_cha - sets the initial charisma players of this race
 **                will start out with
 **
 ** Returns: the initial cha
 **
 ***********************************************************************/

int Race::get_init_cha(void)
{
   return init_cha;
}


/***********************************************************************
 ** load_race - loads a race from a file into memory
 **
 ** Parameters: the_file - where we are getting the race from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Race::load_race(FILE *the_file, ErrLog *error_log, int is_builder)
{
   token_record *the_token;
   char         *tmp_charholder;
   Strings      holder;

   if (is_builder)
   {
       the_token = get_token(the_file,'\0');
 
       if (the_token->token_type != T_NUMERICAL)
       {
           error_log->log_err("Invalid format in race file", "load_race");
           return -1;
       }
       set_modified(atoi(the_token->the_string));
   }

   /* get the description */
   tmp_charholder = read_desc_type(the_file, error_log, NULL); 
   set_description(tmp_charholder);
   delete tmp_charholder;

   /* get the initial description */
   tmp_charholder = read_desc_type(the_file, error_log, NULL); 
   set_init_desc(tmp_charholder);
   delete tmp_charholder;

   /* get the initial brief */
   tmp_charholder = read_desc_type(the_file, error_log, NULL); 
   set_init_brief(tmp_charholder);
   delete tmp_charholder;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format for attribute initability in race %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }

   /* get the initability of this race */
   the_token = get_token(the_file, '^');
   set_init_ability(the_token->the_string);   

   /* Set init_location value */
   the_token = get_token(the_file, '\0');     
   set_init_location(the_token->the_string);

   /* Set death_location value */
   the_token = get_token(the_file, '\0');     
   set_death_location(the_token->the_string);

   /* get the death text */
   tmp_charholder = read_desc_type(the_file, error_log, NULL); 
   set_death_text(tmp_charholder);
   delete tmp_charholder;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in race %s", get_name());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }

   /* get the allowed inclinations of this race */
   the_token = get_token(the_file, '^');
   set_allow_incl(the_token->the_string);   

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in race %s", get_name());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }

   /* get the allowed talents of this race */
   the_token = get_token(the_file, '^');
   set_allow_talent(the_token->the_string);   

   /* Set init_str value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute initstr for race %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }
   set_init_str(atoi(the_token->the_string));


   /* Set init_dex value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute initdex for race %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }
   set_init_dex(atoi(the_token->the_string));

   /* Set init_intel value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute initintel for race %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }
   set_init_intel(atoi(the_token->the_string));

   /* Set init_con value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute initcon for race %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }
   set_init_con(atoi(the_token->the_string));

   /* Set init_wis value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute initwis for race %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }
   set_init_wis(atoi(the_token->the_string));

   /* Set init_cha value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute initcha for race %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }
   set_init_cha(atoi(the_token->the_string));

   the_token = get_token(the_file, '\0');
   while (the_token->token_type == T_PLUS)
   {
      Strings name;
      int     sequence, mandatory;

      /* first we get the tutorial name */
      the_token = get_token(the_file,'\0');
      name = the_token->the_string;

      add_tutorial(name.str_show());

      /* Read in the sequence */
      the_token = get_token(the_file, '\0');
     
      if (the_token->token_type != T_NUMERICAL)
      {
         holder.sprintf("Invalid format for tutorial sequence in race %s@%s",
                                               get_name(), get_area());
         error_log->log_err(holder.str_show(), "load_race");
         return -1;
      }
      sequence = (atoi(the_token->the_string));

      set_tutorial_sequence(name.str_show(), sequence);

      /* get the next item, it should be a '^', if not, raise error */
      the_token = get_token(the_file, '\0');
      if (the_token->token_type != T_CARROT)
      {
         holder.sprintf("Invalid format for attribute prompt in race %s@%s", 
                                                    get_name(), get_area());
         error_log->log_err(holder.str_show(), "load_race");
         return -1;
      }

      /* get the prompt of this tutorial */
      the_token = get_token(the_file, '^');
      set_tutorial_prompt(name.str_show(), the_token->the_string);

      /* Set tutorial_special value */
      the_token = get_token(the_file, '\0');     
      set_tutorial_special(name.str_show(), the_token->the_string);

      /* Read in if the tutorial is mandatory */
      the_token = get_token(the_file, '\0');
     
      if (the_token->token_type != T_NUMERICAL)
      {
         holder.sprintf
             ("Invalid format for tutorial 'mandatory' attribute in race %s@%s",
                                               get_name(), get_area());
         error_log->log_err(holder.str_show(), "load_race");
         return -1;
      }
      mandatory = (atoi(the_token->the_string));

      set_tutorial_mandatory(name.str_show(), mandatory);

      the_token = get_token(the_file, '\0');
   }

   if (the_token->token_type != T_POUND)
   {
      holder.sprintf("Invalid format in tutorial for race %s@%s", get_name(),
                                                                   get_area());
      error_log->log_err(holder.str_show(), "load_race");
      return -1;
   }
   return 1;
}


/***********************************************************************
 ** describe - describes the race to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Race::describe(Builder *the_builder)
{
   Tutorial *tmp_tutorial;

   the_builder->send_bldr("\n&+GRace: \t\t&+M%s&*\n", get_name());
   the_builder->send_bldr("\n&+GInitLoc: \t&+w%s&*\n", get_init_location());
   the_builder->send_bldr("&+GDesc: \n&+w%s&*\n", get_description());
   the_builder->send_bldr("&+GInitDesc: \n&+w%s&*\n", get_init_desc());
   the_builder->send_bldr("&+GInitBrief: \n&+w%s&*\n", get_init_brief());
   the_builder->send_bldr("&+GInitAbility: \t&+w%s&*\n", get_init_ability());
   the_builder->send_bldr("&+GDeathLoc: \t&+w%s&*\n", get_death_location());
   the_builder->send_bldr("&+GDeathText:&* \n%s&*\n", get_death_text());
   the_builder->send_bldr("&+GInitStr: \t&+w%d&*\n", get_init_str());
   the_builder->send_bldr("&+GInitDex: \t&+w%d&*\n", get_init_dex());
   the_builder->send_bldr("&+GInitIntel: \t&+w%d&*\n", get_init_intel());
   the_builder->send_bldr("&+GInitCon: \t&+w%d&*\n", get_init_con());
   the_builder->send_bldr("&+GInitWis: \t&+w%d&*\n", get_init_wis());
   the_builder->send_bldr("&+GInitCha: \t&+w%d&*\n", get_init_cha());
   the_builder->send_bldr("\n\n");

   tmp_tutorial = tutorial_list;
   while (tmp_tutorial != NULL)
   {
      the_builder->send_bldr("\n&+GName: \t\t&+M%s\n", 
                                       tmp_tutorial->name.str_show());
      the_builder->send_bldr("&+GSequence: \t&*%d\n", tmp_tutorial->sequence);
      the_builder->send_bldr("&+GPrompt:\t\t&*%s\n", 
                                       tmp_tutorial->prompt.str_show());
      the_builder->send_bldr("&+GSpecial:\t&*%s\n", 
                                       tmp_tutorial->tutorial_special.str_show());
      the_builder->send_bldr("&+GMandatory: \t%s\n", 
                    (tmp_tutorial->mandatory) ? "&+CYes&*" : "&+cNo&*");

      tmp_tutorial = tmp_tutorial->next_tutorial;
   }
   the_builder->send_bldr("\n");

}


/***********************************************************************
 ** describe - describes the race to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Race::describe(Player *the_player)
{
   Tutorial *tmp_tutorial;

   the_player->send_plr("\n&+GRace: \t\t&+M%s&*\n", get_name());
   the_player->send_plr("\n&+GInitLoc: \t&+w%s&*\n", get_init_location());
   the_player->send_plr("&+GDesc: \n&+w%s&*\n", get_description());
   the_player->send_plr("&+GInitDesc: \n&+w%s&*\n", get_init_desc());
   the_player->send_plr("&+GInitBrief: \n&+w%s&*\n", get_init_brief());
   the_player->send_plr("&+GInitAbility: \t&+w%s&*\n", get_init_ability());
   the_player->send_plr("&+GDeathLoc: \t&+w%s&*\n", get_death_location());
   the_player->send_plr("&+GDeathText:&* \n%s&*\n", get_death_text());
   the_player->send_plr("&+GInitStr: \t&+w%d&*\n", get_init_str());
   the_player->send_plr("&+GInitDex: \t&+w%d&*\n", get_init_dex());
   the_player->send_plr("&+GInitIntel: \t&+w%d&*\n", get_init_intel());
   the_player->send_plr("&+GInitCon: \t&+w%d&*\n", get_init_con());
   the_player->send_plr("&+GInitWis: \t&+w%d&*\n", get_init_wis());
   the_player->send_plr("&+GInitCha: \t&+w%d&*\n", get_init_cha());
   the_player->send_plr("\n\n");

   tmp_tutorial = tutorial_list;
   while (tmp_tutorial != NULL)
   {
      the_player->send_plr("\n&+GName: \t\t&+M%s\n", 
                                       tmp_tutorial->name.str_show());
      the_player->send_plr("&+GSequence: \t&*%d\n", tmp_tutorial->sequence);
      the_player->send_plr("&+GPrompt:\t\t&*%s\n", 
                                       tmp_tutorial->prompt.str_show());
      the_player->send_plr("&+GSpecial:\t&*%s\n", 
                                       tmp_tutorial->tutorial_special.str_show());
      the_player->send_plr("&+GMandatory: \t%s\n", 
                    (tmp_tutorial->mandatory) ? "&+CYes&*" : "&+cNo&*");

      tmp_tutorial = tmp_tutorial->next_tutorial;
   }
   the_player->send_plr("\n");

}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Race::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on a race.\n"
               "   desc, initability, deathloc, deathtext, initstr, \n"
               "   initdex, initintel, initcon, initwis, initcha,\n"
	       "   initbrief, initdesc, sequence, prompt, special,\n"
	       "   and mandatory\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "desc",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&description) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initdesc",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&init_desc) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initloc",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
             send_bldr("You need to specify an InitLoc to set to.\n");
         return -1;
      }
 
      set_init_location(the_parsed->get_speech());

      the_builder->send_bldr("InitLoc on %s set to: %s\n", get_name(), 
                                                     get_init_location());
//      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "deathloc",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
             send_bldr("You need to specify an DeathLoc to set to.\n");
         return -1;
      }
 
      set_death_location(the_parsed->get_speech());

      the_builder->send_bldr("DeathLoc on %s set to: %s\n", get_name(), 
                                                     get_death_location());
//      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initbrief",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&init_brief) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "deathtext",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&death_text) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initability",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
             send_bldr("You need to specify an InitAbility to set to.\n");
         return -1;
      }
 
      set_init_ability(the_parsed->get_speech());

      the_builder->send_bldr("InitAbility on %s set to: %s\n", get_name(), 
                                                     get_init_ability());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initstr",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->
                   send_bldr("You need to specify a number as InitStr.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value <= 0)
      {
         the_builder->send_bldr("Value must be greater than 0.\n");
         return -1;
      }

      set_init_str(value);
      the_builder->send_bldr("InitStr set to %d on race object %s.\n",
                                          get_init_str(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initdex",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->
                   send_bldr("You need to specify a number as InitDex.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value <= 0)
      {
         the_builder->send_bldr("Value must be greater than 0.\n");
         return -1;
      }

      set_init_dex(value);
      the_builder->send_bldr("InitDex set to %d on race object %s.\n",
                                          get_init_dex(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initintel",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->
                   send_bldr("You need to specify a number as InitIntel.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value <= 0)
      {
         the_builder->send_bldr("Value must be greater than 0.\n");
         return -1;
      }

      set_init_intel(value);
      the_builder->send_bldr("InitIntel set to %d on race object %s.\n",
                                          get_init_intel(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initcon",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->
                   send_bldr("You need to specify a number as InitCon.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value <= 0)
      {
         the_builder->send_bldr("Value must be greater than 0.\n");
         return -1;
      }

      set_init_con(value);
      the_builder->send_bldr("InitCon set to %d on race object %s.\n",
                                          get_init_con(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initwis",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->
                   send_bldr("You need to specify a number as InitWis.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value <= 0)
      {
         the_builder->send_bldr("Value must be greater than 0.\n");
         return -1;
      }

      set_init_wis(value);
      the_builder->send_bldr("InitWis set to %d on race object %s.\n",
                                          get_init_wis(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "initcha",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->
                   send_bldr("You need to specify a number as InitCha.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value <= 0)
      {
         the_builder->send_bldr("Value must be greater than 0.\n");
         return -1;
      }

      set_init_con(value);
      the_builder->send_bldr("InitCha set to %d on race object %s.\n",
                                          get_init_cha(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "sequence",
                               strlen(the_parsed->get_target1())))
   {
      Strings    tut_name;
      Strings    number;
      int        newnum;
      int        results;

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("usage: set sequence <tutorialname> <number>\n");
         return -1;
      }

      tut_name.assign_word(the_parsed->get_speech(), 1);
      if (tut_name.str_show() == NULL)
      {
         the_builder->send_bldr("usage: set sequence <tutorialname> <number>\n");
         return -1;
      }
      
      number.assign_word(the_parsed->get_speech(), 2);
      if (number.str_show() == NULL)
      {
         the_builder->send_bldr("usage: set sequence <tutorialname> <number>\n");
         return -1;
      }

      if (!isdigit(*(number.str_show())))
      {
         the_builder->send_bldr("usage: set sequence <tutorialname> <number>\n");
         return -1;
      }

      newnum = atoi(number.str_show());

      if ((results = set_tutorial_sequence(tut_name.str_show(), newnum)) 
                                                                          == -2)
      {
         the_builder->send_bldr("That tutorial does not exist.\n");
         return -1;
      }

      the_builder->send_bldr("Tutorial %s sequence set to: %d\n", 
                                                   tut_name.str_show(), results);
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "prompt",
                               strlen(the_parsed->get_target1())))
   {
      Strings    tut_name;
      Strings    new_prompt;

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("usage: set prompt <tutorialname> <prompt>\n");
         return -1;
      }

      tut_name.assign_word(the_parsed->get_speech(), 1);
      if (tut_name.str_show() == NULL)
      {
         the_builder->send_bldr("usage: set prompt <tutorialname> <prompt>\n");
         return -1;
      }
      
      new_prompt.assign_phrase(the_parsed->get_speech(), 1);
      if (new_prompt.str_show() == NULL)
      {
         the_builder->send_bldr("usage: set prompt <tutorialname> <prompt>\n");
         return -1;
      }

      if (set_tutorial_prompt(tut_name.str_show(), 
                                                 new_prompt.str_show()) == -2)
      {
         the_builder->send_bldr("That tutorial does not exist.\n");
         return -1;
      }

      the_builder->send_bldr("Tutorial %s prompt set to: %s\n", 
                                    tut_name.str_show(), new_prompt.str_show());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "special",
                               strlen(the_parsed->get_target1())))
   {
      Strings    tut_name;
      Strings    new_special;

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
           send_bldr("usage: set special <tutorialname> <specialname>\n");
         return -1;
      }

      tut_name.assign_word(the_parsed->get_speech(), 1);
      if (tut_name.str_show() == NULL)
      {
         the_builder->
            send_bldr("usage: set special <tutorialname> <specialname>\n");
         return -1;
      }
      
      new_special.assign_word(the_parsed->get_speech(), 2);
      if (new_special.str_show() == NULL)
      {
         the_builder->
              send_bldr("usage: set special <tutorialname> <specialname>\n");
         return -1;
      }

      if (set_tutorial_special(tut_name.str_show(), 
                                                 new_special.str_show()) == -2)
      {
         the_builder->send_bldr("That tutorial does not exist.\n");
         return -1;
      }

      the_builder->send_bldr("Tutorial %s special set to: %s\n", 
                                tut_name.str_show(), new_special.str_show());
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "mandatory",
                               strlen(the_parsed->get_target1())))
   {
      Strings    tut_name;
      Strings    yesno;
      int        results;

      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("usage: set mandatory <tutorialname> <yes|no>\n");
         return -1;
      }

      tut_name.assign_word(the_parsed->get_speech(), 1);
      if (tut_name.str_show() == NULL)
      {
         the_builder->send_bldr("usage: set mandatory <tutorialname> <yes|no>\n");
         return -1;
      }
      
      yesno.assign_word(the_parsed->get_speech(), 2);
      if (yesno.str_show() == NULL)
      {
         the_builder->send_bldr("usage: set mandatory <tutorialname> <yes|no>\n");
         return -1;
      }

      if (!STRCASECMP(yesno.str_show(), "yes"))
      {
         results = 1;
      }  
      else
         results = 0;

      if (set_tutorial_mandatory(tut_name.str_show(), results) == -2)
      {
         the_builder->send_bldr("That tutorial does not exist.\n");
         return -1;
      }

      the_builder->send_bldr("Tutorial %s mandatory set to: %s\n", 
                      tut_name.str_show(), results ? "yes" : "no");
      set_modified(1);
      return 1;
   }

   the_builder->send_bldr("The attribute '%s' is not a race attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the race to a specified file in specified
 **                format
 **
 ** Parameters: the_file - the file to write to
 **             build_format - shall we use builder format or not
 **
 ***********************************************************************/
   
void Race::write_object(FILE *the_file, int build_format)
{
   Tutorial *tmp_tutorial;

   fprintf(the_file, "\nrace %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   fprintf(the_file, "^%s^\n", (get_description() == NULL) ? "" : 
                                                        get_description());
   fprintf(the_file, "^%s^\n", (get_init_desc() == NULL) ? "" : 
                                                        get_init_desc());
   fprintf(the_file, "^%s^\n", (get_init_brief() == NULL) ? "" : 
                                                        get_init_brief());
   fprintf(the_file, "^%s^\n", (get_init_ability() == NULL) ? "" : 
                                                      get_init_ability());
   fprintf(the_file, "%s\n", get_init_location());

   fprintf(the_file, "%s\n", get_death_location());
   fprintf(the_file, "^%s^\n", (get_death_text() == NULL) ? "" :
                                                        get_death_text());
   fprintf(the_file, "^%s^\n", (get_allow_incl() == NULL) ? "" : 
                                                      get_allow_incl());
   fprintf(the_file, "^%s^\n", (get_allow_talent() == NULL) ? "" : 
                                                      get_allow_talent());
   fprintf(the_file, "%d\n", get_init_str());
   fprintf(the_file, "%d\n", get_init_dex());
   fprintf(the_file, "%d\n", get_init_intel());
   fprintf(the_file, "%d\n", get_init_con());
   fprintf(the_file, "%d\n", get_init_wis());
   fprintf(the_file, "%d\n", get_init_cha());

   tmp_tutorial = tutorial_list;

   while (tmp_tutorial != NULL)
   {
      fprintf(the_file, "+\n%s\n", tmp_tutorial->name.str_show());
      fprintf(the_file, "%d\n", tmp_tutorial->sequence);
      fprintf(the_file, "^%s^\n", (tmp_tutorial->prompt.str_show() == NULL) ? 
                                       "" : tmp_tutorial->prompt.str_show());
      fprintf(the_file, "%s\n", 
             (tmp_tutorial->tutorial_special.str_show() == NULL) ? 
                          "none" : tmp_tutorial->tutorial_special.str_show());
      fprintf(the_file, "%d\n", tmp_tutorial->mandatory);
      tmp_tutorial = tmp_tutorial->next_tutorial;
   }
   fprintf(the_file, "#\n");
}


/***********************************************************************
 ** is_modified - has this race been modified?
 **
 ** Parameters: None
 **
 ** Returns:  1 for yes, 0 for no
 **
 ***********************************************************************/
   
int Race::is_modified(void)
{
   return modified;
}


/***********************************************************************
 ** add_tutorial - adds a tutorial to the list of tutorials 
 **
 ** Parameters: the_name - the tutorial name
 **
 ** Returns: sequence number for success, -1 for failure, -2 for name already used
 **
 ***********************************************************************/

int Race::add_tutorial(char *the_name)
{
   Tutorial *new_tutorial;
   Tutorial *tmp_tutorial;

   new_tutorial = new Tutorial(the_name);
   new_tutorial->next_tutorial = NULL;

   new_tutorial->sequence = 0;
   new_tutorial->mandatory = 0;
   new_tutorial->prompt = "No prompt";


   if (tutorial_list == NULL)
   {
      tutorial_list = new_tutorial;
      new_tutorial->sequence = 1;
      return 1;
   }

   if (!STRCASECMP(tutorial_list->name.str_show(), the_name))
      return -2;

   tmp_tutorial = tutorial_list;
   while (tmp_tutorial->next_tutorial != NULL)
   {
      tmp_tutorial = tmp_tutorial->next_tutorial;
      if (!STRCASECMP(tmp_tutorial->name.str_show(), the_name))
         return -2;
   }

   tmp_tutorial->next_tutorial = new_tutorial;
   new_tutorial->sequence = tmp_tutorial->sequence + 1;

   return new_tutorial->sequence;
}


/***********************************************************************
 ** del_tutorial - deletes a tutorial from the list of tutorials 
 **
 ** Parameters: the_name - the tutorial name to delete
 **
 ** Returns: 1 for success, -1 for failure, -2 for not found
 **
 ***********************************************************************/

int Race::del_tutorial(char *the_name)
{
   Tutorial *tmp_tutorial;
   Tutorial *prev_tutorial = NULL;

   if (the_name == NULL)
      return -1;

   tmp_tutorial = tutorial_list;
   while (tmp_tutorial != NULL)
   {
      if (!STRCASECMP(tmp_tutorial->name.str_show(), the_name))
         break;
      prev_tutorial = tmp_tutorial;
      tmp_tutorial = tmp_tutorial->next_tutorial;
   }

   if (tmp_tutorial == NULL)
      return -2;

   if (prev_tutorial == NULL)
      tutorial_list = tutorial_list->next_tutorial;
   else
      prev_tutorial->next_tutorial = tmp_tutorial->next_tutorial;

   delete tmp_tutorial;

   return 1;
}


/***********************************************************************
 ** get_tutorial - gets a tutorial that has a specific name from the list
 **
 ** Parameters: the_name - the name of the tutorial we seek
 **
 ** Returns: pointer to the item if found, NULL if not found
 **
 ***********************************************************************/

Tutorial *Race::get_tutorial(char *the_name)
{
   Tutorial *tmp_tutorial;

   tmp_tutorial = tutorial_list;
   while (tmp_tutorial != NULL)
   {
      if (!STRCASECMP(the_name, tmp_tutorial->name.str_show()))
         return tmp_tutorial;
      tmp_tutorial = tmp_tutorial->next_tutorial;
   }
   return NULL;
}


/***********************************************************************
 ** set_tutorial_name - sets the tutorial name for a particular tutorial 
 **                     after first checking if any exist by that name
 **
 ** Parameters: the_name - the alias for this itemname
 **             new_name - the new name to give it
 **
 ** Returns: 1 for success, -1 for failure, -2 for tutorial doesn't exist
 **          -3 for one already exists by that name
 **
 ***********************************************************************/

int Race::set_tutorial_name(char *the_name, char *new_name)
{
   Tutorial *tmp_tutorial;

   if ((the_name == NULL) || (new_name == NULL))
      return -1;

   if ((tmp_tutorial = get_tutorial(the_name)) == NULL)
      return -2;

   if (get_tutorial(new_name) != NULL)
      return -3;

   tmp_tutorial->name = new_name;
   return 1;
}

/***********************************************************************
 ** set_tutorial_prompt - sets the prompt for this tutorial
 **
 ** Parameters: the_name - the name of the tutorial to set
 **             new_prompt - the new prompt to give it
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Race::set_tutorial_prompt(char *the_name, char *new_prompt)
{
   Tutorial *tmp_tutorial;

   if (the_name == NULL)
      return -1;

   if ((tmp_tutorial = get_tutorial(the_name)) == NULL)
      return -2;

   tmp_tutorial->prompt = new_prompt;
   if (tmp_tutorial->prompt.str_len() > (MAXPROMPTLEN - 2))
      tmp_tutorial->prompt.truncate(MAXPROMPTLEN - 2);

   return 1;
}

/***********************************************************************
 ** set_tutorial_special - sets the special for this tutorial
 **
 ** Parameters: the_name - the name of the tutorial to set
 **             new_special - the new special to give it
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Race::set_tutorial_special(char *the_name, char *new_special)
{
   Tutorial *tmp_tutorial;

   if (the_name == NULL)
      return -1;

   if ((tmp_tutorial = get_tutorial(the_name)) == NULL)
      return -2;

   tmp_tutorial->tutorial_special = new_special;

   return 1;
}

/***********************************************************************
 ** set_tutorial_mandatory - sets if this tutorial is mandatory or not
 **
 ** Parameters: the_name - the name of the tutorial to set
 **             new_value - the new value to give it
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Race::set_tutorial_mandatory(char *the_name, int new_value)
{
   Tutorial *tmp_tutorial;

   if (the_name == NULL)
      return -1;

   if ((tmp_tutorial = get_tutorial(the_name)) == NULL)
      return -2;

   tmp_tutorial->mandatory = !(!(new_value));

   return 1;
}

/***********************************************************************
 ** set_tutorial_sequence - changes the sequence of this tutorial
 **
 ** Parameters: the_name - the name of the tutorial to set
 **             new_sequence - the new sequence to give it
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Race::set_tutorial_sequence(char *the_name, int new_sequence)
{
   Tutorial *tmp_tutorial;
   Tutorial *prev_tutorial = NULL;

   if ((the_name == NULL) || (new_sequence <= 0))
      return -1;

   /* first find our tutorial */
   tmp_tutorial = tutorial_list;
   while (tmp_tutorial != NULL)
   {
      if (!STRCASECMP(the_name, tmp_tutorial->name.str_show()))
         break;
      prev_tutorial = tmp_tutorial;
      tmp_tutorial = tmp_tutorial->next_tutorial;
   }
   if (tmp_tutorial == NULL)
      return -2;

   /* now remove it */
   if (prev_tutorial == NULL)
      tutorial_list = tutorial_list->next_tutorial;
   else
      prev_tutorial->next_tutorial = tmp_tutorial->next_tutorial;

   /* set the sequence */
   tmp_tutorial->sequence = new_sequence;

   /* re-insert it */
   insert_tutorial(tmp_tutorial);
   
   /* now reset all sequence numbers, returning the new sequence */
   reset_sequence();

   return tmp_tutorial->sequence;
}

/***********************************************************************
 ** insert_tutorial - inserts a tutorial into the list based on sequence
 **
 ** Parameters: the_name - the name of the tutorial to set
 **             new_sequence - the ne
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Race::insert_tutorial(Tutorial *the_tutorial)
{
   Tutorial *tmp_tutorial;

   if (tutorial_list == NULL)
   {
      tutorial_list = the_tutorial;
      return 1;
   }

   tmp_tutorial = tutorial_list;
   if (the_tutorial->sequence <= tmp_tutorial->sequence)
   {
      the_tutorial->next_tutorial = tutorial_list;
      tutorial_list = the_tutorial;
   }
   else
   {
      while ((tmp_tutorial->next_tutorial != NULL) && 
             (the_tutorial->sequence > (tmp_tutorial->next_tutorial)->sequence))
      { 
         tmp_tutorial = tmp_tutorial->next_tutorial;
      }
      the_tutorial->next_tutorial = tmp_tutorial->next_tutorial;
      tmp_tutorial->next_tutorial = the_tutorial;
   }
   return 1;
}

/***********************************************************************
 ** reset_sequence - sets the sequence of the list in incremental order
 **
 ** Parameters: None
 **
 ** Returns: num in sequence if success, -1 if failed
 **
 ***********************************************************************/

int Race::reset_sequence(void)
{
   Tutorial *tmp_tutorial;
   int      counter = 1;

   tmp_tutorial = tutorial_list;
   while (tmp_tutorial != NULL)
   {
      tmp_tutorial->sequence = counter;
      counter++;
      tmp_tutorial = tmp_tutorial->next_tutorial;
   }
   return counter-1;
}


/***********************************************************************
 ** get_next_tutorial - gets the next tutorial name based on the previous
 **                     tutorial name
 **
 ** Parameters: previous_name - the previous tutorial name
 **
 ** Returns: num in sequence if success, -1 if failed
 **
 ***********************************************************************/

char *Race::get_next_tutorial(char *previous_name)
{
   Tutorial *tmp_tutorial;

   tmp_tutorial = tutorial_list;       
   if (previous_name != NULL)
   {
      while ((tmp_tutorial != NULL) &&
             (STRCASECMP(previous_name, tmp_tutorial->name.str_show())))
         tmp_tutorial = tmp_tutorial->next_tutorial;
   }
   
   if ((tmp_tutorial == NULL) || (tmp_tutorial->next_tutorial == NULL))
      return NULL;

   if (previous_name == NULL)
      return tmp_tutorial->name.str_show();
   return tmp_tutorial->next_tutorial->name.str_show();
}

/***********************************************************************
 ** get_tutorial_prompt - gets the tutorial prompt based on the tutorial
 **                       name given
 **
 ** Parameters: the_name - the name of the prompt to get
 **
 ** Returns: pointer to prompt if success, NULL if failed
 **
 ***********************************************************************/

char *Race::get_tutorial_prompt(char *the_name)
{
   Tutorial *tmp_tutorial;

   tmp_tutorial = tutorial_list;       
   while ((tmp_tutorial != NULL) &&
          (STRCASECMP(the_name, tmp_tutorial->name.str_show())))
      tmp_tutorial = tmp_tutorial->next_tutorial;
   
   if (tmp_tutorial == NULL)
      return NULL;

   return tmp_tutorial->prompt.str_show();
}

/***********************************************************************
 ** get_tutorial_mandatory - gets the tutorial mandatory based on the tutorial
 **                          name given
 **
 ** Parameters: the_name - the name of the prompt to get
 **
 ** Returns: 1 or 0 if successful, -1 if failed
 **
 ***********************************************************************/

int Race::get_tutorial_mandatory(char *the_name)
{
   Tutorial *tmp_tutorial;

   tmp_tutorial = tutorial_list;       
   while ((tmp_tutorial != NULL) &&
          (STRCASECMP(the_name, tmp_tutorial->name.str_show())))
      tmp_tutorial = tmp_tutorial->next_tutorial;
   
   if (tmp_tutorial == NULL)
      return -1;

   return tmp_tutorial->mandatory;
}


/***********************************************************************
 ** start_tutorial - starts the tutorial indicated for the player
 **
 ** Parameters: the_name - the name of the tutorial to start
 **             the_player - the player to start the tutorial for
 **
 ** Returns: 1 if successful, -1 if failed
 **
 ***********************************************************************/

int Race::start_tutorial(char *the_name, Player *the_player)
{
   Tutorial    *tmp_tutorial;
   Specials    *the_special;
   Inp_Handler *the_handler;
   Strings     holder;   
   in_params   fill_param;

   tmp_tutorial = tutorial_list;

   while ((tmp_tutorial != NULL) &&
          (STRCASECMP(the_name, tmp_tutorial->name.str_show())))
      tmp_tutorial = tmp_tutorial->next_tutorial;

   if (tmp_tutorial == NULL)
      return -1;

   the_handler = the_player->get_input_handler();
   
   if ((the_special = mainstruct->get_special(
                            tmp_tutorial->tutorial_special.str_show())) == NULL)
   {
      holder.sprintf("Tutorial %s special %s in race %s not valid.", 
        tmp_tutorial->name.str_show(), tmp_tutorial->tutorial_special.str_show(),
                                           get_name()); 
      mainstruct->log_error(holder.str_show(), "start_tutorial");
      the_player->send_plr("Error, could not execute tutorial.\n");
      return -1;
   }

   if (!tmp_tutorial->mandatory)
      the_handler->pop_input_handler();

   fill_param.primary_obj = NULL;
   fill_param.secondary_obj = NULL;
   fill_param.this_obj = the_player;

   tmp_tutorial->the_environment.trig_used = "ontutorial";
   tmp_tutorial->the_environment.target_str = "none";
   tmp_tutorial->the_environment.exec_vars = NULL;
   the_special->run_special(the_player, &fill_param, 
                                               &tmp_tutorial->the_environment);

   return 1;
}


/***********************************************************************
 ** set_modified - shall we set this?
 **
 ** Parameters: the_num - the number to set it to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
void Race::set_modified(int the_num)
{
   modified = the_num;
}


/***********************************************************************
 ** copy_object - copies the object to an race of a different name
 **
 ** Parameters: copy_from - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Race::copy_object(Entity *copy_obj)
{
   Race *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_RACE)
      return 0;

   copy_from = (Race *) copy_obj;

   /******* set the action attributes *****/
   set_description(copy_from->get_description());
   set_init_desc(copy_from->get_init_desc());
   set_init_ability(copy_from->get_init_ability());
   set_init_location(copy_from->get_init_location());
   set_init_str(copy_from->get_init_str());
   set_init_dex(copy_from->get_init_dex());
   set_init_intel(copy_from->get_init_intel());
   set_init_con(copy_from->get_init_con());
   set_init_wis(copy_from->get_init_wis());
   set_init_cha(copy_from->get_init_cha());
   set_init_brief(copy_from->get_init_brief());
   set_death_location(copy_from->get_death_location());
   set_death_text(copy_from->get_death_text());

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Race::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Race::get_mem_size_dynamic()
{
   int  size = 0;
   Tutorial *tmp_tutorial;
 
   size += description.get_mem_size_dynamic();   
   size += init_desc.get_mem_size_dynamic();   
   size += init_brief.get_mem_size_dynamic();   
   size += init_ability.get_mem_size_dynamic();   
   size += init_location.get_mem_size_dynamic();
   size += death_location.get_mem_size_dynamic();
   size += death_text.get_mem_size_dynamic();

   tmp_tutorial = tutorial_list;
   while (tmp_tutorial != NULL)
   {
      size += tmp_tutorial->get_mem_size();
      tmp_tutorial = tmp_tutorial->next_tutorial;
   }

   size += get_mem_size_entity();

   return size;
}


/***********************************************************************
 ** Tutorial (constructor) - creates the tutorial object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
Tutorial::Tutorial(char *the_name)
{
   name = the_name;
   next_tutorial = NULL;
}


/***********************************************************************
 ** Tutorial (destructor) - destroys this tutorial
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
Tutorial::~Tutorial()
{

}

/***********************************************************************
 ** get_mem_size - gets how much memory this tutorial is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Tutorial::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Tutorial::get_mem_size_dynamic()
{
   int size = 0;

   size += prompt.get_mem_size_dynamic();
   size += tutorial_special.get_mem_size_dynamic();

   return size;
}

#endif
