/**********************************************************************
 ** Rope class:  Has attributes and methods for a rope object
 **
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef ROPE_C
#define ROPE_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "rope.h"
#include "objtype.h"
#include "flags.h"
#include "player.h"

/***********************************************************************
 ** Rope (constructor)
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Rope::Rope(void)
{

}


/***********************************************************************
 ** Rope (constructor) - creates the item
 **
 ** Parameters: the_name - the name of the marker
 **             the_area - the area the marker belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Rope::Rope(char *the_name, char *the_area)
{
   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_ROPE;
   
   if (the_name != NULL)
      set_name(the_name);
   
   if (the_area != NULL)
      set_area(the_area);

   length = size = weight = 0; 
}

/***********************************************************************
 ** ~Rope (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Rope::~Rope()
{
}


/***********************************************************************
 ** write_object - writes the location to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Rope::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nrope %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());
   write_mudobject_attrib(the_file);
   write_item_attrib(the_file);
   write_moveable_attrib(the_file);

   fprintf(the_file, "%d\n", get_rope_len());
}

/***********************************************************************
 ** describe - describes the rope to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Rope::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GRope: \t\t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_builder->send_bldr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_builder->send_bldr("&+GClones: \t&+g%s&*\n", get_clones());
   the_builder->send_bldr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_builder->send_bldr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_builder->send_bldr("&+GLocation: \t&+M%s&*\n", get_location());
   the_builder->send_bldr("&+GSize: \t\t&+w%d&*\n", get_size());
   the_builder->send_bldr("&+GWeight:\t\t&+w%d&*\n", get_weight());
   if (itemflags->get_flag(ITEMFLAG_CONTAINER))
      the_builder->send_bldr("&+GCapacity:\t&+w%d&*\n", get_capacity());
   the_builder->send_bldr("&+GLength:\t\t&+w%d&*\n", get_rope_len());
   the_builder->send_bldr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_builder->send_bldr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_builder->send_bldr("&+GDesc:&*\n%s\n", get_desc());

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the rope to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Rope::describe(Player *the_player)
{
   MudObject *tmp_container;
   Strings   container_name;

   the_player->send_plr("\n&+GRope: \t\t&+M%s&*\n", get_name());
   the_player->send_plr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_player->send_plr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_player->send_plr("&+GClones: \t&+g%s&*\n", get_clones());
   the_player->send_plr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_player->send_plr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_player->send_plr("&+GStartLoc: \t&+M%s&*\n", get_location());

   tmp_container = get_contained_by();
   if (tmp_container == NULL)
      container_name = "nowhere";
   else
      container_name.sprintf("%s@%s", tmp_container->get_name(), 
                                         tmp_container->get_area());

   the_player->send_plr("&+GCurrentLoc: \t&+M%s&*\n", 
                                                 container_name.str_show());

   the_player->send_plr("&+GSize: \t\t&+w%d&*\n", get_size());
   the_player->send_plr("&+GWeight:\t\t&+w%d&*\n", get_weight());
   if (itemflags->get_flag(ITEMFLAG_CONTAINER))
      the_player->send_plr("&+GCapacity:\t&+w%d&*\n", get_capacity());
   the_player->send_plr("&+GLength:\t\t&+w%d&*\n", get_rope_len());
   the_player->send_plr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_player->send_plr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_player->send_plr("&+GDesc:&*\n%s\n", get_desc());
   the_player->send_plr("&+YSize: \t\t\t&+W%d&*\n", get_mem_size());

   list_specials(the_player);

   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Rope::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
            send_bldr("You can set the following attributes on a rope.\n"
               "   title, weight, size, altnames, desc, brief0, brief1,\n"
               "   guards, itemflags, location, specials, merger, length\n"
               "   and capacity\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "title",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_title(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "weight",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_weight(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "size",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_size(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "altnames",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_altnames(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "clones",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_clones(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "desc", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_desc(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "brief0", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 0);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "itemflags",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_itemflags(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "guards",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_guard(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "specials", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "brief1", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 1);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "location", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_loc(the_parsed, the_builder);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "length",
                               strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as length.\n");
         return -1;
      }

      set_rope_len(atoi(the_parsed->get_speech()));
      the_builder->send_bldr("Length set to %d on rope object %s.\n",
                                          get_rope_len(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "capacity", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_capacity(the_parsed, the_builder);
   }

   the_builder->send_bldr("The attribute '%s' is not a rope attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** copy_object - copies the rope to an object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Rope::copy_object(Entity *copy_obj)
{
   Rope *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_ROPE)
      return 0;

   copy_from = (Rope *) copy_obj;

   /******* set the mudobject attributes *****/
   copy_mudobject_attrib((MudObject *) copy_from);

   /****** set the item attributes ******/
   copy_item_attrib((Item *) copy_from);

   /****** set the moveable attributes ******/
   copy_moveable_attrib((Moveable *) copy_from);

   /****** set up rope attributes ******/
   set_rope_len(copy_from->get_rope_len());

   return 1;
}


/***********************************************************************
 ** set_rope_len - sets the rope length for the rope
 **
 ** Parameters: new_len - the new length of the rope
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/

int Rope::set_rope_len(int new_len)
{
   if (new_len <= 0)
      return -1;

   length = new_len;
   return 1;
}


/***********************************************************************
 ** get_rope_len - gets the length of the rope
 **
 ** Parameters: Nothing
 **
 ** Returns:  The rope length
 **
 ***********************************************************************/

int Rope::get_rope_len(void)
{
   return length;
}



/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Rope *Rope::operator = (Rope *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}



/***********************************************************************
 ** read_rope_attrib - reads in rope attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Rope::read_rope_attrib(FILE *read_file, ErrLog *error_log)
{
   token_record *the_token;
   Strings      holder;

   
   /* Set rope length */
   the_token = get_token(read_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute length in rope %s", 
                                                            get_name());
      error_log->log_err(holder.str_show(), "read_rope_attrib");
      return -1;
   }
   set_rope_len(atoi(the_token->the_string));

   return 1;
}

/***********************************************************************
 ** set_tied - sets that it is tied to something
 **
 ** Parameters: None
 **
 ***********************************************************************/

void Rope::set_tied()
{
   state = 0;
   itemflags->set_flag(ITEMFLAG_ISTIED);
}

/***********************************************************************
 ** set_untied - sets that it is untied
 **
 ** Parameters: None
 **
 ***********************************************************************/

void Rope::set_untied()
{
   state = 1;
   itemflags->clr_flag(ITEMFLAG_ISTIED);
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Rope::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Rope::get_mem_size_dynamic()
{
   int  size = 0;

   size += get_mem_size_moveable();
   size += get_mem_size_item();
   size += get_mem_size_mudobj();
   size += get_mem_size_entity();

   return size;
}

#endif





