/**********************************************************************
 **
 ** Quest - contains methods and attributes for a quest object
 **
 ** Reviewed through:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/


#ifndef QUEST_C
#define QUEST_C

#include "sysdep.h"
#include "strings.h"
#include "quest.h"
#include "lexer.h"
#include "builder.h"
#include "parse.h"
#include "utils.h"
#include "player.h"
#include "objtype.h"
#include "config.h"
#include "newfuncts.h"


char *diff_name[] = {"Easy", "Medium", "Hard"};


/***********************************************************************
 ** Quest (constructor) - loads quest name and attributes
 **
 ** Parameters: quest_name - this quest name 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Quest::Quest(char *quest_name)
{
   Strings tmp_name;

   obj_type = OBJ_TYPE_QUEST;
   tmp_name.assign_word(quest_name, 1);

   set_name(tmp_name.str_show());

   modified = 0;
   status = NotStarted;
   difficulty = Easy;
   puzzles = combat = qpoints = 1;
   required = 0;
   set_description("No desc\n");
}


/***********************************************************************
 ** ~Quest (destructor) - cleans up the quest
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Quest::~Quest(void)
{
}


/***********************************************************************
 ** set_description - sets the description string for this quest
 **
 ** Parameters: the_string - the string we are setting it to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Quest::set_description(char *the_string)
{
   if (the_string == NULL)
      return -1;

   description = the_string;
   return 1;
}

/***********************************************************************
 ** get_description - gets the description string for this quest
 **
 ** Parameters: None
 **
 ** Returns: pointer to the description string
 **
 ***********************************************************************/

char *Quest::get_description(void)
{
   return description.str_show();
}


/***********************************************************************
 ** set_difficulty - sets the difficulty string for this quest
 **
 ** Parameters: new_diff - the new difficulty value
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Quest::set_difficulty(diff_type new_diff)
{
   difficulty = new_diff;
}


/***********************************************************************
 ** get_difficulty - gets the difficulty for this quest
 **
 ** Parameters: None
 **
 ** Returns: pointer to the difficulty string
 **
 ***********************************************************************/

diff_type Quest::get_difficulty(void)
{
   return difficulty;
}


/***********************************************************************
 ** set_puzzles - sets the toughness of the puzzles in the quest
 **
 ** Parameters: new_val - the new value to set to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Quest::set_puzzles(int new_val)
{
   if ((new_val < 0) || (new_val > 5))
      return;

   puzzles = new_val;
}


/***********************************************************************
 ** get_puzzles - gets the toughness of the puzzles
 **
 ** Parameters: None
 **
 ** Returns: puzzle value
 **
 ***********************************************************************/

int Quest::get_puzzles(void)
{
   return puzzles;
}


/***********************************************************************
 ** set_combat - sets the toughness of the combat in the quest
 **
 ** Parameters: new_val - the new value to set to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Quest::set_combat(int new_val)
{
   if ((new_val < 0) || (new_val > 5))
      return;

   combat = new_val;
}


/***********************************************************************
 ** get_combat - gets the toughness of the combat
 **
 ** Parameters: None
 **
 ** Returns: puzzle value
 **
 ***********************************************************************/

int Quest::get_combat(void)
{
   return combat;
}



/***********************************************************************
 ** set_qpoints - sets the number of questpoints this quest offers
 **
 ** Parameters: new_val - the new value to set to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Quest::set_qpoints(int new_val)
{
   if (new_val < 0)
      return;

   qpoints = new_val;
}


/***********************************************************************
 ** get_qpoints - gets the number of questpoints for this quest
 **
 ** Parameters: None
 **
 ** Returns: puzzle value
 **
 ***********************************************************************/

int Quest::get_qpoints(void)
{
   return qpoints;
}


/***********************************************************************
 ** set_required - sets if this quest is required (1) or not (0)
 **
 ** Parameters: new_val - the new value to set to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Quest::set_required(int new_val)
{
   if ((new_val < 0) || (new_val > 1))
      return;

   required = new_val;
}


/***********************************************************************
 ** get_required - gets if this is required or not
 **
 ** Parameters: None
 **
 ** Returns: puzzle value
 **
 ***********************************************************************/

int Quest::get_required(void)
{
   return required;
}


/***********************************************************************
 ** load_quest - loads a quest from a file into memory
 **
 ** Parameters: the_file - where we are getting the action from
 **
 ** Returns: pointer to the next quest
 **
 ***********************************************************************/

int Quest::load_quest(FILE *the_file, ErrLog *error_log, int is_builder)
{
   token_record *the_token;
   char         *tmp_charholder;
   Strings      holder;

   if (is_builder)
   {
       the_token = get_token(the_file,'\0');
 
       if (the_token->token_type != T_NUMERICAL)
       {
           error_log->log_err("Invalid format in quest file", "load_quest");
           return -1;
       }
       set_modified(atoi(the_token->the_string));
   }

   /* get the description */
   tmp_charholder = read_desc_type(the_file, error_log, NULL); 
   set_description(tmp_charholder);
   delete tmp_charholder;


   /* Set difficulty value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute difficulty for quest %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_quest");
      return -1;
   }
   set_difficulty((diff_type) atoi(the_token->the_string));

   /* Set puzzles value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute puzzles for quest %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_quest");
      return -1;
   }
   set_puzzles(atoi(the_token->the_string));

   /* Set combat value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute combat for quest %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_quest");
      return -1;
   }
   set_combat(atoi(the_token->the_string));

   /* Set qpoints value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute qpoints for quest %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_quest");
      return -1;
   }
   set_qpoints(atoi(the_token->the_string));

   /* Set 'required' value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute required for quest %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_quest");
      return -1;
   }
   set_required(atoi(the_token->the_string));

   return 1;
}


/***********************************************************************
 ** describe - describes the quest to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Quest::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GQuest: \t\t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GDifficulty: \t&+W%s&*\n", 
                                       diff_name[(int) get_difficulty()]);
   the_builder->send_bldr("&+GRequired: \t&+G%s&*\n", (get_required()) ? 
			  "Yes" : "No");
   the_builder->send_bldr("&+GPuzzles: \t&+W%d\n", get_puzzles());
   the_builder->send_bldr("&+GCombat: \t&+W%d\n", get_combat());
   the_builder->send_bldr("&+GQPoints: \t&+Y%d\n", get_qpoints());
   the_builder->send_bldr("&+GDesc: \n&+w%s&*\n", get_description());
   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the quest to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Quest::describe(Player *the_player)
{
   the_player->send_plr("\n&+GQuest: \t\t&+M%s&*\n", get_name());
   the_player->send_plr("&+GDifficulty: \t&+W%s&*\n", 
                                       diff_name[(int) get_difficulty()]);
   the_player->send_plr("&+GRequired: \t&+G%s&*\n", (get_required()) ? 
			  "Yes" : "No");
   the_player->send_plr("&+GPuzzles: \t&+W%d\n", get_puzzles());
   the_player->send_plr("&+GCombat: \t&+W%d\n", get_combat());
   the_player->send_plr("&+GQPoints: \t&+Y%d\n", get_qpoints());
   the_player->send_plr("&+GDesc: \n&+w%s&*\n", get_description());
   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Quest::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on a quest.\n"
               "   difficulty and desc\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "desc",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&description) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "difficulty",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;      
      int     i;

      if ((the_parsed->get_speech() == NULL) || 
          (strlen(the_parsed->get_speech()) <= 0))
      {
         the_builder->
              send_bldr("You need to specify a difficulty to set to.\n"
                        "Either Easy, Medium, or Hard\n");
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);
      for (i=0; i<3; i++)
      {
         if (!STRNCASECMP(holder.str_show(), diff_name[i], holder.str_len()))
	 {
            set_difficulty((diff_type) i);
            break;
         }
      }

      if (i==3)
      {
         the_builder->send_bldr("Invalid difficulty.\n"
                                "Valid are: Easy, Medium, and Hard\n");
         return -1;
      }

      the_builder->send_bldr("Difficulty on %s set to: %s\n", get_name(), 
                                        diff_name[(int) get_difficulty()]);
      set_modified(1);
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "puzzles",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as puzzles.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if ((value < 0) || (value > 5))
      {
         the_builder->send_bldr("Value must be between 0 and 5.\n");
         return -1;
      }

      set_puzzles(value);
      the_builder->send_bldr("Puzzles set to %d on quest object %s.\n",
                                          get_puzzles(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "combat",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as combat.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if ((value < 0) || (value > 5))
      {
         the_builder->send_bldr("Value must be between 0 and 5.\n");
         return -1;
      }

      set_combat(value);
      the_builder->send_bldr("Combat set to %d on quest object %s.\n",
                                          get_combat(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "qpoints",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as qpoints.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if ((value < 0) || (value > 5))
      {
         the_builder->send_bldr("Value must be between 0 and 5.\n");
         return -1;
      }

      set_qpoints(value);
      the_builder->send_bldr("QPoints set to %d on quest object %s.\n",
                                          get_qpoints(), get_name());
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "required",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;      

      if ((the_parsed->get_speech() == NULL) || 
          (strlen(the_parsed->get_speech()) <= 0))
      {
         the_builder->
              send_bldr("You need to specify Yes or No for required\n");
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);

      if (!STRCASECMP(holder.str_show(), "yes"))
      {
         set_required(1);
      }
      else if (!STRCASECMP(holder.str_show(), "no"))
      {
         set_required(0);
      }
      else
      {
         the_builder->
              send_bldr("You need to specify Yes or No for required\n");
         return -1;
      }
      the_builder->send_bldr("Required on %s set to: %s\n", get_name(), 
                                       get_required() ? "Yes" : "No");
      set_modified(1);
      return 1;
   }
   the_builder->send_bldr("The attribute '%s' is not a quest attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the quest to a specified file in specified
 **                format
 **
 ** Parameters: the_file - the file to write to
 **             build_format - shall we use builder format or not
 **
 ***********************************************************************/
   
void Quest::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nquest %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   fprintf(the_file, "^%s^\n", (get_description() == NULL) ? "" : 
                                                        get_description());
   fprintf(the_file, "%d\n", get_difficulty());
   fprintf(the_file, "%d\n", get_puzzles());
   fprintf(the_file, "%d\n", get_combat());
   fprintf(the_file, "%d\n", get_qpoints());
   fprintf(the_file, "%d\n", get_required());

   return;
}


/***********************************************************************
 ** is_modified - has this quest been modified?
 **
 ** Parameters: None
 **
 ** Returns:  1 for yes, 0 for no
 **
 ***********************************************************************/
   
int Quest::is_modified(void)
{
   return modified;
}



/***********************************************************************
 ** set_modified - shall we set this?
 **
 ** Parameters: the_num - the number to set it to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
void Quest::set_modified(int the_num)
{
   modified = the_num;
}


/***********************************************************************
 ** copy_object - copies the object to an quest of a different name
 **
 ** Parameters: copy_from - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Quest::copy_object(Entity *copy_obj)
{
   Quest *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_QUEST)
      return 0;

   copy_from = (Quest *) copy_obj;

   /******* set the action attributes *****/
   set_difficulty(copy_from->get_difficulty());
   set_puzzles(copy_from->get_puzzles());
   set_combat(copy_from->get_combat());
   set_qpoints(copy_from->get_qpoints());
   set_description(copy_from->get_description());
   set_required(copy_from->get_required());

   return 1;
}



/***********************************************************************
 ** display_qinfo - describes the quest to a player in qinfo format
 **
 ** Parameters: the_player - who we describe this object to
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/

int Quest::display_qinfo(Player *the_player)
{
   Strings blueline;
   Strings stars1;
   Strings stars2;
   int     i;
   int     counter = 0;


   blueline = "====================================================================";

   blueline.truncate(68 - strlen(get_name()));
   the_player->send_plr(
    "\n&+B==&+W[%s Quest]&+B%s&*\n\n", get_name(), blueline.str_show());
   the_player->send_plr("%s\n", get_description());


   stars1.truncate(0);
   counter = puzzles;
   for (i=0; i<5; i++)
   {
      if (counter != 0)
      {
         stars1.str_cat("*");
         counter--;
      }
      else
         stars1.str_cat(" ");
   }

   stars2.truncate(0);
   counter = combat;
   for (i=0; i<5; i++)
   {
      if (counter != 0)
      {
         stars2.str_cat("*");
         counter--;
      }
      else
         stars2.str_cat(" ");
   }

   the_player->send_plr("&+cQPoints: &+W%d\n\n", get_qpoints());
 
   the_player->send_plr("&+cPuzzles: &+R[&+W%s&+R]\t\t"
                        "&+cCombat: &+R[&+W%s&+R]&*\n\n", stars1.str_show(), 
                                                          stars2.str_show()); 
   the_player->send_plr("&+B==============================================================================&*\n");
   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Quest::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Quest::get_mem_size_dynamic()
{
   int  size = 0;

   size += description.get_mem_size_dynamic();
  
   size += get_mem_size_entity();

   return size;
}


#endif
