// ---------------------------------------------------------------------------
// - Closure.hpp                                                             -
// - aleph engine - closure class implementation                             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Stack.hpp"
#include "Vector.hpp"
#include "Return.hpp"
#include "Boolean.hpp"
#include "Lexical.hpp"
#include "Closure.hpp"
#include "Runnable.hpp"
#include "Reserved.hpp"
#include "Argument.hpp"
#include "Exception.hpp"

namespace aleph {
  // the supported closure quarks
  static const long QUARK_SELF    = String::intern ("self");
  static const long QUARK_ARGS    = String::intern ("args");
  static const long QUARK_CONST   = String::intern ("const");
  static const long QUARK_LAMBDAP = String::intern ("lambda-p");
  static const long QUARK_SETFORM = String::intern ("set-form");
  static const long QUARK_GETFORM = String::intern ("get-form");
  static const long QUARK_ADDARG  = String::intern ("add-argument");
  
  // create a default closure
  
  Closure::Closure (void) {
    d_lflag = true;
    p_form  = nilp;
    d_argc  = 0;
    Object::iref (p_lset = new Localset);
  }

  // create a default closure with a type
  
  Closure::Closure (const bool type) {
    d_lflag = type;
    d_argc  = 0;
    p_form  = nilp;
    Object::iref (p_lset = new Localset);
  }

  // create a new closure 

  Closure::Closure (const bool type, Cons* argl, Object* form) {
    // save the arguments
    d_lflag = type;
    d_argc  = 0;
    Object::iref (p_form = form);
    Object::iref (p_lset = new Localset);
    // add the arguments
    try {
      while (argl != nilp) {
	addarg (argl->getcar ());
	argl = argl->getcdr ();
      }
    } catch (...) {
      Object::dref (p_lset);
      throw;
    }
  }

  // destroy this closure

  Closure::~Closure (void) {
    Object::dref (p_form);
    Object::dref (p_lset);
  }

  // return the class name

  String Closure::repr (void) const {
    return "Closure";
  }

  // make this closure a shared object

  void Closure::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_lset != nilp) p_lset->mksho ();
    if (p_form != nilp) p_form->mksho ();
  }

  // add an argument by name to this closure

  void Closure::addarg (const long quark, const bool flag) {
    wrlock ();
    // check if the argument already exists
    if (p_lset->exists (quark) == true) {
      unlock ();
      throw Exception ("argument-error", "duplicate argument name", 
		       String::qmap (quark));
    }
    // create a new argument and bind it
    try {
      Argument* arg = new Argument (quark, d_argc++);
      arg->setconst (flag);
      p_lset->bind (quark,arg);
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add an argument by object - we can either add a lexical name, a
  // form with a const flag or a string

  void Closure::addarg (Object* object) {
    String* sobj = dynamic_cast <String*> (object);
    if (sobj != nilp) {
      addarg (sobj->toquark (), false);
      return;
    }
    Lexical* lex = dynamic_cast <Lexical*> (object);
    if (lex != nilp) {
      addarg (lex->toquark (), false);
      return;
    }
    Cons* form = dynamic_cast <Cons*> (object);
    if (form != nilp) {
      if (form->length () != 2)
	throw Exception ("argument-error", "invalid argument list form");
      Reserved* crsv = dynamic_cast <Reserved*> (form->getcar  ());
      Lexical*  alex = dynamic_cast <Lexical*>  (form->getcadr ());
      if ((crsv == nilp) || (alex == nilp))
	throw Exception ("argument-error", "invalid argument list form");
      if (crsv->toquark () != QUARK_CONST)
	throw Exception ("argument-error", "invalid argument list form");
      addarg (alex->toquark (), true);
      return;
    }
    throw Exception ("argument-error", "invalid argument list form");
  }

  // add a closed variable to this closure

  void Closure::addclv (const long quark, Object* object) {
    wrlock ();
    // check if the argument already exists
    if (p_lset->exists (quark) == true) {
      unlock ();
      throw Exception ("argument-error", "duplicate argument name", 
		       String::qmap (quark));
    }
    try {
      p_lset->bind (quark, object);
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set this closure form

  void Closure::setform (Object* form) {
    wrlock ();
    Object::iref (form);
    Object::dref (p_form);
    p_form = form;
    unlock ();
  }

  // return true if the closure is a lambda expression

  bool Closure::islambda (void) const {
    rdlock ();
    bool result = d_lflag;
    unlock ();
    return result;
  }

  // create a closure in a generic way

  Object* Closure::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new Closure;

    // check for 1 argument
    if (argc == 1) {
      bool type = argv->getbool (0);
      return new Closure (type);
    }
    // illegal arguments
    throw Exception ("argument-error", "too many arguments with closure");
  }

  // set this closure as a const object

  Object* Closure::cdef (Runnable* robj, Nameset* nset, const long quark,
			 Object* object) {
    // get thr write lock
    wrlock ();
    // define constant
    try {
      Object* result = p_lset->cdef (robj, nset, quark, object);
      robj->post (result);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set this closure closed object

  Object* Closure::vdef (Runnable* robj, Nameset* nset, const long quark,
			 Object* object) {
    // get thr write lock
    wrlock ();
    // define constant
    try {
      Object* result = p_lset->vdef (robj, nset, quark, object);
      robj->post (result);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // evaluate a closure closed object

  Object* Closure::eval (Runnable* robj, Nameset* nset, const long quark) {
    // get thr write lock
    rdlock ();
    // define constant
    try {
      Object* result = nilp;
      if (p_lset->exists (quark) == true) {
	result = p_lset->eval (robj, nset, quark);
      } else {
	result = Object::eval (robj, nset, quark);
      }
      robj->post (result);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // apply this closure with a set of arguments

  Object* Closure::apply (Runnable* robj, Nameset* nset, Cons* args) {
    // get the stack context
    Stack*   stk = robj->getstk ();
    Object** sp  = stk->getsp ();
    Object** fp  = stk->getfp ();
    // evaluate the arguments on the stack
    long argc = 0;
    while (args != nilp) {
      if (argc++ == d_argc) break;
      Object* car = args->getcar ();
      stk->push ((car == nilp) ? nilp : car->eval (robj, nset));
      args = args->getcdr ();
    }

    // set the parent nameset
    Nameset* lset = new Localset (p_lset);
    if (d_lflag == true) 
      lset->setparent (nset);
    else
      lset->setparent (robj->getgset ());

    // bind the self argument
    lset->symcst (QUARK_SELF, this);

    // check for the remaining arguments
    if (args != nilp) {
      Cons* larg = nilp;
      try {
	while (args != nilp) {
	  Object* car = args->getcar ();
	  Object* arg = (car == nilp) ? nilp : car->eval (robj, nset);
	  if (larg == nilp) {
	    larg = new Cons (arg);
	  } else {
	    larg->append (arg);
	  }
	  args = args->getcdr ();
	}
      } catch (...) {
	delete larg;
	lset->clear ();
	delete lset;
	throw;
      }
      lset->symcst (QUARK_ARGS, larg);
    } else {
      lset->symcst (QUARK_ARGS, (Object*) nilp);
    }

    // execute this closure - catch any exception so we unlink and destroy
    // the local set. The stack is unwind after the execution. Note how the
    // stack frame is created and restored. If you have a better idea - well
    // let me know.
    Object*  result = nilp;
    try {
      stk->setfp (sp);
      result = Object::iref (p_form->eval (robj,lset));
      stk->unwind (sp, fp);
      lset->clear ();
      delete lset;
    } catch (const Return& r) {
      result = Object::iref (r.getobj ());
      stk->unwind (sp, fp);
      lset->clear ();
      delete lset;
    } catch (...) {
      stk->unwind (sp, fp);
      lset->clear ();
      delete lset;
      throw;
    }
    Object::tref (result);
    return result;
  }

  // apply this closure with a set of arguments and a quark

  Object* Closure::apply (Runnable* robj, Nameset* nset, const long quark,
			  Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LAMBDAP) return new Boolean (islambda ());
      if (quark == QUARK_GETFORM) {
	rdlock ();
	Object* result = p_form;
	robj->post (result);
	unlock ();
	return result;
      }
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETFORM) {
	Object* form = argv->get (0);
	setform (form);
	return nilp;
      }
      if (quark == QUARK_ADDARG) {
	Object* arg = argv->get (0);
	addarg (arg);
	return nilp;
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
