// ---------------------------------------------------------------------------
// - Multicast.cpp                                                           -
// - aleph:net library - multicast udp client socket implementation          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Multicast.hpp"
#include "Exception.hpp"

namespace aleph {

  // create a mct client by host and port

  Multicast::Multicast (const String& host, t_word port) {
    // save the address and port
    d_addr = Address (host);
    d_port = port;
    // bind this socket
    if (bind (d_port, d_addr) == false)
      throw Exception ("multicast-error", "cannot bind socket");
    // join the multicast group
    if (join (d_addr) == false)
      throw Exception ("multicast-error", "cannot join multicast group");
  }

  // create a mct client by ip address and port

  Multicast::Multicast (const Address& addr, t_word port) {
    // save the address and port
    d_addr = addr;
    d_port = port;
    // bind this socket
    if (bind (d_port, d_addr) == false)
      throw Exception ("multicast-error", "cannot bind socket");
    // join the multicast group
    if (join (d_addr) == false)
      throw Exception ("multicast-error", "cannot join multicast group");
  }

  // destroy this multicast socket

  Multicast::~Multicast (void) {
    close ();
  }

  // return the class name

  String Multicast::repr (void) const {
    return "Multicast";
  }

  // close this multicast socket

  bool Multicast::close (void) {
    wrlock ();
    if ((d_sid == -1) || (Object::uref (this) == false)) {
      unlock ();
      return true;
    }
    // drop membership and close
    drop (d_addr);
    bool result = Socket::close ();
    unlock ();
    return result;
  }

  // create a new multicast client in a generic way

  Object* Multicast::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc != 2) 
      throw Exception ("argument-error", 
                       "invalid arguments with with multicast client"); 
    // get the mct client arguments
    Object* obj = argv->get (0);
    if (dynamic_cast <String*> (obj) != nilp) {
      String host = argv->getstring (0);
      long   port = argv->getint (1);
      return new Multicast (host, port);
    }
    Address* addr = dynamic_cast <Address*> (obj);
    if (addr != nilp) {
      long   port = argv->getint (1);
      return new Multicast (*addr, port);
    }
    throw Exception ("argument-error", "invalid object with multicast client",
		     Object::repr (obj));
  }
}
