/*
Copyright (c) 2000, The JAP-Team 
All rights reserved.
Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

	- Redistributions of source code must retain the above copyright notice, 
	  this list of conditions and the following disclaimer.

	- Redistributions in binary form must reproduce the above copyright notice, 
	  this list of conditions and the following disclaimer in the documentation and/or 
		other materials provided with the distribution.

	- Neither the name of the University of Technology Dresden, Germany nor the names of its contributors 
	  may be used to endorse or promote products derived from this software without specific 
		prior written permission. 

	
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS'' AND ANY EXPRESS 
OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY 
AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, 
OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER 
IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY 
OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
*/
#include "StdAfx.h"
#include "CAQueue.hpp"
#include "CAMsg.hpp"
#include "CAUtil.hpp"
#include "CAThread.hpp"

#ifdef DO_TRACE
UINT32 CAQueue::m_aktAlloc=0;
UINT32 CAQueue::m_maxAlloc=0;
#endif
/** Deletes this Queue and all stored data*/
CAQueue::~CAQueue()
	{
		m_csQueue.lock();
#ifdef DO_TRACE
		CAMsg::printMsg(LOG_DEBUG,"CAQueue deleting - current alloc: %u Current Size (of this[%p] queue) %u\n",m_aktAlloc,this,m_nQueueSize);
#endif
		
		while(m_Queue!=NULL)
			{
#ifndef DO_TRACE
				delete m_Queue->pBuff;
#else
				deleteUINT8Buff(m_Queue->pBuff,m_Queue->shadow_pBuff,m_Queue->allocSize);
#endif
				m_lastElem=m_Queue;
				m_Queue=m_Queue->next;
#ifndef DO_TRACE
				delete m_lastElem;
#else
				deleteQUEUE(m_lastElem);
#endif
			}
#ifdef DO_TRACE
		CAMsg::printMsg(LOG_DEBUG,"CAQueue deleted QUEUE [%p] Current alloc now: %u\n",this,m_aktAlloc);
#endif
		m_csQueue.unlock();
	}

/** Adds data to the Queue.
	* @param buff pointer to the data buffer
	* @param size size of data to add
	* @retval E_UNKNOWN in case of an error
	* @retval E_SUCCESS if succesful
	*/
SINT32 CAQueue::add(const void* buff,UINT32 size)
	{
		if(size==0)
			return E_SUCCESS;
		if(buff==NULL)
			return E_UNKNOWN;
		m_csQueue.lock();
		if(m_Queue==NULL)
			{
#ifndef DO_TRACE
				m_Queue=new QUEUE;
#else
				m_Queue=newQUEUE();
#endif
				if(m_Queue==NULL)
					{
						m_csQueue.unlock();
						return E_UNKNOWN;
					}
#ifndef DO_TRACE
				m_Queue->pBuff=new UINT8[size];
#else
				m_Queue->pBuff=newUINT8Buff(size);
				m_Queue->allocSize=size;
				m_Queue->shadow_pBuff=m_Queue->pBuff;
#endif
				if(m_Queue->pBuff==NULL)
					{
#ifndef DO_TRACE
						delete m_Queue;
#else
						deleteQUEUE(m_Queue);
#endif
						m_Queue=NULL;
						m_csQueue.unlock();
						return E_UNKNOWN;
					}
				m_Queue->next=NULL;
				m_Queue->size=size;
				memcpy(m_Queue->pBuff,buff,size);
				m_lastElem=m_Queue;
			}
		else
			{
#ifndef DO_TRACE
				m_lastElem->next=new QUEUE;
#else
				m_lastElem->next=newQUEUE();
#endif
				if(m_lastElem->next==NULL)
					{
						m_csQueue.unlock();
						return E_UNKNOWN;
					}
#ifndef DO_TRACE
				m_lastElem->next->pBuff=new UINT8[size];
#else
				m_lastElem->next->pBuff=newUINT8Buff(size);
				m_lastElem->next->allocSize=size;
				m_lastElem->next->shadow_pBuff=m_lastElem->next->pBuff;
#endif
				if(m_lastElem->next->pBuff==NULL)
					{
#ifndef DO_TRACE
						delete m_lastElem->next;
#else
						deleteQUEUE(m_lastElem->next);
#endif
						m_lastElem->next=NULL;
						m_csQueue.unlock();
						return E_UNKNOWN;
					}
				m_lastElem=m_lastElem->next;
				m_lastElem->next=NULL;
				m_lastElem->size=size;
				memcpy(m_lastElem->pBuff,buff,size);
			}
		m_nQueueSize+=size;
	//	m_convarSize.unlock();
		m_csQueue.unlock();
		m_convarSize.signal();
		return E_SUCCESS;
	}

/** Gets data from the Queue. The data is removed from the Queue
  * @param pbuff pointer to a buffer, there the data should be stored
	* @param psize on call contains the size of pbuff, on return contains 
	*								the size of returned data
	* @retval E_SUCCESS if succesful
	* @retval E_UNKNOWN in case of an error
	*/
SINT32 CAQueue::get(UINT8* pbuff,UINT32* psize)
	{
		if(pbuff==NULL||psize==NULL)
			return E_UNKNOWN;
		if(*psize==0)
			return E_SUCCESS;
		if(m_Queue==NULL)
			{
				*psize=0;
				return E_SUCCESS;
			}
		m_csQueue.lock();
		UINT32 space=*psize;
		*psize=0;
		while(space>=m_Queue->size)
			{
				memcpy(pbuff,m_Queue->pBuff,m_Queue->size);
				*psize+=m_Queue->size;
				pbuff+=m_Queue->size;
				space-=m_Queue->size;
				m_nQueueSize-=m_Queue->size;
#ifndef DO_TRACE
				delete []m_Queue->pBuff;
#else
				deleteUINT8Buff(m_Queue->pBuff,m_Queue->shadow_pBuff,m_Queue->allocSize);
#endif
				QUEUE* tmp=m_Queue;
				m_Queue=m_Queue->next;
#ifndef DO_TRACE
				delete tmp;
#else
				deleteQUEUE(tmp);
#endif
				if(m_Queue==NULL)
					{
						m_csQueue.unlock();
						return E_SUCCESS;
					}
			}
		if(space>0)
			{
				memcpy(pbuff,m_Queue->pBuff,space);
				*psize+=space;
				m_Queue->size-=space;
				m_nQueueSize-=space;
				memmove(m_Queue->pBuff,m_Queue->pBuff+space,m_Queue->size);
			}
		m_csQueue.unlock();
		return E_SUCCESS;
	}

/** Gets data from the Queue or waits until some data is available, if the Queue is empty.
	* The data is removed from the Queue
  * @param pbuff pointer to a buffer, there the data should be stored
	* @param psize on call contains the size of pbuff, on return contains 
	*								the size of returned data
	* @retval E_SUCCESS if succesful
	* @retval E_UNKNOWN in case of an error
	*/
SINT32 CAQueue::getOrWait(UINT8* pbuff,UINT32* psize)
	{
		m_convarSize.lock();
		while(m_Queue==NULL)
			m_convarSize.wait();
		SINT32 ret=get(pbuff,psize);
		m_convarSize.unlock();
		return ret;
	}

/** Gets data from the Queue or waits until some data is available, 
	* if the Queue is empty or a timeout is reached.
	* The data is removed from the Queue.
  * @param pbuff pointer to a buffer, there the data should be stored
	* @param psize on call contains the size of pbuff, on return contains 
	*								the size of returned data
	* @param msTimeout timeout in milli seconds
	* @retval E_SUCCESS if succesful
	* @retval E_IMEDOUT if timeout was reached
	* @retval E_UNKNOWN in case of an error
	*/
SINT32 CAQueue::getOrWait(UINT8* pbuff,UINT32* psize,UINT32 msTimeout)
	{
		m_convarSize.lock();
		SINT32 ret;
		while(m_Queue==NULL)
			{
				ret=m_convarSize.wait(msTimeout);
				if(ret==E_TIMEDOUT)
					{
						m_convarSize.unlock();
						return E_TIMEDOUT;
					}
			}		
		ret=get(pbuff,psize);
		m_convarSize.unlock();
		return ret;
	}

	/** Peeks data from the Queue. The data is NOT removed from the Queue.
  * @param pbuff pointer to a buffer, where the data should be stored
	* @param psize on call contains the size of pbuff, 
	*							 on return contains the size of returned data
	* @retval E_SUCCESS if succesful
	* @retval E_UNKNOWN in case of an error
	*/
SINT32 CAQueue::peek(UINT8* pbuff,UINT32* psize)
	{
		if(m_Queue==NULL||pbuff==NULL||psize==NULL)
			return E_UNKNOWN;
		if(*psize==0)
			return E_SUCCESS;
		m_csQueue.lock();
		UINT32 space=*psize;
		*psize=0;
		QUEUE* tmpQueue=m_Queue;
		while(space>=tmpQueue->size)
			{
				memcpy(pbuff,tmpQueue->pBuff,tmpQueue->size);
				*psize+=tmpQueue->size;
				pbuff+=tmpQueue->size;
				space-=tmpQueue->size;
				tmpQueue=tmpQueue->next;
				if(tmpQueue==NULL)
					{
						m_csQueue.unlock();
						return E_SUCCESS;
					}
			}
		memcpy(pbuff,tmpQueue->pBuff,space);
		*psize+=space;
		m_csQueue.unlock();
		return E_SUCCESS;
	}	
	
	
/** Removes data from the Queue.
	* @param psize on call contains the size of data to remove, 
	*								on return contains the size of removed data
	* @retval E_SUCCESS if succesful
	* @retval E_UNKNOWN in case of an error
	*/
SINT32 CAQueue::remove(UINT32* psize)
	{
		if(m_Queue==NULL||psize==NULL)
			return E_UNKNOWN;
		if(*psize==0)
			return E_SUCCESS;
		m_csQueue.lock();
		UINT32 space=*psize;
		*psize=0;
		while(space>=m_Queue->size)
			{
				*psize+=m_Queue->size;
				space-=m_Queue->size;
				m_nQueueSize-=m_Queue->size;
#ifndef DO_TRACE
				delete []m_Queue->pBuff;
#else
				deleteUINT8Buff(m_Queue->pBuff,m_Queue->shadow_pBuff,m_Queue->allocSize);
#endif
				QUEUE* tmp=m_Queue;
				m_Queue=m_Queue->next;
#ifndef DO_TRACE
				delete tmp;
#else
				deleteQUEUE(tmp);
#endif
				if(m_Queue==NULL)
					{
						m_csQueue.unlock();
						return E_SUCCESS;
					}
			}
		if(space>0)
			{
				*psize+=space;
				m_Queue->size-=space;
				m_nQueueSize-=space;
				memmove(m_Queue->pBuff,m_Queue->pBuff+space,m_Queue->size);
			}
		m_csQueue.unlock();
		return E_SUCCESS;
	}

struct __queue_test
	{
		CAQueue* pQueue;
		UINT8* buff;
		UINT32 len;
	};

THREAD_RETURN producer(void* param)
	{
		struct __queue_test* pTest=(struct __queue_test *)param;
		UINT32 count=0;
		UINT32 aktSize;
		while(pTest->len>10000)
				{
					aktSize=rand();
					aktSize%=0xFFFF;
					aktSize%=pTest->len;
					if(pTest->pQueue->add(pTest->buff+count,aktSize)!=E_SUCCESS)
						THREAD_RETURN_ERROR;
					count+=aktSize;
					pTest->len-=aktSize;
					msSleep(rand()%100);
				}
		if(pTest->pQueue->add(pTest->buff+count,pTest->len)!=E_SUCCESS)
			THREAD_RETURN_ERROR;
		THREAD_RETURN_SUCCESS;
	}

THREAD_RETURN consumer(void* param)
	{
		struct __queue_test* pTest=(struct __queue_test *)param;
		UINT32 count=0;
		UINT32 aktSize;
		do
			{
				aktSize=rand();
				aktSize%=0xFFFF;
				if(pTest->pQueue->getOrWait(pTest->buff+count,&aktSize)!=E_SUCCESS)
					THREAD_RETURN_ERROR;
				count+=aktSize;
				pTest->len-=aktSize;
			}while(pTest->len>0);
		THREAD_RETURN_SUCCESS;
	}

SINT32 CAQueue::test()
	{
		CAQueue oQueue;
		#define TEST_SIZE 1000000
		UINT8* source=new UINT8[TEST_SIZE];
		UINT8* target=new UINT8[TEST_SIZE];
		getRandom(source,TEST_SIZE);
		UINT32 count=0;
		UINT32 aktSize;
		
		srand((unsigned)time( NULL ));
		
		//Single Thread.....
		//adding
		
		while(TEST_SIZE-count>10000)
				{
					aktSize=rand();
					aktSize%=0xFFFF;
					aktSize%=(TEST_SIZE-count);
					if(oQueue.add(source+count,aktSize)!=E_SUCCESS)
						return E_UNKNOWN;
					count+=aktSize;
					if(oQueue.getSize()!=count)
						return E_UNKNOWN;
				}
		if(oQueue.add(source+count,TEST_SIZE-count)!=E_SUCCESS)
			return E_UNKNOWN;
		if(oQueue.getSize()!=TEST_SIZE)
			return E_UNKNOWN;
		
		//getting
		count=0;
		while(!oQueue.isEmpty())
			{
				aktSize=rand();
				aktSize%=0xFFFF;
				if(oQueue.get(target+count,&aktSize)!=E_SUCCESS)
					return E_UNKNOWN;
				count+=aktSize;
			}
		if(count!=TEST_SIZE)
			return E_UNKNOWN;
		if(memcmp(source,target,TEST_SIZE)!=0)
			return E_UNKNOWN;
		//Multiple Threads....
		CAThread othreadProducer;
		CAThread othreadConsumer;
		othreadProducer.setMainLoop(producer);
		othreadConsumer.setMainLoop(consumer);
		struct __queue_test t1,t2;
		t1.buff=source;
		t2.buff=target;
		t2.len=t1.len=TEST_SIZE;
		t2.pQueue=t1.pQueue=&oQueue;
		othreadProducer.start(&t1);
		othreadConsumer.start(&t2);
		othreadProducer.join();
		othreadConsumer.join();
		if(memcmp(source,target,TEST_SIZE)!=0)
			return E_UNKNOWN;
		
		delete []source;
		delete []target;
#ifdef DO_TRACE
		m_maxAlloc=m_aktAlloc=0;
#endif

		return E_SUCCESS;
	}
