/*
 * prop8.c
 * Copyright (C) 1998-2001 A.J. van Os; Released under GPL
 *
 * Description:
 * Read the property information from a MS Word 8, 9 or 10 file
 *
 * Word 8 is better known as Word 97
 * Word 9 is better known as Word 2000
 * Word 10 is better known as Word 2002 or as Word XP
 */

#include <stdlib.h>
#include <string.h>
#include "antiword.h"


/*
 * iGet8InfoLength - the length of the information for Word 8/9/10 files
 */
static int
iGet8InfoLength(int iByteNbr, const unsigned char *aucFpage)
{
	int	iTmp, iDel, iAdd;
	unsigned short	usOpCode;

	usOpCode = usGetWord(iByteNbr, aucFpage);

	switch (usOpCode & 0xe000) {
	case 0x0000: case 0x2000:
		return 3;
	case 0x4000: case 0x8000: case 0xa000:
		return 4;
	case 0xe000:
		return 5;
	case 0x6000:
		return 6;
	case 0xc000:
		iTmp = (int)ucGetByte(iByteNbr + 2, aucFpage);
		if (usOpCode == 0xc615 && iTmp == 255) {
			iDel = (int)ucGetByte(iByteNbr + 3, aucFpage);
			iAdd = (int)ucGetByte(
					iByteNbr + 4 + iDel * 4, aucFpage);
			iTmp = 2 + iDel * 4 + iAdd * 3;
		}
		return 3 + iTmp;
	default:
		DBG_HEX(usOpCode);
		DBG_FIXME();
		return 1;
	}
} /* end of iGet8InfoLength */

/*
 * Translate the rowinfo to a member of the row_info enumeration
 */
row_info_enum
eGet8RowInfo(int iFodo,
	const unsigned char *aucGrpprl, int iBytes, row_block_type *pRow)
{
	int	iFodoOff, iInfoLen;
	int	iIndex, iSize, iCol;
	int	iPosCurr, iPosPrev;
	BOOL	bFound2416_0, bFound2416_1, bFound2417_0, bFound2417_1;
	BOOL	bFoundd608;

	fail(iFodo < 0 || aucGrpprl == NULL || pRow == NULL);

	iFodoOff = 0;
	bFound2416_0 = FALSE;
	bFound2416_1 = FALSE;
	bFound2417_0 = FALSE;
	bFound2417_1 = FALSE;
	bFoundd608 = FALSE;
	while (iBytes >= iFodoOff + 2) {
		iInfoLen = 0;
		switch (usGetWord(iFodo + iFodoOff, aucGrpprl)) {
		case 0x2416:	/* fIntable */
			if (odd(ucGetByte(iFodo + iFodoOff + 2, aucGrpprl))) {
				bFound2416_1 = TRUE;
			} else {
				bFound2416_0 = TRUE;
			}
			break;
		case 0x2417:	/* fTtp */
			if (odd(ucGetByte(iFodo + iFodoOff + 2, aucGrpprl))) {
				bFound2417_1 = TRUE;
			} else {
				bFound2417_0 = TRUE;
			}
			break;
		case 0xd608:	/* cDefTable */
			iSize = (int)ucGetByte(iFodo + iFodoOff + 2, aucGrpprl);
			if (iSize < 6 || iBytes < iFodoOff + 8) {
				DBG_DEC(iSize);
				DBG_DEC(iFodoOff);
				iInfoLen = 2;
				break;
			}
			iCol = (int)ucGetByte(iFodo + iFodoOff + 4, aucGrpprl);
			if (iCol < 1 ||
			    iBytes < iFodoOff + 4 + (iCol + 1) * 2) {
				DBG_DEC(iCol);
				DBG_DEC(iFodoOff);
				iInfoLen = 2;
				break;
			}
			if (iCol >= (int)elementsof(pRow->asColumnWidth)) {
				werr(1, "The number of columns is corrupt");
			}
			pRow->ucNumberOfColumns = (unsigned char)iCol;
			pRow->iColumnWidthSum = 0;
			iPosPrev = (int)(short)usGetWord(
					iFodo + iFodoOff + 5,
					aucGrpprl);
			for (iIndex = 0; iIndex < iCol; iIndex++) {
				iPosCurr = (int)(short)usGetWord(
					iFodo + iFodoOff + 7 + iIndex * 2,
					aucGrpprl);
				pRow->asColumnWidth[iIndex] =
						(short)(iPosCurr - iPosPrev);
				pRow->iColumnWidthSum +=
					pRow->asColumnWidth[iIndex];
				iPosPrev = iPosCurr;
			}
			bFoundd608 = TRUE;
			break;
		default:
			break;
		}
		if (iInfoLen <= 0) {
			iInfoLen =
				iGet8InfoLength(iFodo + iFodoOff, aucGrpprl);
			fail(iInfoLen <= 0);
		}
		iFodoOff += iInfoLen;
	}
	if (bFound2416_1 && bFound2417_1 && bFoundd608) {
		return found_end_of_row;
	}
	if (bFound2416_0 && bFound2417_0 && !bFoundd608) {
		return found_not_end_of_row;
	}
	if (bFound2416_1) {
		return found_a_cell;
	}
	if (bFound2416_0) {
		return found_not_a_cell;
	}
	return found_nothing;
} /* end of eGet8RowInfo */

/*
 * Fill the style information block with information
 * from a Word 8/9/10 file.
 * Returns TRUE when successful, otherwise FALSE
 */
static BOOL
bGet8StyleInfo(int iFodo, const unsigned char *aucFpage,
		style_block_type *pStyle)
{
	int	iBytes, iFodoOff, iInfoLen;
	int	iTmp, iDel, iAdd;
	short	sTmp;

	fail(iFodo <= 0 || aucFpage == NULL || pStyle == NULL);

	iBytes = 2 * (int)ucGetByte(iFodo, aucFpage);
	iFodoOff = 3;
	if (iBytes == 0) {
		iFodo++;
		iBytes = 2 * (int)ucGetByte(iFodo, aucFpage);
	}
	NO_DBG_DEC(iBytes);
	if (iBytes < 2) {
		return FALSE;
	}
	(void)memset(pStyle, 0, sizeof(*pStyle));
	sTmp = (short)usGetWord(iFodo + 1, aucFpage);
	if (sTmp >= 0 && sTmp <= (short)UCHAR_MAX) {
		pStyle->ucStyle = (unsigned char)sTmp;
	}
	NO_DBG_DEC(pStyle->ucStyle);
	while (iBytes >= iFodoOff + 2) {
		iInfoLen = 0;
		switch (usGetWord(iFodo + iFodoOff, aucFpage)) {
		case 0x2403:	/* jc */
			pStyle->ucAlignment = ucGetByte(
					iFodo + iFodoOff + 2, aucFpage);
			break;
		case 0x260a:	/* ilvl */
			pStyle->bInList = TRUE;
			pStyle->ucListLevel =
				ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			NO_DBG_DEC(pStyle->ucListLevel);
			break;
		case 0x4600:	/* istd */
			sTmp = (short)usGetWord(iFodo + iFodoOff + 2, aucFpage);
			NO_DBG_DEC(sTmp);
			NO_DBG_DEC(pStyle->ucStyle);
			break;
		case 0x460b:	/* ilfo */
			NO_DBG_DEC(usGetWord(iFodo + iFodoOff + 2, aucFpage));
			break;
		case 0x4610: /* Nest dxaLeft */
			sTmp = (short)usGetWord(iFodo + iFodoOff + 2, aucFpage);
			pStyle->sLeftIndent += sTmp;
			if (pStyle->sLeftIndent < 0) {
				pStyle->sLeftIndent = 0;
			}
			DBG_DEC(sTmp);
			DBG_DEC(pStyle->sLeftIndent);
			break;
		case 0x6c0d:	/* ChgTabsPapx */
			iTmp = (int)ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			if (iTmp < 2) {
				iInfoLen = 1;
				break;
			}
			DBG_DEC(iTmp);
			iDel = (int)ucGetByte(iFodo + iFodoOff + 3, aucFpage);
			if (iTmp < 2 + 2 * iDel) {
				iInfoLen = 1;
				break;
			}
			DBG_DEC(iDel);
			iAdd = (int)ucGetByte(
				iFodo + iFodoOff + 4 + 2 * iDel, aucFpage);
			if (iTmp < 2 + 2 * iDel + 2 * iAdd) {
				iInfoLen = 1;
				break;
			}
			DBG_DEC(iAdd);
			break;
		case 0x840e:	/* dxaRight */
			pStyle->sRightIndent = (short)usGetWord(
					iFodo + iFodoOff + 2, aucFpage);
			NO_DBG_DEC(pStyle->sRightIndent);
			break;
		case 0x840f:	/* dxaLeft */
			pStyle->sLeftIndent = (short)usGetWord(
					iFodo + iFodoOff + 2, aucFpage);
			NO_DBG_DEC(pStyle->sLeftIndent);
			break;
		case 0xc63e:	/* anld */
			iTmp = (int)ucGetByte(
					iFodo + iFodoOff + 2, aucFpage);
			if (iTmp < 84) {
				DBG_DEC(iTmp);
				break;
			}
			pStyle->ucListType = ucGetByte(
					iFodo + iFodoOff + 3, aucFpage);
			pStyle->bInList = TRUE;
			pStyle->ucListCharacter = ucGetByte(
					iFodo + iFodoOff + 23, aucFpage);
			break;
		default:
			break;
		}
		if (iInfoLen <= 0) {
			iInfoLen =
				iGet8InfoLength(iFodo + iFodoOff, aucFpage);
			fail(iInfoLen <= 0);
		}
		iFodoOff += iInfoLen;
	}
	return TRUE;
} /* end of bGet8StyleInfo */

/*
 * Build the lists with Paragraph Information for Word 8/9/10 files
 */
void
vGet8PapInfo(FILE *pFile, const pps_info_type *pPPS,
	const long *alBBD, size_t tBBDLen, const long *alSBD, size_t tSBDLen,
	const unsigned char *aucHeader)
{
	row_block_type		tRow;
	style_block_type	tStyle;
	long		*alParfPage;
	const long	*alBlockDepot;
	unsigned char	*aucBuffer;
	long	lParmOffset, lParmOffsetFirst, lParmOffsetLast;
	long	lBeginParfInfo, lTableSize, lTableStartBlock;
	size_t	tParfInfoLen, tBlockDepotLen;
	size_t	tBlockSize, tOffset, tSize, tLen;
	int	iIndex, iIndex2, iRun, iFodo, iLen;
	row_info_enum	eRowInfo;
	unsigned short	usDocStatus;
	unsigned char	aucFpage[BIG_BLOCK_SIZE];

	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(alBBD == NULL || alSBD == NULL);

	lBeginParfInfo = (long)ulGetLong(0x102, aucHeader); /* fcPlcfbtePapx */
	NO_DBG_HEX(lBeginParfInfo);
	tParfInfoLen = (size_t)ulGetLong(0x106, aucHeader); /* lcbPlcfbtePapx */
	NO_DBG_DEC(tParfInfoLen);
	if (tParfInfoLen < 4) {
		DBG_DEC(tParfInfoLen);
		return;
	}

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		lTableStartBlock = pPPS->t1Table.lSb;
		lTableSize = pPPS->t1Table.lSize;
	} else {
		lTableStartBlock = pPPS->t0Table.lSb;
		lTableSize = pPPS->t0Table.lSize;
	}
	DBG_DEC(lTableStartBlock);
	if (lTableStartBlock < 0) {
		DBG_DEC(lTableStartBlock);
		DBG_MSG("No paragraph information");
		return;
	}
	DBG_HEX(lTableSize);
	if (lTableSize < MIN_SIZE_FOR_BBD_USE) {
	  	/* Use the Small Block Depot */
		alBlockDepot = alSBD;
		tBlockDepotLen = tSBDLen;
		tBlockSize = SMALL_BLOCK_SIZE;
	} else {
	  	/* Use the Big Block Depot */
		alBlockDepot = alBBD;
		tBlockDepotLen = tBBDLen;
		tBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(tParfInfoLen);
	if (!bReadBuffer(pFile, lTableStartBlock,
			alBlockDepot, tBlockDepotLen, tBlockSize,
			aucBuffer, lBeginParfInfo, tParfInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, tParfInfoLen);

	tLen = (tParfInfoLen / 4 - 1) / 2;
	tSize = tLen * sizeof(long);
	alParfPage = xmalloc(tSize);
	for (iIndex = 0, tOffset = (tLen + 1) * 4;
	     iIndex < (int)tLen;
	     iIndex++, tOffset += 4) {
		 alParfPage[iIndex] = (long)ulGetLong(tOffset, aucBuffer);
		 NO_DBG_DEC(alParfPage[iIndex]);
	}
	DBG_HEX(ulGetLong(0, aucBuffer));
	aucBuffer = xfree(aucBuffer);
	aucBuffer = NULL;
	NO_DBG_PRINT_BLOCK(aucHeader, HEADER_SIZE);

	(void)memset(&tStyle, 0, sizeof(tStyle));
	(void)memset(&tRow, 0, sizeof(tRow));
	lParmOffsetFirst = -1;
	for (iIndex = 0; iIndex < (int)tLen; iIndex++) {
		fail(alParfPage[iIndex] > LONG_MAX / BIG_BLOCK_SIZE);
		if (!bReadBuffer(pFile, pPPS->tWordDocument.lSb,
				alBBD, tBBDLen, BIG_BLOCK_SIZE,
				aucFpage,
				alParfPage[iIndex] * BIG_BLOCK_SIZE,
				BIG_BLOCK_SIZE)) {
			break;
		}
		NO_DBG_PRINT_BLOCK(aucFpage, BIG_BLOCK_SIZE);
		iRun = (int)ucGetByte(0x1ff, aucFpage);
		NO_DBG_DEC(iRun);
		for (iIndex2 = 0; iIndex2 < iRun; iIndex2++) {
			NO_DBG_HEX(ulGetLong(iIndex2 * 4, aucFpage));
			iFodo = 2 * (int)ucGetByte(
				(iRun + 1) * 4 + iIndex2 * 13, aucFpage);
			if (iFodo <= 0) {
				continue;
			}

			if (bGet8StyleInfo(iFodo, aucFpage, &tStyle)) {
				lParmOffset = (long)ulGetLong(
						iIndex2 * 4, aucFpage);
				NO_DBG_HEX(lParmOffset);
				tStyle.lFileOffset = lTextOffset2FileOffset(
							lParmOffset);
				vAdd2StyleInfoList(&tStyle);
				(void)memset(&tStyle, 0, sizeof(tStyle));
			}

			iLen = 2 * (int)ucGetByte(iFodo, aucFpage);
			if (iLen == 0) {
				iFodo++;
				iLen = 2 * (int)ucGetByte(iFodo, aucFpage);
			}
			eRowInfo = eGet8RowInfo(iFodo,
					aucFpage + 3, iLen - 3, &tRow);
			switch (eRowInfo) {
			case found_a_cell:
				if (lParmOffsetFirst >= 0) {
					break;
				}
				lParmOffsetFirst = (long)ulGetLong(
						iIndex2 * 4, aucFpage);
				NO_DBG_HEX(lParmOffsetFirst);
				tRow.lTextOffsetStart = lParmOffsetFirst;
				tRow.lFileOffsetStart = lTextOffset2FileOffset(
							lParmOffsetFirst);
				DBG_HEX_C(tRow.lFileOffsetStart < 0,
							lParmOffsetFirst);
				break;
			case found_end_of_row:
				lParmOffsetLast = (long)ulGetLong(
						iIndex2 * 4, aucFpage);
				NO_DBG_HEX(lParmOffsetLast);
				tRow.lTextOffsetEnd = lParmOffsetLast;
				tRow.lFileOffsetEnd = lTextOffset2FileOffset(
							lParmOffsetLast);
				DBG_HEX_C(tRow.lFileOffsetEnd < 0,
							lParmOffsetLast);
				vAdd2RowInfoList(&tRow);
				(void)memset(&tRow, 0, sizeof(tRow));
				lParmOffsetFirst = -1;
				break;
			case found_nothing:
				break;
			default:
				DBG_DEC(eRowInfo);
				break;
			}
		}
	}
	alParfPage = xfree(alParfPage);
} /* end of vGet8PapInfo */

/*
 * Fill the font information block with information
 * from a Word 8/9/10 file.
 * Returns TRUE when successful, otherwise FALSE
 */
static BOOL
bGet8FontInfo(int iFodo, const unsigned char *aucFpage,
		font_block_type *pFont)
{
	int	iBytes, iFodoOff, iInfoLen;
	unsigned short	usTmp;
	short		sTmp;
	unsigned char	ucTmp;

	fail(iFodo <= 0 || aucFpage == NULL || pFont == NULL);

	iBytes = (int)ucGetByte(iFodo, aucFpage);
	iFodoOff = 1;
	if (iBytes == 0) {
		iFodo++;
		iBytes = 2 * (int)ucGetByte(iFodo, aucFpage);
	}
	NO_DBG_DEC(iBytes);
	if (iBytes < 2) {
		return FALSE;
	}
	(void)memset(pFont, 0, sizeof(*pFont));
	pFont->sFontsize = DEFAULT_FONT_SIZE;
	while (iBytes >= iFodoOff + 2) {
		switch (usGetWord(iFodo + iFodoOff, aucFpage)) {
		case 0x0835:	/* fBold */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_BOLD;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_BOLD;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_BOLD;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x0836:	/* fItalic */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_ITALIC;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_ITALIC;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_ITALIC;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x0837:	/* fStrike */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_STRIKE;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_STRIKE;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_STRIKE;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x083a:	/* fSmallCaps */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_SMALL_CAPITALS;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_SMALL_CAPITALS;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_SMALL_CAPITALS;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x083b:	/* fCaps */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_CAPITALS;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_CAPITALS;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_CAPITALS;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x083c:	/* fVanish */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			switch (ucTmp) {
			case   0:	/* Unset */
				pFont->ucFontstyle &= ~FONT_HIDDEN;
				break;
			case   1:	/* Set */
				pFont->ucFontstyle |= FONT_HIDDEN;
				break;
			case 128:	/* Unchanged */
				break;
			case 129:	/* Negation */
				pFont->ucFontstyle ^= FONT_HIDDEN;
				break;
			default:
				DBG_DEC(ucTmp);
				DBG_FIXME();
				break;
			}
			break;
		case 0x2a32:	/* cDefault */
			pFont->ucFontstyle &= FONT_HIDDEN;
			pFont->ucFontcolor = FONT_COLOR_DEFAULT;
			break;
		case 0x2a3e:	/* cKul */
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			if (ucTmp != 0 && ucTmp != 5) {
				NO_DBG_MSG("Underline text");
				pFont->ucFontstyle |= FONT_UNDERLINE;
				if (ucTmp == 6) {
					DBG_MSG("Bold text");
					pFont->ucFontstyle |= FONT_BOLD;
				}
			}
			break;
		case 0x2a42:	/* cIco */
			pFont->ucFontcolor =
				ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			NO_DBG_DEC(pFont->ucFontcolor);
			break;
		case 0x4a30:	/* cIstd */
			sTmp = (short)usGetWord(iFodo + iFodoOff + 2, aucFpage);
			DBG_DEC(sTmp);
			break;
		case 0x4a43:	/* cHps */
			usTmp = usGetWord(iFodo + iFodoOff + 2, aucFpage);
			if (usTmp > (unsigned short)SHRT_MAX) {
				pFont->sFontsize = SHRT_MAX;
			} else {
				pFont->sFontsize = (short)usTmp;
			}
			NO_DBG_DEC(pFont->sFontsize);
			break;
		case 0x4a51:	/* cFtc */
			usTmp = usGetWord(iFodo + iFodoOff + 2, aucFpage);
			if (usTmp <= (unsigned short)UCHAR_MAX) {
				pFont->ucFontnumber = (unsigned char)usTmp;
			}
			break;
		default:
			break;
		}
		iInfoLen = iGet8InfoLength(iFodo + iFodoOff, aucFpage);
		fail(iInfoLen <= 0);
		iFodoOff += iInfoLen;
	}
	return TRUE;
} /* end of bGet8FontInfo */

/*
 * Fill the picture information block with information
 * from a Word 8/9/10 file.
 * Returns TRUE when successful, otherwise FALSE
 */
static BOOL
bGet8PicInfo(int iFodo, const unsigned char *aucFpage,
		picture_block_type *pPicture)
{
	int	iBytes, iFodoOff, iInfoLen;
	BOOL	bFound;
	unsigned char	ucTmp;

	fail(iFodo <= 0 || aucFpage == NULL || pPicture == NULL);

	iBytes = (int)ucGetByte(iFodo, aucFpage);
	iFodoOff = 1;
	if (iBytes == 0) {
		iFodo++;
		iBytes = 2 * (int)ucGetByte(iFodo, aucFpage);
	}
	NO_DBG_DEC(iBytes);
	if (iBytes < 2) {
		return FALSE;
	}
	bFound = FALSE;
	(void)memset(pPicture, 0, sizeof(*pPicture));
	while (iBytes >= iFodoOff + 2) {
		switch (usGetWord(iFodo + iFodoOff, aucFpage)) {
		case 0x0806:
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			if (ucTmp == 0x01) {
				/* Not a picture, but a form field */
				return FALSE;
			}
			DBG_DEC_C(ucTmp != 0, ucTmp);
			break;
		case 0x080a:
			ucTmp = ucGetByte(iFodo + iFodoOff + 2, aucFpage);
			if (ucTmp == 0x01) {
				/* Not a picture, but an OLE object */
				return FALSE;
			}
			DBG_DEC_C(ucTmp != 0, ucTmp);
			break;
		case 0x6a03:
			pPicture->lPictureOffset = (long)ulGetLong(
					iFodo + iFodoOff + 2, aucFpage);
			bFound = TRUE;
			break;
		default:
			break;
		}
		iInfoLen = iGet8InfoLength(iFodo + iFodoOff, aucFpage);
		fail(iInfoLen <= 0);
		iFodoOff += iInfoLen;
	}
	return bFound;
} /* end of bGet8PicInfo */

/*
 * Build the lists with Character Information for Word 8/9/10 files
 */
void
vGet8ChrInfo(FILE *pFile, const pps_info_type *pPPS,
	const long *alBBD, size_t tBBDLen, const long *alSBD, size_t tSBDLen,
	const unsigned char *aucHeader)
{
	font_block_type		tFont;
	picture_block_type	tPicture;
	long		*alCharPage;
	const long	*alBlockDepot;
	unsigned char	*aucBuffer;
	long	lFileOffset, lParmOffset, lBeginCharInfo;
	long	lTableSize, lTableStartBlock;
	size_t	tCharInfoLen, tBlockDepotLen;
	size_t	tOffset, tBlockSize, tSize, tLen;
	int	iIndex, iIndex2, iRun, iFodo;
	unsigned short	usDocStatus;
	unsigned char	aucFpage[BIG_BLOCK_SIZE];

	fail(pFile == NULL || pPPS == NULL || aucHeader == NULL);
	fail(alBBD == NULL || alSBD == NULL);

	lBeginCharInfo = (long)ulGetLong(0xfa, aucHeader); /* fcPlcfbteChpx */
	NO_DBG_HEX(lBeginCharInfo);
	tCharInfoLen = (size_t)ulGetLong(0xfe, aucHeader); /* lcbPlcfbteChpx */
	NO_DBG_DEC(tCharInfoLen);
	if (tCharInfoLen < 4) {
		DBG_DEC(tCharInfoLen);
		return;
	}

	/* Use 0Table or 1Table? */
	usDocStatus = usGetWord(0x0a, aucHeader);
	if (usDocStatus & BIT(9)) {
		lTableStartBlock = pPPS->t1Table.lSb;
		lTableSize = pPPS->t1Table.lSize;
	} else {
		lTableStartBlock = pPPS->t0Table.lSb;
		lTableSize = pPPS->t0Table.lSize;
	}
	DBG_DEC(lTableStartBlock);
	if (lTableStartBlock < 0) {
		DBG_DEC(lTableStartBlock);
		DBG_MSG("No character information");
		return;
	}
	DBG_HEX(lTableSize);
	if (lTableSize < MIN_SIZE_FOR_BBD_USE) {
	  	/* Use the Small Block Depot */
		alBlockDepot = alSBD;
		tBlockDepotLen = tSBDLen;
		tBlockSize = SMALL_BLOCK_SIZE;
	} else {
	  	/* Use the Big Block Depot */
		alBlockDepot = alBBD;
		tBlockDepotLen = tBBDLen;
		tBlockSize = BIG_BLOCK_SIZE;
	}
	aucBuffer = xmalloc(tCharInfoLen);
	if (!bReadBuffer(pFile, lTableStartBlock,
			alBlockDepot, tBlockDepotLen, tBlockSize,
			aucBuffer, lBeginCharInfo, tCharInfoLen)) {
		aucBuffer = xfree(aucBuffer);
		return;
	}
	NO_DBG_PRINT_BLOCK(aucBuffer, tCharInfoLen);

	tLen = (tCharInfoLen / 4 - 1) / 2;
	tSize = tLen * sizeof(long);
	alCharPage = xmalloc(tSize);
	for (iIndex = 0, tOffset = (tLen + 1) * 4;
	     iIndex < (int)tLen;
	     iIndex++, tOffset += 4) {
		 alCharPage[iIndex] = (long)ulGetLong(tOffset, aucBuffer);
		 NO_DBG_DEC(alCharPage[iIndex]);
	}
	DBG_HEX(ulGetLong(0, aucBuffer));
	aucBuffer = xfree(aucBuffer);
	aucBuffer = NULL;
	NO_DBG_PRINT_BLOCK(aucHeader, HEADER_SIZE);

	(void)memset(&tFont, 0, sizeof(tFont));
	for (iIndex = 0; iIndex < (int)tLen; iIndex++) {
		fail(alCharPage[iIndex] > LONG_MAX / BIG_BLOCK_SIZE);
		if (!bReadBuffer(pFile, pPPS->tWordDocument.lSb,
				alBBD, tBBDLen, BIG_BLOCK_SIZE,
				aucFpage,
				alCharPage[iIndex] * BIG_BLOCK_SIZE,
				BIG_BLOCK_SIZE)) {
			break;
		}
		NO_DBG_PRINT_BLOCK(aucFpage, BIG_BLOCK_SIZE);
		iRun = (int)ucGetByte(0x1ff, aucFpage);
		NO_DBG_DEC(iRun);
		for (iIndex2 = 0; iIndex2 < iRun; iIndex2++) {
			lParmOffset = (long)ulGetLong(
						iIndex2 * 4, aucFpage);
			lFileOffset = lTextOffset2FileOffset(lParmOffset);
			iFodo = 2 * (int)ucGetByte(
				(iRun + 1) * 4 + iIndex2, aucFpage);
			if (iFodo == 0) {
				NO_DBG_HEX(lParmOffset);
				vReset2FontInfoList(lFileOffset);
			} else {
				if (bGet8FontInfo(iFodo, aucFpage, &tFont)) {
					tFont.lFileOffset = lFileOffset;
					vAdd2FontInfoList(&tFont);
				}
			}
			(void)memset(&tFont, 0, sizeof(tFont));
			if (iFodo <= 0) {
				continue;
			}
			if (bGet8PicInfo(iFodo, aucFpage, &tPicture)) {
				tPicture.lFileOffset = lFileOffset;
				tPicture.lFileOffsetPicture =
					lDataOffset2FileOffset(
						tPicture.lPictureOffset);
				vAdd2PicInfoList(&tPicture);
			}
			(void)memset(&tPicture, 0, sizeof(tPicture));
		}
	}
	alCharPage = xfree(alCharPage);
} /* end of vGet8ChrInfo */
