 /* Copyright (C) 2006 Jan Wedekind.
   This file is part of the recipe database application AnyMeal.

   AnyMeal is free software; you can redistribute it and/or modify it under
   the terms of the GNU GENERAL PUBLIC LICENSE as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   AnyMeal is distributed in the hope that it will be useful, but WITHOUT ANY
   WARRANTY; without even the implied warranty of MERCHANTIBILITY or FITNESS
   FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
   details.

   You should have received a copy of the GNU General Public License along
   with AnyMeal; if not, contact one of the authors of this software. */
#include <boost/shared_array.hpp>
#include <cstddef>
#include <cstdlib>
#include <errno.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include "mysqlDaemon.hpp"
#include "utils.hpp"

using namespace boost;
using namespace std;

MySQLDaemon::MySQLDaemon( const string &_datadir,
                          const string &_socket ) throw (Error)
{
  ERRORMACRO( pipe( pipes ) == 0, Error, , "Failed to create pipes." );
  pid = fork();
  ERRORMACRO( pid >= 0, Error, , "Couldn't fork process." );

  if ( pid == 0 ) {

    // New process for embedded server.
    int nulldev = open( "/dev/null", O_RDWR );
    dup2( nulldev, STDOUT_FILENO );
    dup2( nulldev, STDIN_FILENO );
    close( nulldev );
    
    close( pipes[0] );
    dup2( pipes[1], STDERR_FILENO );
    close( pipes[1] );

    execlp( "/usr/sbin/mysqld", "mysqld", "--skip-networking",
            ( string ( "--datadir=" ) + _datadir ).c_str(),
            ( string( "--socket=" ) + _socket ).c_str(), NULL );
#ifndef NDEBUG
    cerr << "Start of server failed (" << strerror( errno ) << ")." << endl;
#endif
    exit( EXIT_FAILURE );

  } else {

    // AnyMeal process continues here.
    close( pipes[1] );
    
  };
}

MySQLDaemon::~MySQLDaemon(void)
{
  kill( pid, SIGQUIT );// SIGTERM required on older systems.
  int status;
  waitpid( pid, &status, 0 );
}

bool MySQLDaemon::running(void)
{
  int status;
  return waitpid( pid, &status, WNOHANG ) == 0;
}

const string MySQLDaemon::getLog(void)
{
  readPending();
  return log.str();
}

string MySQLDaemon::create( const string &_datadir,
                            const string &_socket ) throw (Error)
{
  createDirRecursively( _datadir );
  FILE *messages =
    popen( ( string( "/usr/bin/mysql_install_db --skip-networking "
                     "--datadir='" ) + _datadir + "' --socket='" + _socket +
             '\'' ).c_str(), "r" );
  ERRORMACRO( messages != NULL, Error, ,
              "Failed to start '/usr/bin/mysql_install_db'." );
  stringstream s;
  while ( !feof( messages ) ) {
    char *p = NULL;
    size_t n;
    getline( &p, &n, messages );
    s << p;
  };
  return s.str();
}

void MySQLDaemon::readPending(void)
{
  int filedes = pipes[0];

  fd_set set;
  struct timeval timeout;

  FD_ZERO(&set);
  FD_SET (filedes, &set);

  timeout.tv_sec = 0;
  timeout.tv_usec = 0;

  while( true ) {

    int count;
    ioctl( filedes, FIONREAD, &count );
    if ( count == 0 )
      break;
    shared_array< char > buffer( new char[ count + 1 ] );
    read( filedes, buffer.get(), count );
    buffer[ count ] = 0;
    log << buffer.get();

  };
}
