# logic.rb: contains most logic from original apt-listbugs.
#
# Copyright (C) 2002  Masato Taruishi <taru@debian.org>
# Copyright (C) 2006-2008  Junichi Uekawa <dancer@debian.org>
# Copyright (C) 2008-2012  Francesco Poli <invernomuto@paranoici.org>
# Copyright (C) 2009-2010  Ryan Niebur <ryan@debian.org>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License with
#  the Debian GNU/Linux distribution in file /usr/share/common-licenses/GPL;
#  if not, write to the Free Software Foundation, Inc., 59 Temple Place,
#  Suite 330, Boston, MA  02111-1307  USA

require 'getoptlong'
require 'debian'
require 'debian/bug'
require 'debian/bts'
require 'thread'
require 'tempfile'
require 'gettext'
require 'rss/maker'
include GetText


class AppConfig
  QUERYBTS = "/usr/bin/querybts"
  WWW_BROWSER = "/usr/bin/www-browser"
  SENSIBLE_BROWSER = "/usr/bin/sensible-browser"

  def usage
    $stderr.print _("Usage: "), File.basename($0),
      _(" [options] <command> [arguments]"),
      "\n",
      _("Options:\n"),
      _(" -h               : Display this help and exit.\n"),
      _(" -v               : Show version number and exit.\n"),
      sprintf(_(" -s <severities>  : Severities you want to see [%s], or [all].\n"), @severity.join(',')),
      _(" -T <tags>        : Tags you want to see.\n"),
      sprintf(_(" -S <stats>       : Stats you want to see [%s].\n"), @stats.join(',')),
      _(" -B <bug#>        : Restrict reporting to specified bug#s only.\n"),
      _(" -D               : Show downgraded packages, too.\n"),
      sprintf(_(" -H <hostname>    : Hostname of Debian Bug Tracking System [%s].\n"), @hostname),
      sprintf(_(" -p <port>        : Port number of the server [%s].\n"), @port),
      sprintf(_(" -P <priority>    : Specifies Pin-Priority value [%s].\n"), @pin_priority),
      _(" -E <title>       : Specifies the title of RSS output.\n"),
      _(" -q               : Don't display progress bar.\n"),
      _(" -C <apt.conf>    : Specify apt.conf.\n"),
      _(" -y               : Assume that you select yes for all questions.\n"),
      _(" -n               : Assume that you select no for all questions.\n"),
      _(" -d               : Debug.\n"),
      _("Commands:\n"),
      _(" apt              : Apt mode.\n"),
      _(" list <pkg...>    : List bug reports of the specified packages.\n"),
      _(" rss <pkg...>     : List bug reports of the specified packages in RSS.\n"),
      _("See the manual page for the long options.\n")
  end

  def initialize
    @severity = ["critical", "grave", "serious"]
    @tag = nil
    @stats = ["forwarded", "done", "pending", "pending-fixed", ""]
    @statmap = [["forwarded", _("forwarded")],
                ["done", _("marked as done in some version")],
                ["pending", _("unfixed")],
                ["pending-fixed", _("tagged as pending a fix")]]
    @fbugs = nil
    @show_downgrade = false
    @hostname = "bugs.debian.org"
    @port = 80
    @quiet = false
    @command = nil
    @parser = nil
    @querybts = nil

    @ignore_bugs = read_ignore_bugs("/etc/apt/listbugs/ignore_bugs")
    @system_ignore_bugs = read_ignore_bugs("/var/lib/apt-listbugs/ignore_bugs")
    @ignore_bugs.each { |bug|
      @system_ignore_bugs.add(bug, false)
    }
    @frontend = ConsoleFrontend.new( self )
    @pin_priority = "1000"
    @apt_conf = nil

    @yes = nil

  end

  attr_accessor :severity, :stats, :quiet, :title
  attr_accessor :show_downgrade, :hostname, :tag, :fbugs
  attr_accessor :frontend, :pin_priority, :yes, :ignore_regexp
  attr_reader :command, :parser, :querybts, :ignore_bugs, :system_ignore_bugs, :browser

  def parse_options
    opt_parser = GetoptLong.new
    opt_parser.set_options(['--help', '-h', GetoptLong::NO_ARGUMENT],
			   ['--severity', '-s', GetoptLong::REQUIRED_ARGUMENT],
			   ['--version', '-v', GetoptLong::NO_ARGUMENT],
			   ['--tag', '-T', GetoptLong::REQUIRED_ARGUMENT],
			   ['--stats', '-S', GetoptLong::REQUIRED_ARGUMENT],
			   ['--bugs', '-B', GetoptLong::REQUIRED_ARGUMENT],
			   ['--show-downgrade', '-D', GetoptLong::NO_ARGUMENT],
			   ['--hostname', '-H', GetoptLong::REQUIRED_ARGUMENT],
			   ['--port', '-p', GetoptLong::REQUIRED_ARGUMENT],
			   ['--pin-priority', '-P', GetoptLong::REQUIRED_ARGUMENT],
			   ['--title', '-E', GetoptLong::REQUIRED_ARGUMENT],
			   ['--quiet', '-q', GetoptLong::NO_ARGUMENT],
			   ['--aptconf', '-C', GetoptLong::REQUIRED_ARGUMENT],
			   ['--force-yes', '-y', GetoptLong::NO_ARGUMENT],
			   ['--force-no', '-n', GetoptLong::NO_ARGUMENT],
			   ['--debug', '-d', GetoptLong::NO_ARGUMENT]
			   );

    begin
      opt_parser.each_option { |optname, optargs|
	case optname
	when '--help'
	  usage
	  exit 0
        when '--version'
          puts $VERSION
          exit 0
	when '--severity'
          case optargs
          when "all"
            @severity = ["critical","grave","serious","important","normal","minor","wishlist"]
          else
            @severity = optargs.split(',')
          end
	when '--tag'
	  @tag = optargs.split(',')
	when '--stats'
	  @stats = optargs.split(',')
	when '--bugs'
	  @fbugs = optargs.split(',')
	when '--show-downgrade'
	  @show_downgrade = true
	when '--hostname'
	  @hostname = optargs
	when '--port'
	  @port = optargs.to_i
	when '--pin-priority'
	  @pin_priority = optargs
	when '--title'
	  @title = optargs
	when '--quiet'
	  @quiet = true
	when '--aptconf'
	  @apt_conf = " -c " + optargs
	when '--debug'
	  $DEBUG = 1
	when '--force-yes'
	  @yes = true
	when '--force-no'
	  @yes = false
	end
      }
    rescue GetoptLong::AmbigousOption, GetoptLong::NeedlessArgument,
	GetoptLong::MissingArgument, GetoptLong::InvalidOption
      usage
      exit 1
    end

    if ! $stdout.isatty
      @quiet = true
      @yes = false if @yes.nil?
    end

    @title = "Debian Bugs (#{@severity.join(', ')})" if ! @title

    # http_proxy sanity check
    if ENV["HTTP_PROXY"] != nil && ENV["http_proxy"] == nil
      $stderr.puts _("W: sanity check failed: environment variable http_proxy is unset and HTTP_PROXY is set.")
    end

    # enable proxy for SOAP
    if ENV["http_proxy"] != nil && ENV["soap_use_proxy"] != "on"
      ENV["soap_use_proxy"] = "on"
    end

    # proxy settings in apt.conf
    if /http_proxy='(.*)'/ =~ `apt-config #{@apt_conf} shell http_proxy acquire::http::proxy`
      puts "proxy configuration from apt.conf: #{$1}" if $DEBUG
      if $1 == 'DIRECT' || $1 == ''
        puts "Disabling proxy due to DIRECT, or empty string" if $DEBUG
        ENV.delete("http_proxy")
        ENV.delete("soap_use_proxy")
      else
        ENV["http_proxy"] = $1
        ENV["soap_use_proxy"] = "on"
      end
    end
    if /http_proxy='(.*)'/ =~ `apt-config #{@apt_conf} shell http_proxy acquire::http::proxy::bugs.debian.org`
      puts "proxy configuration from apt.conf, specific for bugs.debian.org: #{$1}" if $DEBUG
      if $1 == 'DIRECT'
        puts "Disabling proxy due to DIRECT" if $DEBUG
        ENV.delete("http_proxy")
        ENV.delete("soap_use_proxy")
      else
        ENV["http_proxy"] = $1
        ENV["soap_use_proxy"] = "on"
      end
    end

    # command
    command = ARGV.shift
    case command
    when nil
      STDERR.puts _("E: You need to specify a command.")
      usage
      exit 1
    when "list"
      @command = "list"
    when "apt"
      @command = "apt"
    when "rss"
      @command = "rss"
    else
      STDERR.puts _("E: Unknown command ") +  "'#{command}'."
      usage
      exit 1
    end

    if @command == "apt"
      begin
        test_tty = File.open("/dev/tty")
        test_tty.close if test_tty
      rescue
        if @yes.nil?
          $stderr.puts _("W: cannot open /dev/tty - running inside su -c \"command\"? Switching to non-interactive failure mode (see /usr/share/doc/apt-listbugs/README.Debian.gz)")
          @yes = false
        end
        @quiet = true
      end
    end

    @parser =
      Debian::BTS::Parser::SoapIndex.new(@hostname, @port)

    if FileTest.executable?("#{QUERYBTS}")
      @querybts = QUERYBTS
    end

    if FileTest.executable?("#{SENSIBLE_BROWSER}")
      @browser = SENSIBLE_BROWSER
    else
      @browser = WWW_BROWSER
    end

    if /ignore_regexp='(.*)'/ =~ `apt-config #{@apt_conf} shell ignore_regexp AptListbugs::IgnoreRegexp`
      @ignore_regexp = $1
    end
  end

  # return the descriptive name for a status
  def statmap(stat)
    r=@statmap.assoc(stat)
    if r then
      r[1]
    else
      stat
    end
  end

  def read_ignore_bugs(path)
    ignore_bugs = IgnoreBugs.new(path)
  end
end

class IgnoreBugs < Array

  @@path_mutex = {}

  def initialize(path)
    super()
    @path = path
    @@path_mutex[path] = Mutex.new if @@path_mutex[path] == nil

    if FileTest.exist?(path)
      begin
        open(path).each { |bug|
          if /\s*#/ =~ bug
            next
          end
          if /\s*(\S+)/ =~ bug
            self << $1
          end
        }
      rescue Errno::EACCES
        # read-access is not possible, warn the user that the
        # file won't be taken into account
        $stderr.puts sprintf(_("W: Cannot read from %s"), @path)
      end
    end

    @gavewritewarning = nil
  end

  def add(entry, write=true)
    if write == true
      @@path_mutex[@path].synchronize {
        begin
          open(@path, "a") { |file|
            file.puts entry
          }
        rescue Errno::EACCES
          # write-access is not possible, warn the user that the
          # file won't be updated
          if @gavewritewarning.nil?
            $stderr.puts sprintf(_("W: Cannot write to %s"), @path)
            @gavewritewarning = true
          end
        end
      }
    end
    self << entry
  end

end

class Viewer

  def initialize(config)
    @config = config
  end

  class SimpleViewer < Viewer

    DeprecatedWarning = _("********** on_hold IS DEPRECATED. USE p INSTEAD to use pin **********")
    DeprecatedWarningHeader = "*" * DeprecatedWarning.length

    def view(new_pkgs, cur_pkgs, bugs)
      if display_bugs(bugs, new_pkgs.keys, cur_pkgs, new_pkgs) == false
        return true
      end

      if @config.command == "list"
	return true
      end

      answer = "n"
      hold_pkgs = []
      while true
	ask_str = _("Are you sure you want to install/upgrade the above packages?").dup
	if @config.querybts != nil || @config.browser != nil
	  if hold_pkgs.empty?
	    ask_str << " [Y/n/?/...]"
	  else
	    ask_str << " [N/?/...]"
	  end
	else
	  ask_str << " [Y/n]"
	end
	if @config.yes.nil?
          a = @config.frontend.ask ask_str
        else
          a = "y" if @config.yes
          a = "n" if ! @config.yes
        end
	if a == ""
	  if hold_pkgs.empty?
	    answer = "y"
	  else
	    answer = "n"
	  end
	else
	  answer = a.downcase
	end
	case answer
	when "y"
	  return true
	when "a"
	  if hold_pkgs.empty?
            bugs.each { |bug|
              if ! @config.system_ignore_bugs.include?(bug.bug_number)
                @config.system_ignore_bugs.add(bug)
                @config.system_ignore_bugs.add(bug.bug_number)
              end
            }
	    return true
	  end
	when "n"
	  return false
	when /^#?(\d+)$/
	  if @config.querybts != nil
	    system("#{@config.querybts} -u text #{$1} < /dev/tty")
          else
            @config.frontend.puts sprintf(_("You must install the reportbug package to be able to do this"))
          end
	when /^i\s+(\d+)$/
	  if ! @config.system_ignore_bugs.include?($1)
	    @config.system_ignore_bugs.add($1)
	    Factory::BugsFactory.delete_ignore_bugs(bugs)
	    @config.frontend.puts sprintf(_("%s ignored"), $1)
	  else
	    @config.frontend.puts sprintf(_("%s already ignored"), $1)
	  end
	when "r"
	  display_bugs(bugs, new_pkgs.keys - hold_pkgs, cur_pkgs, new_pkgs)

	when /^(h|p)\s+(.+)$/
	  key = $1
	  if key == "h"
	    @config.frontend.puts DeprecatedWarningHeader
	    @config.frontend.puts DeprecatedWarning
	    @config.frontend.puts DeprecatedWarningHeader
	  end
	  pkgs = $2.split(/\s+/)
	  if key == "h"
	    h = on_hold(pkgs)
	  else
	    h = pinned(pkgs, cur_pkgs, bugs)
	  end
	  hold_pkgs.concat(h) if h != nil

	when "w"
	  puts bugs if $DEBUG
	  display_bugs_as_html(bugs, cur_pkgs.keys - hold_pkgs, cur_pkgs, new_pkgs) if @config.browser != nil

        when /(h|p)/
	  key = $1
	  if key == "h"
	    @config.frontend.puts DeprecatedWarningHeader
	    @config.frontend.puts DeprecatedWarning
	    @config.frontend.puts DeprecatedWarningHeader
	  end
	  pkgs = {}
	  if key == "p"
	    bugs.each { |bug|
	      ## FIXME: need to parse preferences correctly?
	      if ! system("grep -q \"Package: #{bug.pkg_name}\" /etc/apt/preferences 2> /dev/null")
	        pkgs[bug.pkg_name] = 1
	      end
            }
	  else
            bugs.each { |bug|
              pkgs[bug.pkg_name] = 1
            }
	  end
	  if pkgs.size != 0
            if @config.frontend.yes_or_no? ngettext(
             # TRANSLATORS: %{plist} is a comma-separated list of %{npkgs} packages to be pinned or put on hold.
             "The following package will be pinned or on hold:\n %{plist}\nAre you sure?",
             "The following %{npkgs} packages will be pinned or on hold:\n %{plist}\nAre you sure?",
             pkgs.size) % {:npkgs => pkgs.size,
                           :plist => pkgs.keys.join(', ')}
	      if key == "h"
                h = on_hold(pkgs.keys)
              else
                h = pinned(pkgs.keys, cur_pkgs, bugs)
              end
            end
	    hold_pkgs.concat(h) if h != nil
	  else
	    @config.frontend.puts sprintf(_("All selected packages are already pinned or on hold. Ignoring %s command."), key)
	  end
	else
	  if hold_pkgs.empty?
	    @config.frontend.puts "" +
              _("     y     - continue the apt installation, but do not mark the bugs as ignored.\n") +
              _("     a     - continue the apt installation and mark all the above bugs as ignored.\n")
	  end
	  @config.frontend.puts "" +
	    _("     n     - stop the apt installation.\n") +
	    _("   <num>   - query the specified bug number (requires reportbug).\n") +
	    _("  #<num>   - same as <num>\n") +
	    _("     r     - redisplay bug lists.\n") +
	    _(" p <pkg..> - pin pkgs (restart APT session to enable).\n") +
	    _(" p         - pin all the above pkgs (restart APT session to enable).\n") +
	    _(" i <num>   - mark bug number <num> as ignored.\n") +
	    _("     ?     - print this help.\n")
	  if @config.browser != nil
	    @config.frontend.puts sprintf(_("     w     - display bug lists in HTML (uses %s).\n"), File.basename(@config.browser))
	  end
	end
      end
    end

    def bugs_of_pkg( bugs, pkg )
      b = []
      bugs.each { |bug|
        b << bug if bug.pkg_name == pkg
      }
      b
    end

    def pinned(pkgs, cur_pkgs, bugs)
      holdstr = ""
      pkgs.each { |pkg|
        pin_ver = "0.no.version"
        pin_pri = @config.pin_priority
        if cur_pkgs[pkg] != nil
	  pin_ver = cur_pkgs[pkg]['version']
        else
          pin_ver = "*"
          pin_pri = "-30000"
	end
        holdstr << "\nExplanation: Pinned by apt-listbugs at #{Time.now}"
        bugs_of_pkg( bugs, pkg ).each { |bug|
          holdstr << "\nExplanation:   ##{bug.bug_number}: #{bug.desc}"
        }
        holdstr << "\nPackage: #{pkg}\nPin: version #{pin_ver}"
        holdstr << "\nPin-Priority: #{pin_pri}\n"
      }
      $stderr.puts holdstr if $DEBUG
      if holdstr != ""
        File.open("/etc/apt/preferences", "a") { |io|
          io.puts holdstr
	  @config.frontend.puts sprintf(_("%s pinned by adding Pin preferences in /etc/apt/preferences. Restart APT session to enable"), pkgs.join(' '))
	  return pkgs
        }
      end
      return nil
    end

    def on_hold (pkgs)
      holdstr = ""
      pkgs.each { |pkg|
        holdstr << "#{pkg} hold\n"
      }
      if system("echo '#{holdstr}' | dpkg --set-selections")
        @config.frontend.puts sprintf(_("%s held. Restart APT session to enable"), pkgs.join(' '))
        return pkgs
      end
      return nil
    end

    def display_bugs(bugs, pkgs, cur_pkgs, new_pkgs)
      # routine to display every bug that is available and relevant

      p_bug_numbers = []
      bugs_statistics = {}
      @config.stats.each { |stat|
	@config.severity.each { |severity|
	  pkgs.each { |pkg|
	    bug_exist = 0
	    bugs_statistics[pkg] = 0 unless bugs_statistics[pkg]
	    bugs.each_by_category(pkg, severity, stat) { |bug|
	      next if p_bug_numbers.include?(bug.bug_number)
	      bugs_statistics[pkg] += 1
	      p_bug_numbers << bug.bug_number
	      if bug_exist == 0
                # TRANSLATORS: %{sevty} is the severity of some of the bugs found for package %{packg}.
                buf = _("%{sevty} bugs of %{packg} (") % {:sevty => severity,
                                                          :packg => pkg}
		buf += "#{cur_pkgs[pkg]['version']} " if cur_pkgs[pkg] != nil
		buf += "-> #{new_pkgs[pkg]['version']}) <#{@config.statmap(bug.stat)}>"
		@config.frontend.puts buf
		bug_exist = 1
	      end
	      bug_str = " ##{bug.bug_number} - #{bug.desc}"
              bug_str += sprintf(_(" (Found: %s)"), "#{bug.found}") if ( ! bug.found.nil? ) && $DEBUG
              bug_str += sprintf(_(" (Fixed: %s)"), "#{bug.fixed}") if ! bug.fixed.nil?
	      @config.frontend.puts bug_str
	      if bug.mergeids.size > 0
		bug_str =  _("   Merged with:").dup()
		bug.mergeids.each { |m|
		  bug_str << " #{m}"
		  p_bug_numbers << m
		}
		@config.frontend.puts bug_str
	      end
	    }
          }
        }
      }
      stat_str_ary = []
      bugs_statistics.each { |pkg, num|
	if num > 0
          # TRANSLATORS: %{nbugs} is the number of bugs found for package %{packg}.
          buf = ngettext("%{packg}(%{nbugs} bug)",
                         "%{packg}(%{nbugs} bugs)", num) % {:packg => pkg,
                                                            :nbugs => num}
	  stat_str_ary << buf
	end
      }
      if stat_str_ary.size > 0
	@config.frontend.puts _("Summary:\n ") + stat_str_ary.join(', ')
	return true
      else
        return false
      end
    end

    def each_state_table(o, bugs, stats)
      stats.each { |stat|
	sub = bugs.sub("stat", stat)
	if sub.size > 0
	  o.puts "<table border=2 width=100%>"
          # TRANSLATORS: %s is a bug status such as forwarded, done, pending, pending-fixed, etc. see the -S option in the man page.
          o.puts sprintf(" <caption>" + _("Bug reports which are marked as %s in the bug tracking system") + "</caption>", stat)
          o.puts " <tr><th>" + _("package") + "</th><th>" + _("severity") + "</th><th>" + _("bug number") + "</th><th>" + _("description") + "</th></tr>"
	  yield sub
	  o.puts "</table><br>"
	end
      }
    end

    def display_bugs_as_html(bugs, pkgs, cur_pkgs, new_pkgs)
      bug_exist_for_stat = 0
      bug_exist_for_pkg = 0
      bug_exist = 0
      displayed_pkgs = []

      tmp = Tempfile.new(["apt-listbugs", ".html"])
      tmp.chmod(0644)
      tmp.puts "<html><head><title>" + _("Critical bugs for your upgrade") + "</title><meta http-equiv=\"Content-Type\" content=\"text/html; charset=#{Locale.charset}\"></head><body>"
      tmp.puts "<h1 align=\"center\">" + _("Critical bugs for your upgrade") + "</h1>"
      tmp.puts "<p align=\"right\">" + _("by apt-listbugs") + "</p><hr>"
      tmp.puts "<h2>" + _("Bug reports") + "</h2>"

      each_state_table(tmp, bugs, @config.stats) { |bugs|
	bugs.each { |bug|
	  tmp.puts "<tr><td>#{bug.pkg_name}</td><td>#{bug.severity}</td><td><a href=\"http://bugs.debian.org/cgi-bin/bugreport.cgi?archive=no&bug=#{bug.bug_number}\">##{bug.bug_number}</a></td><td>#{bug.desc}</td></tr>"
	  displayed_pkgs << bug.pkg_name if !displayed_pkgs.include?(bug.pkg_name)
        }
      }

      tmp.puts "<h2>" + _("Package upgrade information in question") + "</h2>"
      tmp.puts "<ul>"
      displayed_pkgs.each { |pkg|
	tmp.puts "<li>#{pkg}("
	tmp.puts "#{cur_pkgs[pkg]['version']} " if cur_pkgs[pkg] != nil
	tmp.puts "-&gt; #{new_pkgs[pkg]['version']}" if new_pkgs[pkg] != nil
	tmp.puts ")"
      }
      tmp.puts "</ul>"

      tmp.puts "</body></html>"
      tmp.close

      puts "Invoking browser for #{tmp.path}" if $DEBUG
      browsercommandline = "#{@config.browser} #{tmp.path} < /dev/tty"
      if system(browsercommandline)
        puts "successfully invoked browser" if $DEBUG
      else
        $stderr.puts _("W: Failed to invoke browser.")
        $stderr.puts " #{browsercommandline}"
      end
      clear_stdin
    end

    private
    def clear_stdin(parent = true)
      fd = @config.frontend.tty
      flags=fd.fcntl(Fcntl::F_GETFL)
      if parent
        while clear_stdin(false)
          nil
        end
      else
        begin
          fd.read_nonblock(10000000)
          return true
        rescue Errno::EAGAIN
          return false
        ensure
          fd.fcntl(Fcntl::F_SETFL, flags)
        end
      end
    end
  end


  class RSSViewer < Viewer

    def initialize(config)
      super(config)
    end

    def encode(str)
      buf = str.gsub("<", "&lt;")
      buf
    end

    def view(new_pkgs, cur_pkgs, bugs)
      rss = RSS::Maker.make("2.0") { |maker|
        maker.channel.about = ""
        maker.channel.title = @config.title
        maker.channel.description = @config.title
        maker.channel.link = "http://bugs.debian.org/"

        bugs.each { |bug|
          if @config.stats.include?( bug.stat )
            item = maker.items.new_item
            item.link = "http://bugs.debian.org/cgi-bin/bugreport.cgi?archive=no&amp;bug=#{bug.bug_number}"
	    item.title = encode("Bug##{bug.bug_number}: #{bug.desc}")
	    item.date = Time.parse("#{bug.time.year}/#{bug.time.month}/#{bug.time.day} #{bug.time.hour}:#{bug.time.min}:#{bug.time.sec}")

            buf = ""

            buf << "<ul>\n"
            buf << "<li>Bug##{bug.bug_number}</li>\n"
            buf << "<li>Package: #{bug.pkg_name}</li>\n"
            buf << "<li>Severity: #{bug.severity}</li>\n"
            buf << "<li>Status: #{bug.stat}</li>\n"
            buf << "<li>Tags: #{bug.tags.join(',')}</li>\n" if bug.tags != nil

            if bug.mergeids.size > 0
              buf << "<li>Merged with:\n"
              bug.mergeids.each { |id|
                url  = "http://bugs.debian.org/cgi-bin/bugreport.cgi?archive=no&amp;bug=#{id}"
                buf << "<a href=\"#{url}\">#{id}</a>"
              }
            end
            buf << "</li>\n"
            buf << "</ul>\n"

            item.description = buf

	  end
        }

      }
      @config.frontend.puts rss.to_s
    end

  end

end


module Factory

  CONCURRENCY_LEVEL = 3

  def done?(done)
    _("Done") == done
  end

  def config
      @@config
  end

  def config=(c)
      @@config = c
  end

  def create(arg, *args)
    raise _("Not Implemented")
  end

  module_function :config, :config=, :create, :done?
  public :create

  module BugsFactory
    extend Factory

    def delete_ignore_pkgs(new_pkgs)
      new_pkgs.delete_if { |name, pkg|
	config.system_ignore_bugs.include?(name)
      }
    end

    def create(new_pkgs, *args, &progress)
      cur_pkgs = args[0]
      bugs = Debian::Bugs.new
      pkg_step = 100 / new_pkgs.size.to_f
      retrycount = 10 # retry 10 times

      size = new_pkgs.size
      mutex = Mutex.new
      threads = []
      yield _("Retrieving bug reports..."), "0%"
      begin
        # obtain a list of package names
        tmppkgs = []
        new_pkgs.each_key { |k| tmppkgs << k }

        # send the list of package names and severity to be parsed.
        bugs = config.parser.parse(tmppkgs, config.severity) { |pct|
          yield _("Retrieving bug reports..."), pct
        }
      rescue SOAP::HTTPStreamError => exception
        config.frontend.puts _(" Fail")
        config.frontend.puts " Exception: " + exception.class.to_s if $DEBUG
        $stderr.puts _(" E: HTTP GET failed")
        retrycount -= 1
        retry if config.frontend.yes_or_no?(_("Retry downloading bug information?")) && retrycount > 0
        raise _("Exiting with error") if ! config.frontend.yes_or_no?(_("Continue the installation anyway?"), false)
        bugs = []
      rescue SOAP::EmptyResponseError => exception
        config.frontend.puts _(" Fail")
        config.frontend.puts " Exception: " + exception.class.to_s if $DEBUG
        $stderr.puts _(" E: Empty stream from SOAP")
        retrycount -= 1
        retry if config.frontend.yes_or_no?(_("Retry downloading bug information?")) && retrycount > 0
        raise _("Exiting with error") if ! config.frontend.yes_or_no?(_("Continue the installation anyway?"), false)
        bugs = []
      rescue Exception => exception
        config.frontend.puts _(" Fail")
        config.frontend.puts " Exception: " + exception.class.to_s if $DEBUG
        config.frontend.puts _("Error retrieving bug reports from the server with the following error message:")
        config.frontend.puts " W: #{$!}"
        if exception.kind_of? SocketError
          config.frontend.puts _("It appears that your network connection is down. Check network configuration and try again")
        else
          config.frontend.puts _("It could be because your network is down, or because of broken proxy servers, or the BTS server itself is down. Check network configuration and try again")
        end
        retrycount -= 1
        retry if config.frontend.yes_or_no?(_("Retry downloading bug information?")) && retrycount > 0
        raise _("Exiting with error") if ! config.frontend.yes_or_no?(_("Continue the installation anyway?"), false)
        bugs = []
      end
      yield _("Retrieving bug reports..."), "100%"

      if block_given?
        yield _("Retrieving bug reports..."), _("Done")
      end
      bugs
    end

    def delete_ignore_bugs(bugs)
      # ignoring ignore_bugs
      bugs.delete_if { |bug| config.system_ignore_bugs.include?(bug.bug_number)}
    end

    def delete_regexp_bugs(bugs, regexp)
      puts "Ignoring regexp: #{regexp}" if $DEBUG
      bugs.delete_if {|bug| bug.desc =~ /#{config.ignore_regexp}/}
    end

    def delete_uninteresting_bugs(bugs)
      # ignoring all bugs but the requested ones
      bugs.delete_if { |bug| !config.fbugs.include?(bug.bug_number)}
    end

    def iterate_fixed_found_version(bts_versions, pkg_name)
      # iterate relevant versions, used to parsing Fixed and Found tags of BTS
      if bts_versions.nil?
        return;
      end
      bts_versions.split(" ").each { |version|
        # check each fixed_version
        case version
        when /^(.*)\/(.*)$/
          if $1 == pkg_name # TODO: actually, this need to be source_name
            yield $2
          else
            # TODO: ignore this until I figure out how to get source_name instead of pkg_name
            #fixed_ver=nil
            yield $2
          end
        else
          yield version
        end
      }
    end

    def find_max_version_below_ver(bts_versions, new_ver, pkg_name)
      # find the max version from found/fixed that is below or equal to new_ver
      # data format of bts_versions:
      # space-delimited sequence of PACKAGE/VERSION or VERSION items.

      maxver=nil
      iterate_fixed_found_version(bts_versions, pkg_name) { |each_ver|
        # check each fixed_ver
        if Debian::Dpkg.compare_versions(each_ver, "le", new_ver) &&
            ( maxver == nil || Debian::Dpkg.compare_versions(maxver, "le", each_ver) )
          maxver = each_ver
        end
      }
      maxver
    end

    def find_min_version_above_ver(bts_versions, new_ver, pkg_name)
      # find the min version from found/fixed that is strictly above new_ver
      # data format of bts_versions:
      # space-delimited sequence of PACKAGE/VERSION or VERSION items.

      minver=nil
      iterate_fixed_found_version(bts_versions, pkg_name) { |each_ver|
        # check each each_ver
        if Debian::Dpkg.compare_versions(each_ver, "gt", new_ver) &&
            ( minver == nil || Debian::Dpkg.compare_versions(minver, "ge", each_ver) )
          minver = each_ver
        end
      }
      minver
    end

    def am_i_buggy(ver, fixed, found)
      # find out if this version is buggy or not depending on the
      # fixed / found arrays.

      puts " .. checking ver #{ver} against fixed:#{fixed} found:#{found}" if $DEBUG

      fixed_max_below = nil
      found_max_below = nil

      fixed_min_above = nil
      found_min_above = nil

      fixed_max_below = find_max_version_below_ver(fixed, ver, name) if ! fixed.nil?
      found_max_below = find_max_version_below_ver(found, ver, name) if ! found.nil?

      fixed_min_above = find_min_version_above_ver(fixed, ver, name) if ! fixed.nil?
      found_min_above = find_min_version_above_ver(found, ver, name) if ! found.nil?

      val=true

      if fixed_max_below == nil && found_max_below == nil
        # this is a hard thing to handle, but do some guessing here...
        # the bug was not fixed or found before this version:
        # it either means it wasn't found before this version,
        # or 'found' version info is missing from BTS
        if found_min_above == nil
          # no new version found;
          # which I guess means that the BTS is missing version info
          puts " ... no found info: I guess I am buggy" if $DEBUG
          val=true
        else
          # found_min_above is not nil, which means I may not be buggy;
          # except for a case where it's fixed before the found_min,
          # which probably means 'found' info is missing from BTS
          if fixed_min_above == nil ||
              Debian::Dpkg.compare_versions(fixed_min_above, "gt", found_min_above)
            puts " ... bug found in a later version: I guess I am not buggy" if $DEBUG
            val=false
          else
            puts " ... bug fixed in a later version before it's found again: I guess I am buggy" if $DEBUG
            val=true
          end
        end
      else if fixed_max_below == nil
             # the bug was found before (or in) this version, but fixed
             # later (or never): it means I am buggy
             puts " ... bug found in a prior version, but not yet fixed: I am buggy" if $DEBUG
             val=true
           else if found_max_below == nil || Debian::Dpkg.compare_versions(fixed_max_below, "gt", found_max_below)
                  # the bug was not found between the latest fixed version
                  # and this version: it means I am not buggy
                  puts " ... bug not found between the latest fixed version and this version: I am not buggy" if $DEBUG
                  val=false
                else
                  # the bug was indeed found between the latest fixed version
                  # and this version: it means I am buggy
                  puts " ... bug found between the latest fixed version and this version: I am buggy" if $DEBUG
                  val=true
                end
           end
      end

      val
    end

    def bug_is_irrelevant(name, cur_ver, new_ver, bug_number, fixed, found, bug_stat="")
      # find out if the bug number is irrelevant for this specific upgrade, from fixed/found information.
      # @return false: bug is relevant, true: bug is irrelevant, should be removed.
      val = false

      puts "Start checking: #{bug_number}" if $DEBUG

      # ignore bugs that have no fixed version info, and are closed with a XXXX-done
      if fixed.nil? && bug_stat == "done"
        puts "bug apparently closed with XXXX-done without version info" if $DEBUG
        val = true
      else if new_ver.nil?
             # no specific version was given for the package, which means that we are interested in all (non archived) bugs
             puts "no package version given" if $DEBUG
             val = false
           else if cur_ver.nil?
                  # no known installed version, which means that we want to check the new version
                  val = true if ! am_i_buggy(new_ver, fixed, found)
                else
                  # both versions are known, which means that we want to check whether the upgrade may introduce this bug into the system
                  val = true if am_i_buggy(cur_ver, fixed, found) || ( ! am_i_buggy(new_ver, fixed, found))
                end
           end
      end

      puts "in conclusion, for bug #{bug_number} comparing fixed:[#{fixed}], found:[#{found}], cur_ver:#{cur_ver} and new_ver:#{new_ver} results in removal:#{val}" if $DEBUG
      val
    end

    def delete_irrelevant_bugs (bugs, cur_pkgs, new_pkgs)
      # Ignore bugs that do not apply to the installing version.

      max=bugs.size
      step=(max/100)*10+1
      i=0
      yield _("Parsing Found/Fixed information..."), "0%"

      bugs.delete_if { |bug|
	val = false
	name = bug.pkg_name
	new_ver = nil
	cur_ver = nil
	new_ver = new_pkgs[name]["version"] if new_pkgs[name] != nil
	cur_ver = cur_pkgs[name]["version"] if cur_pkgs[name] != nil

        # show progress
        yield _("Parsing Found/Fixed information..."),
        "#{(i.to_f/max.to_f*100).to_i}%" if (i % step) == 0
        i += 1

        val = true if bug_is_irrelevant(name, cur_ver, new_ver,
                                        bug.bug_number, bug.fixed, bug.found, bug.stat)
        val
      }
      yield _("Parsing Found/Fixed information..."), "100%"
      yield _("Parsing Found/Fixed information..."), _("Done")
      bugs
    end

    def delete_unwanted_tag_bugs( bugs )
      puts "checking unwanted bugs: #{bugs}" if $DEBUG
      bugs.delete_if { |bug|
        val = false
        puts "#{bug}" if $DEBUG
        config.tag.each { |tag|
	  if bug.tags && bug.tags.include?( tag )
            puts "#{bug} has {tag}" if $DEBUG
	  else
	    val = true
	  end
	}
	val
      }
    end

    module_function :delete_ignore_pkgs, :create, :delete_ignore_bugs,
    :delete_uninteresting_bugs,
    :delete_regexp_bugs,
    :bug_is_irrelevant,
    :am_i_buggy,
    :delete_irrelevant_bugs, :delete_unwanted_tag_bugs,
    :find_max_version_below_ver,
    :find_min_version_above_ver,
    :iterate_fixed_found_version

  end

end

class ConsoleFrontend

  def initialize( config )
    @tty = nil
    @old = ""
    @config = config
  end

  def progress(msg, val)
    $stderr.print "\r"
    $stderr.print " " * @old.length
    $stderr.print "\r"
    @old = "#{msg} #{val}"
    $stderr.print @old
    $stderr.flush
    $stderr.puts "" if Factory.done?(val)
  end

  def puts(msg)
    $stdout.puts msg
  end


  def tty
    @tty ||= open("/dev/tty")
  end

  def ask(msg)
    $stdout.print "#{msg} "
    $stdout.flush
    line = nil
    line = self.tty.gets
    if line != nil
      line.chomp!
    end
    return line
  end

  def yes_or_no?(msg, default = true)
    return @config.yes if ! @config.yes.nil?
    while true
      msgyn = "#{msg}"
      if default == true
	msgyn << " [Y/n]"
      else
	msgyn << " [y/N]"
      end
      a = ask msgyn
      if a == ""
	return default
      elsif a == "Y" || a == "y"
	return true
      elsif a == "N" || a == "n"
	return false
      end
    end
  end

  def close
    @tty.close if @tty
  end
end

### ;;;
### Local Variables: ***
### dancer-test-run-chdir: "../.." ***
### End: ***
