/* apt-spy (c) Steven Holmes, 2003. This software is licensed as detailed in the LICENSE file. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <libgen.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "include/global.h"
#include "include/file.h"

/* Selects an infile */
FILE *select_infile(char *infile)
{
	if (infile != NULL)
		return fopen(infile, "r");
	
	/* Else return a temporary file */
	return tmpfile();
}

/* Selects an output file */

FILE *select_outfile(char *outfile)
{
	FILE *fp;

	/* Check if we've to use standard output... */
	if (strcmp(outfile, "-") == (int) NULL)
		return stdout;

	/* Otherwise, we open the file */
	else {
		if (backup(outfile) == 1)	/* backup old file */
			return NULL;

		fp = fopen(outfile, "w");
	}
	return fp;
}

/* Open configuration file */

FILE *select_config(char *config_file)
{
	FILE *fp;
	
	if (config_file == NULL)
		config_file = d_config;
	
	fp = fopen(config_file, "r");
	
	return fp;
}

FILE *select_mirror(char *mirror_list, int are_updating)
{
	FILE *fp;
	
	if (mirror_list == NULL)
		mirror_list = d_mirror;
	
	if (are_updating == 1)
		fp = fopen(mirror_list, "w");
	else
		fp = fopen(mirror_list, "r");
	
	return fp;
}

char *next_entry(FILE *infile_p)
{
	char *temp;
	unsigned long orig_position, buffsize;
	int c;

	/* Save original file position */
	orig_position = ftell(infile_p);

	/* Fast-forward to new line */
	while ((c = getc(infile_p)) != '\n') {
		if (c == EOF)
			return NULL;		/* We've exhausted the list */
	}

	buffsize = ftell(infile_p) -  orig_position;	/* Calculate buffer size */
	
	/* create storage space for line */
	temp = malloc(buffsize + 1);

	if (temp == NULL) {
		perror("malloc");
		exit(1);
	}

	/* Rewind file to original position */
	fseek(infile_p, orig_position, SEEK_SET);

	/* We now read the line into the newly created buffer */
	fgets(temp, buffsize + 1, infile_p);

	return temp;
}

/* If orig_file exists, copy it to a new file with ".bak" appended */
/* Upon error, "1" is returned*/

int backup(char *orig_file)
{
	FILE *in_file, *out_file;
	char *new_name;
	int c, namesize;
	struct stat file_stat;

	if (stat(orig_file, &file_stat) == -1) {
		if (errno == ENOENT)
			return 0;	/* doesn't exist */
		else {
			perror("stat");		/* or there's been an error */
			return 1;
		}
	}

	if (file_stat.st_size == 0)
		return 0;		/* Don't backup empty file */

	in_file = fopen(orig_file, "r");
	
	if (in_file == NULL) {
		perror("fopen");	/* eep! */
		return 1;
	}

	/* Saves multiple calls to strlen() later */
	namesize = strlen(orig_file);

	/* We need to create the new filename. */
	new_name = (char *) malloc(namesize + 5);
	strncpy(new_name, orig_file, namesize + 1);

	/* We append ".bak" to the end of the backup file */
	strncat(new_name, ".bak", namesize + 5);

	out_file = fopen(new_name, "w");
	
	if (out_file == NULL)
		return 1;
	
	/* Do the copying */
	while((c = getc(in_file)) != EOF)
		putc(c, out_file);

	if (ferror(in_file) || ferror(out_file)) {
		perror("putc");
		return 1;
	}		

	fclose(in_file);
	fclose(out_file);
	
	free(new_name);
	return 0;
}

int check_write_access(char *path)
{
	/* Thanks to Brian from warner-debbugs@lothar.com for pointing out and
	   fixing the programming error here. */

	char *dir = strdup(path);		/* Leak it, it's small */

	if (access(path, R_OK | W_OK) == -1) {
		if (access(dirname(dir), R_OK | W_OK) == -1) {
			perror("access");
			return 1;
		}
		else if (errno == ENOENT)
			return 0;		/* Doesn't exist */
		else {
			perror("access");	/* Error */
			return 1;
		}
	}
	return 0;
}
