// config_signal.h			-*-c++-*-
//
//  A framework for signals that are based on the apt configuration
// code.  This uses a thin wrapper around the apt Configuration class.
//
//  This works in large part because apt-pkg itself does not (thankfully)
// modify configuration items after it is initialized.
//
//  Important note: this configuration class supports "transparent themes".
//  If PACKAGE::Theme is set, Find*(key) operations will look up key,
// followed by PACKAGE::Themes::(PACKAGE::theme)::key.
//
//  This only breaks for things that use Tree -- which means that such code
// might need to explicitly understand themes in order to work.  That is: if
// it is appropriate for two disparate hierarchies to be "merged", you have to
// do this on your own.  The default Tree call simply uses the theme as
// a backing store.
//
//  IMPORTANT NOTE: setting values in the PACKAGE::Themes hierarchy will
// not trigger signal emission on "shadowing" nodes!  Because themes are
// expected to be static, I do not think this is a big deal, but you should
// be aware of it.
//
//  MORE IMPORTANT NOTE: Altering the value of PACKAGE::Theme will
// not trigger ANY signal emission right now.  (this *will* be a nuisance
// and I hope to change it eventually)

#ifndef CONFIG_SIGNAL_H
#define CONFIG_SIGNAL_H

#include <apt-pkg/configuration.h>
#include <sigc++/basic_signal.h>
#include <sigc++/slot.h>

#ifdef HAVE_CONFIG_H
#include "../../config.h"
#endif

#ifdef HAVE_HASH_MAP
#include <hash_map>
#else
#ifdef HAVE_EXT_HASH_MAP
#include <ext/hash_map>
#else
// Fallback to the non-hashing map class
#include <map>
#define hash_map map
#endif
#endif

#include <vector>

#include "strhash.h"

class signalling_config:public SigC::Object
{
  // Having two configuration objects is a workaround for a bug in apt;
  // the long configuration-item names were causing problems with the
  // http fetcher.
  Configuration *real_config, *theme_config;

  typedef std::hash_map<string, SigC::Signal1<void, string> *> connhash;

  connhash conn_table;

  // The current "theme" root, cached.  (ie, if the theme is "bland",
  // this will be "PACKAGE::UI::Themes::bland::")
  std::string themeroot;
  void update_theme(std::string theme);
public:
  signalling_config(Configuration *_real_config,
		    Configuration *_theme_config);
  ~signalling_config();

  inline string Find(const char *Name,const char *Default = 0)
  {
    if(themeroot.empty() || real_config->Exists(Name))
      return real_config->Find(Name, Default);
    else
      return theme_config->Find(themeroot+Name, Default);
  }
  inline string Find(string Name,const char *Default = 0)
  {
    if(themeroot.empty() || real_config->Exists(Name))
      return real_config->Find(Name, Default);
    else
      return theme_config->Find((themeroot+Name).c_str(), Default);
  }

  inline string FindFile(const char *Name,const char *Default = 0)
  {
    if(themeroot.empty() || real_config->Exists(Name))
      return real_config->FindFile(Name, Default);
    else
      return theme_config->FindFile((themeroot+Name).c_str(), Default);
  }

  inline string FindDir(const char *Name,const char *Default = 0)
  {
    if(themeroot.empty() || real_config->Exists(Name))
      return real_config->FindDir(Name, Default);
    else
      return theme_config->FindDir((themeroot+Name).c_str(), Default);
  }

  inline int FindI(const char *Name,int Default = 0)
  {
    if(themeroot.empty() || real_config->Exists(Name))
      return real_config->FindI(Name, Default);
    else
      return theme_config->FindI(themeroot+Name, Default);
  }

  inline int FindI(string Name,bool Default = 0)
  {
    if(themeroot.empty() || real_config->Exists(Name))
      return real_config->FindI(Name, Default);
    else
      return theme_config->FindI(themeroot+Name, Default);
  }

  inline bool FindB(const char *Name,bool Default = false)
  {
    if(themeroot.empty() || real_config->Exists(Name))
      return real_config->FindB(Name, Default);
    else
      return theme_config->FindB(themeroot+Name, Default);
  }

  inline bool FindB(string Name,bool Default = false)
  {
    if(themeroot.empty() || real_config->Exists(Name))
      return real_config->FindB(Name, Default);
    else
      return theme_config->FindB(themeroot+Name, Default);
  }

  void Set(string Name,string Value);
  void Set(const char *Name,string Value);
  void Set(const char *Name,int Value);

  Configuration *setcfg(Configuration *newcfg);
  // Switches to a different configuration tree (this will involve emitting
  // various signals)  Returns the old configuration tree.

  inline bool Exists(string Name)
  {
    if(real_config->Exists(Name))
      return true;
    else
      return !themeroot.empty() && theme_config->Exists(themeroot+Name);
  }

  bool Exists(const char *Name)
  {
    if(real_config->Exists(Name))
      return true;
    else
      return !themeroot.empty() && theme_config->Exists(themeroot+Name);
  }
      
  inline const Configuration::Item *Tree(const char *Name)
  {
    if(themeroot.empty() || real_config->Exists(Name))
      return real_config->Tree(Name);
    else
      return theme_config->Tree((themeroot+Name).c_str());
  }

  // FIXME: this is a hack needed to support the workaround.  (for load_config)
  //        I think.
  Configuration *get_cfg(bool theme)
  {
    return theme?theme_config:real_config;
  }

  void connect(string name, const SigC::Slot1<void, std::string> &slot);
  void connectI(string name, const SigC::Slot1<void, int> &slot);
  void connectB(string name, const SigC::Slot1<void, bool> &slot);

  void Dump()
  {real_config->Dump();}
};

#endif
