# 
# indicated-revision.sh - figure out what revision is indicated
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "figure out what revision is indicated\\n"
                printf "usage: indicated-revision [options] [archive/]revision\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
		printf " -e --errname errname           specify a name for error messages\\n"
                printf "\\n"
                printf "Parse the REVISION argument and print these values:\\n"
                printf "\\n"
                printf "	\"%%s/%%s\" \$archive \$revision\\\n\\n"
                printf "\\n"
                printf "ARCHIVE may be omitted.  REVISION may be a version name or\\n"
                printf "branch name.  The output will be a complete archive/revision\\n"
		printf "name\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
errname=indicated-revision

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "indicated-revision: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "indicated-revision: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -e|--errname)      	shift
                        if test $# -eq 0 ; then
                          printf "indicated-revision: --errname require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        errname="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;

    -*)                 printf "indicated-revision: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: indicated-revision [options] [archive/]version\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

spec="$1"
shift

################################################################
# Validate Argument and Compute Implied Revision
# 

larch valid-package-name -e indicated-revision -t "$spec"

archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$spec\"`"
non_arch="`larch parse-package-name --non-arch \"$spec\"`"
category="`larch parse-package-name --basename \"$spec\"`"
branch="`larch parse-package-name --package \"$spec\"`"

################################################################
# Is This a Trivial Case?
# 

if larch valid-package-name --lvl "$non_arch" ; then
  printf "%s\\n" "$archive/$non_arch" 
  exit 0
fi


################################################################
# Ensure an Archive Connection
# 

if test "$WITHARCHIVE" != "$archive" ; then
  exec larch with-archive -R "$archroot" -A "$archive" \
      larch indicated-revision -e "$errname" "$archive/$non_arch"
fi

################################################################
# Which Revision?
# 

if larch valid-package-name --vsn "$non_arch" ; then
  #
  # Latest Revision in a Specific Version
  # 
  version="`larch parse-package-name --package-version \"$non_arch\"`"
  lvl="`larch revisions -r \"$archive/$version\" | head -1`"
  if test -z "$lvl" ; then
    printf "%s: no revisions in version\\n" "$errname" 1>&2
    printf "  archive: %s\\n" "$archive" 1>&2
    printf "  version: %s\\n" "$version" 1>&2
    printf "\\n" 1>&2
    exit 1
  fi
  printf "%s\\n" "$archive/$version--$lvl"
  exit 0
else
  #
  # Search for the Latest Revision in the Highest Numbered Version
  # 
  for version in "`larch versions -r \"$archive/$branch\"`" ; do
    lvl="`larch revisions -r \"$archive/$version\" | head -1`"
    if test ! -z "$lvl" ; then
      printf "%s\\n" "$archive/$version--$lvl"
      exit 0
    fi
  done
  printf "%s: no revisions in any version\\n" "$errname" 1>&2
  printf "  archive: %s\\n" "$archive" 1>&2
  printf "  branch: %s\\n" "$branch" 1>&2
  printf "\\n" 1>&2
  exit 1
fi
