# valid-config-name.sh: test a config name for validity
# 
################################################################
# Copyright (C) 2002 Jonathan Geisler
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# Special Options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "test a config name for validity\\n"
		printf "usage: valid-config-name [options] [--] config\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -e --errname prog             specify program name for errors\\n"
		printf "\\n"
		printf " --exists                      ensure that the config file exists\\n"
		printf " --new                         ensure that the config file does\\n"
		printf "                                not exist\\n"
		printf " -f --force                    suppress the check implied by --new\\n"
		printf " --dir DIR                     search the configs directory of the\\n"
		printf "                                projects tree containing DIR.\\n"
		printf " --config-dir CFG              search directory CFG for config files\\n"
		printf "\\n"
		printf "Exit with status 0 if CONFIG is a valid config name,\\n"
		printf "status 1 otherwise.\\n"
		printf "\\n"
		printf "A valid CONFIG name matches the pattern:\\n"
		printf "\\n"
		printf "    [a-zA-Z][-.@a-zA-Z0-9]*(/[a-zA-Z][-.@a-zA-Z0-9]*)*\\n"
		printf "\\n"
		printf "With \"--exists\", also ensure that a config file with that\\n"
                printf "name exists.  By default, search the \"configs\" directory at the\\n"
                printf "root of the project tree containing DIR (or the current\\n"
                printf "directory).  The option \"--config-dir\" can be used to specify an\\n"
                printf "alternative configs directory.\\n"
                printf "\\n"
                printf "With \"--new\", but without \"--force\", ensure that the config\\n"
                printf "file does not already exist.  With both options, the existence\\n"
                printf "or nonexistence of the file is not checked.\\n"
		printf "\\n"
		printf "If an error name is specified (-e or --errname), then invalid\\n"
		printf "names cause an error message on stdout.  Otherwise, the exit\\n"
		printf "status is the only output.\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

errname=
dir=.
config_dir=
exists=
new=
force=

while test $# -ne 0 ; do

  case "$1" in 
    --)				shift
    				break
    				;;

    -e|--errname)		shift
				if test $# -eq 0 ; then
				  printf "valid-config-name: -e and --errname require an argument\\n" 1>&2
				  printf "try --help\\n" 1>&2
				  exit 1
				fi
				errname="$1"
				shift
				;;

    --exists)                   exists=--exists
				new=
				shift
				;;
    
    --new)                      new=--new
				exists=
				shift
				;;

    -f|--force)                 force=--force
				shift
				;;

    --dir)                      shift
				if test $# -eq 0 ; then
				  printf "valid-config-name: --dir requires an argument\\n" 1>&2
				  printf "try --help\\n" 1>&2
				  exit 1
				fi
				dir="$1"
				shift
				;;

    --config-dir)               shift
				if test $# -eq 0 ; then
				  print "valid-config-name: --config-dir requires an argument\\n" 1>&2
				  print "try --help\\n" 1>&2
				  exit 1
				fi
				config_dir="$1"
				shift
				;;

    -*)				printf "valid-config-name: unrecognized option (%s)\\n" "$1" 1>&2
				printf "try --help\\n" 1>&2
				exit 1
				;;

    *)				break
    				;;
  esac

done


################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: valid-config-name [options] config\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

config="$1"

################################################################
# Sanity Check and Process Defaults
#

here=`pwd`

if test ! -z "$exists" -o '(' -z "$force" -a ! -z "$new" ')' ; then
  if test -z "$config_dir" ; then

    cd "$dir"
    wdroot="`larch tree-root`"
    config_dir="$wdroot/configs"

  fi

  if test ! -d "$config_dir" ; then
    if test ! -z "$errname" ; then
      printf "\\n" 1>&2
      printf "%s: config-dir does not exist\\n" "$errname" 1>&2
      printf "  config-dir: %s\\n" "$config_dir" 1>&2
      printf "\\n" 1>&2
    fi
    exit 1
  fi

  config_file="$config_dir/$config"
fi

################################################################
# Check Config Name
# 

configre="[a-zA-Z][-.@a-zA-Z0-9]*(/[a-zA-Z][-.@a-zA-Z0-9]*)*"

set +e
filtered="`printf \"%s\\n\" \"$config\" | grep -E -e \"^($configre)\$\"`"
set -e

if test -z "$filtered" ; then
  if test ! -z "$errname" ; then
    printf "\\n" 1>&2
    printf "%s:  config name is not valid\\n" "$errname" 1>&2
    printf "  config: %s\\n" "$config" 1>&2
    printf "\\n" 1>&2
  fi
  exit 1
fi

if test ! -z "$exists" ; then
  if test ! -f "$config_file" ; then
    if test ! -z "$errname" ; then
      printf "\\n" 1>&2
      printf "%s: config file not found\\n" "$errname" 1>&2
      printf "  config: %s\\n" "$config" 1>&2
      printf "  config file directory: %s\\n" "$config_dir" 1>&2
      printf "\\n" 1>&2
    fi
    exit 1
  fi
fi

if test -z "$force" -a ! -z "$new" ; then
  if test -e "$config_file" ; then
    if test ! -z "$errname" ; then
      printf "\\n" 1>&2
      printf "%s: config file already exists\\n" "$errname" 1>&2
      printf "  config: %s\\n" "$config" 1>&2
      printf "  config file directory: %s\\n" "$config_dir" 1>&2
      printf "\\n" 1>&2
    fi
    exit 1
  fi
fi


# tag: Jonathan Geisler Fri Mar  1 19:24:22 CST 2002 (naming-conventions/valid-config-name.sh)
#


