/*
    Copyright (C) 2001 Paul Davis
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: opts.cc,v 1.13 2003/04/04 18:12:23 jthall Exp $
*/

#include <getopt.h>
#include <iostream>
#include <stdlib.h>
#include <pbd/error.h>

#include "opts.h"

#include "i18n.h"

using namespace std;

char *KSI_ARDOUR::ui_rcfile = "ksi_ui.rc";
char *KSI_ARDOUR::remote = "";
string KSI_ARDOUR::master_outs = "alsa_pcm:playback_1,alsa_pcm:playback_2";
string KSI_ARDOUR::control_outs = "alsa_pcm:playback_1,alsa_pcm:playback_2";
bool KSI_ARDOUR::with_diskstreams = false;
string KSI_ARDOUR::session_path = "./ksi-session";

using namespace KSI_ARDOUR;

int
print_help (const char *execname)

{
	cout << "Usage: " << execname << " -h\n"
	     << "  [-r HOST:PORT ] [-M MASTER-LEFT,MASTER-RIGHT ]\n"
	     << "  [-C CONTROL-LEFT,CONTROL-RIGHT ]\n\n"
	     << "  -h,  --help                      Print this message\n"
	     << "  -U,  --ui-rcfile=FILENAME        Filename for UI defs\n"
	     << "                                   [default: "
	     << ui_rcfile << "]\n"
	     << "  -r, --remote=HOST[:PORT]         use remote physical keyboard server on HOST:PORT\n"
	     << "  -M, --master-outs=LEFT,RIGHT     assign \"master outs\" (default: channels 1+2)\n"
	     << "  -C, --control-outs=LEFT,RIGHT    assign \"control room outs\" (default: off)\n"
	     << "  -D, --disk-recording             use diskstreams for recording and playback"
	     << "  session path                     load SESSION_PATH as the session [ default: "
	     << session_path << " ]\n"
		;

	return 1;
}

int
KSI_ARDOUR::parse_engine_opts (int argc, char *argv[])

{
	const char *optstring = "U:l:hr:M:C:D";
	const char *execname = strrchr (argv[0], '/');

	if (execname == 0) {
		execname = argv[0];
	} else {
		execname++;
	}

	const struct option longopts[] = {
		{ "master-outs", 1, 0, 'M' },
		{ "control-outs", 1, 0, 'C' },
		{ "ui-rcfile", 1, 0, 'U' },
		{ "help", 0, 0, 'h' },
		{ "remote", 0, 0, 'r' },
		{ "disk-recording", 0, 0, 'D' },
		{ 0, 0, 0, 0 }
	};

	int option_index = 0;
	int c = 0;

	while (1) {
		c = getopt_long (argc, argv, optstring, longopts, &option_index);

		if (c == -1) {
			break;
		}

		switch (c) {
		case 0:
			break;
		case 'M':
			master_outs = optarg;
			break;
		case 'C':
			control_outs = optarg;
			break;
		case 'U':
			ui_rcfile = optarg;
			break;
		case 'r':
			remote = optarg;
			break;
		case 'D':
			with_diskstreams = true;
			break;
		break;
		case 'h':
			print_help(execname);
			exit(0);
			break;
		default:
			error << compose(_("unknown option character code %1%2%3"), &hex, c, &dec) << endmsg;
			print_help(execname);
			exit(1);
		}

	}

	if (optind < argc) {
		session_path = argv[optind++];
	}

	return 0;
}

