
/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (C) 2000-2001 QoSient, LLC.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * rasrvstats - argus server statistics.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <rasrvstats.h>

#include <math.h>

int RaReadServicesFile(char *);
void RaProcessRecord (struct ArgusRecord *argus) { }
void RaProcessThisSrvRecord (struct ArgusServiceRecord *);
void RaProcessSrvRecord (struct ArgusServiceRecord *);
 
char *RaSrvStatsFlowModelFile [] = {
"Flow     100       *                  *                 *       *        *         200      31536000",
"Flow     101       *                  *                 *       *        *         201      31536000",
"Flow     103       *                  *                 *       *        *         203      31536000",
"Flow     104       *                  *                 *       *        *         204      31536000",
"Flow     105       *                  *                 *       *        *         205      31536000",

"Model    200   0.0.0.0             0.0.0.0             yes      no       yes",
"Model    201   0.0.0.0             255.255.255.0       yes      no       no",
"Model    203   0.0.0.0             255.255.255.255     yes      no       yes",
"Model    204   255.255.255.0       255.255.255.255     yes      no       yes",
"Model    205   255.255.255.255     255.255.255.255     yes      no       yes",
NULL,
};


int RaInitialized = 0;
struct ArgusRecordStore RaGlobalStoreBuf, *RaGlobalStore = &RaGlobalStoreBuf;

void
ArgusClientInit ()
{
   if (!(RaInitialized)) {
      RaWriteOut = 0;
   
      bzero ((char *) RaFlowArray,  sizeof(RaFlowArray));
      bzero ((char *) RaModelArray, sizeof(RaModelArray));
      bzero ((char *) RaGlobalStore, sizeof(RaGlobalStoreBuf));
   
      RaFlowModelFile = RaReadFlowModelFile (RaSrvStatsFlowModelFile);

      if (!(RaFlowModelFile))
         usage();
   
      if ((RaModelerQueue = RaNewQueue()) == NULL)
         exit(0);
   
      if ((RaSrvHashTable.array = (struct RaSrvHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaSrvHashTableHeader *))) != NULL) {
         RaSrvHashTable.size = RA_HASHTABLESIZE;
      }
   
      if ((RaHashTable.array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                    sizeof (struct RaHashTableHeader *))) != NULL) {
         RaHashTable.size = RA_HASHTABLESIZE;
      }

      RaGlobalStore->queue   =  RaModelerQueue;
      RaGlobalStore->htable  = &RaHashTable;
      RaModelerQueue->htable = &RaHashTable;

      if (ArgusFlowModelFile)
         if ((RaReadServicesFile(ArgusFlowModelFile)) < 0)
            exit(0);

      if (Hflag) {
         RaPrintTransactions++;
         RaHistoTimeSeries = ARGUS_TIMESERIES;
      }

      RaInitialized++;
   }
}

void RaSortQueue (struct RaQueueStruct *);
void RaPrintOutQueue (struct RaQueueStruct *, int);

int RaParseCompleting = 0;

#define ARGUS_MAXFLOWDEFS       5

#define ARGUS_SERVICE		0
#define ARGUS_SERVICE_SUBNET	1
#define ARGUS_SERVER		2
#define ARGUS_CLIENT_SUBNET	3
#define ARGUS_CLIENT		4

int RaTotals[ARGUS_MAXFLOWDEFS] = {0, 0, 0, 0, 0,};


void
RaParseComplete (int sig)
{
   if (!RaParseCompleting) {
      RaParseCompleting++;

      if (RaTotals[ARGUS_SERVICE] > 0) {
         printf ("%s  Total Services %d  Total Servers %d  Total Clients %d\n", ArgusProgramName,
            RaTotals[ARGUS_SERVICE], RaTotals[ARGUS_SERVER], RaTotals[ARGUS_CLIENT]);

         RaSortQueue (RaModelerQueue);
         RaPrintOutQueue (RaModelerQueue, 0);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (1, "RaParseComplete: returning\n");
#endif
}

int ArgusHourlyUpdate = 1;
int ArgusMinuteUpdate = 1;

void
ArgusClientTimeout ()
{
   RaProcessQueue (RaModelerQueue, ARGUS_STATUS);

#ifdef ARGUSDEBUG
   ArgusDebug (6, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));

   if (ArgusMinuteUpdate++ == 60) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s HashHdrs %d Hash %d Queue %d\n",
                     print_time(&ArgusGlobalTime), RaAllocHashTableHeaders,
                     RaHashTable.count, RaModelerQueue->count);
      ArgusMinuteUpdate = 1;
   }

   if (ArgusHourlyUpdate++ == 3600) {
      ArgusDebug (2, "ArgusClientTimeout: current time %s.\n", print_time(&ArgusGlobalTime));
      ArgusHourlyUpdate = 1;
   }
#endif
}

void
parse_arg (int argc, char**argv)
{ 
}

void
usage ()
{
   fprintf (stderr, "Rasrvstats Version %d.%d\n", major_version, minor_version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [ra-options] [- filter-expression]\n", ArgusProgramName);

   fprintf (stderr, "options: -b             dump packet-matching code.\n");
   fprintf (stderr, "         -C             treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -d <level>     specify debug level\n");
   fprintf (stderr, "         -F <conffile>  read configuration from <conffile>.\n");
   fprintf (stderr, "         -h             print help.\n");
   fprintf (stderr, "         -m             print MAC addresses.\n");
   fprintf (stderr, "         -n             don't convert numbers to names.\n");
   fprintf (stderr, "         -P <portnum>   specify remote argus <portnum> (tcp/561).\n");
   fprintf (stderr, "         -r <file>      read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -S <host>      specify remote argus <host>.\n");
   fprintf (stderr, "         -t <timerange> specify <timerange> for reading records.\n");
   fprintf (stderr, "               format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                        timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                            mm/dd[/yy]\n");
   fprintf (stderr, "         -T <secs>      attach to remote server for T seconds.\n");
   fprintf (stderr, "         -u             print time in Unix time format.\n");
   fprintf (stderr, "         -U <digits>    print fractional time with this precision.\n");
   fprintf (stderr, "         -w <file>      write output to <file>. '-' denotes stdout.\n");
   exit(1);
}




void
RaProcessThisSrvRecord (struct ArgusServiceRecord *srv)
{
   if (srv->status == RA_SVCPASSED)
      RaProcessSrvRecord(srv);
}


extern int RaFlowMajorModified;

void
RaProcessSrvRecord (struct ArgusServiceRecord *srv)
{
   struct ArgusRecord *argus = srv->argus;
   struct ArgusRecord *targus = NULL;

   struct ArgusRecordStore *RaThisPrvStore = RaGlobalStore;
   struct ArgusRecordStore *store;
   struct RaPolicyStruct *rap;
   int RaTimeout = -1, i, index = 0;

   RaThisActiveDuration = RaGetActiveDuration(argus);
   RaFlowMajorModified = 1;

   if (Hflag) {
      RaThisActiveIndex = (RaThisActiveDuration < 100)        ? 0 :
                          (RaThisActiveDuration < 500)        ? 1 :
                          (RaThisActiveDuration < 1000)       ? 2 :
                          (RaThisActiveDuration < 5000)       ? 3 :
                          (RaThisActiveDuration < 10000)      ? 4 :
                          (RaThisActiveDuration < 50000)      ? 4 :
                          (RaThisActiveDuration < 100000)     ? 5 :
                          (RaThisActiveDuration < 500000)     ? 6 :
                          (RaThisActiveDuration < 100000)     ? 7 :
                          (RaThisActiveDuration < 250000)     ? 8 :
                          (RaThisActiveDuration < 500000)     ? 9 :
                          (RaThisActiveDuration < 750000)     ? 10:
                          (RaThisActiveDuration < 1000000)    ? 11:
                          (RaThisActiveDuration < 5000000)    ? 12:
                          (RaThisActiveDuration < 10000000)   ? 13:
                          (RaThisActiveDuration < 50000000)   ? 14: 15;
   } else
      RaThisActiveIndex = 0;

   for (i = 0; i < ARGUS_MAXFLOWDEFS; i++) {
      if (i == ARGUS_CLIENT)
         index = RaThisActiveIndex;

      rap =  RaFlowArray[i];
      if (RaPolicyMatch (argus, rap)) {
         targus = RaCopyArgusRecord(argus);

         RaModifyFlow(rap, targus);
         RaTimeout = rap->ArgusTimeout;

         if ((store = RaFindArgusRecord(RaThisPrvStore->htable, targus)) != NULL) {
            RaThisArgusStore = store;

            if ((srv->status & RA_SVCTEST) != (store->status & RA_SVCTEST)) {
               RaSendArgusRecord(store);
               store->status &= ~RA_SVCTEST;
               store->status |= (srv->status & RA_SVCTEST);
            }

            if (!(store->data[index])) {
               struct ArgusRecordData *data = NULL;

               if ((data = RaNewArgusData(targus)) != NULL) {
                  data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
                  data->status |= RA_MODIFIED;
                  data->status |= srv->status & RA_SVCTEST;

                  if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
                     double sumtime;

                     bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], &data->agr, sizeof(data->agr));
                     data->act.n        = data->agr.act.n;
                     sumtime            = data->agr.act.mean * data->agr.act.n;
                     data->act.sumtime  = sumtime;
                     data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

                     data->idle.n       = data->agr.idle.n;
                     sumtime            = data->agr.idle.mean * data->agr.idle.n;
                     data->idle.sumtime = sumtime;
                     data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
                  }
      
                  store->data[index] = data;
               } else
                  ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
      
            } else
               RaMergeArgusRecord(targus, store->data[index]);

            RaUpdateArgusStore(targus, store);
      
         } else {
            if ((store = RaNewArgusStore(targus)) != NULL) {
               struct ArgusRecordData *data = NULL;

               RaTotals[i]++;
               if ((store->rahtblhdr = RaAddHashEntry (RaThisPrvStore->htable, store)) != NULL) {
                  if (RaThisPrvStore != NULL)
                     RaAddToQueue(RaThisPrvStore->queue, &store->qhdr);

                  else 
                     RaAddToQueue(RaModelerQueue, &store->qhdr);
               }

               if ((store->htable = (struct RaHashTableStruct *) ArgusCalloc (1, sizeof (*store->htable))) != NULL) {
                  if ((store->htable->array = (struct RaHashTableHeader **) ArgusCalloc (RA_HASHTABLESIZE,
                                                 sizeof (struct RaHashTableHeader *))) != NULL) {
                     store->htable->size = RA_HASHTABLESIZE;
                  }
               }

               store->ArgusTimeout = RaTimeout;
               RaThisArgusStore = store;

               store->status |= RA_MODIFIED;
               store->status |= srv->status & RA_SVCTEST;

               if ((data = RaNewArgusData(targus)) != NULL) {
                  data->farhdrstatus = ArgusIndexRecord (data->argus, data->farhdrs);
                  data->status |= RA_MODIFIED;

                  if (data->farhdrstatus & ARGUS_AGR_DSR_STATUS) {
                     double sumtime;

                     bcopy((char *)data->farhdrs[ARGUS_AGR_DSR_INDEX], &data->agr, sizeof(data->agr));
                     data->act.n        = data->agr.act.n;
                     sumtime            = data->agr.act.mean * data->agr.act.n;
                     data->act.sumtime  = sumtime;
                     data->act.sumsqrd  = (data->agr.act.n * pow(data->agr.act.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.act.n;

                     data->idle.n       = data->agr.idle.n;
                     sumtime            = data->agr.idle.mean * data->agr.idle.n;
                     data->idle.sumtime = sumtime;
                     data->idle.sumsqrd = (data->agr.idle.n * pow(data->agr.idle.stdev, 2.0)) + pow(sumtime, 2.0)/data->agr.idle.n;
                  }

                  store->data[index] = data;

               } else
                  ArgusLog (LOG_ERR, "RaNewArgusData failed %s\n", strerror(errno));
            } else
               ArgusLog (LOG_ERR, "RaNewArgusStore failed %s\n", strerror(errno));
         }

         RaThisPrvStore = store;

         if (i != ARGUS_CLIENT)
            if (store->queue == NULL)
               store->queue = RaNewQueue();

         ArgusFree(targus);
      }
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaProcessRecord: done\n");
#endif
}


void
process_man (struct ArgusRecord *argus)
{
}


struct ArgusServiceRecord ArgusThisSrv;

void
process_tcp (struct ArgusRecord *argus)
{
   struct ArgusTCPObject *tcp = NULL;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if (ArgusThisFarStatus & ARGUS_TCP_DSR_STATUS) {
         if ((tcp = (struct ArgusTCPObject *) ArgusThisFarHdrs[ARGUS_TCP_DSR_INDEX]) != NULL) {
            if (tcp->state & (ARGUS_SAW_SYN || ARGUS_SAW_SYN_SENT)) {
               if (tcp->state & ARGUS_RESET) {
                  if (!(tcp->src.ackbytes || tcp->dst.ackbytes))
                     ArgusThisSrv.status = RA_SVCFAILED;
               } else
                  if (!(tcp->state & (ARGUS_FIN | ARGUS_FIN_ACK | ARGUS_NORMAL_CLOSE)))
                     ArgusThisSrv.status = RA_SVCINCOMPLETE;
            } else
               ArgusThisSrv.status = RA_SVCINCOMPLETE;
         }
      }
   }

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
process_icmp (struct ArgusRecord *argus)
{
/*
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;
 
      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   RaProcessThisSrvRecord (&ArgusThisSrv);
*/
}


void
process_udp (struct ArgusRecord *argus)
{
   int srccount, dstcount;

   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   srccount = argus->argus_far.src.count;
   dstcount = argus->argus_far.dst.count;

   if (Vflag)
      if ((srccount == 0) || (dstcount == 0))
         ArgusThisSrv.status = RA_SVCFAILED;


   if (argus->argus_far.flow.ip_flow.tp_p == ARGUS_RTP_FLOWTAG)
      if (!((argus->argus_far.src.count > 3) || (argus->argus_far.dst.count > 3)))
         argus->argus_far.flow.ip_flow.tp_p = 0;

   if (Rflag) {
      struct ArgusAGRStruct *agr;

      if ((agr = (struct ArgusAGRStruct *) ArgusThisFarHdrs[ARGUS_AGR_DSR_INDEX]) != NULL) {
         if (!((srccount == dstcount) && (srccount == agr->count)))
            ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         if (!((srccount == 1) && (dstcount == 1)))
            ArgusThisSrv.status = RA_SVCFAILED;
      }
   }

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
process_ip (struct ArgusRecord *argus)
{
   ArgusThisSrv.argus = argus;
   ArgusThisSrv.status = RA_SVCPASSED;

   if (Vflag) {
      if ((argus->argus_far.src.count > 0) &&
             (argus->argus_far.dst.count == 0)) {
         ArgusThisSrv.status = RA_SVCFAILED;

      } else {
         ArgusThisSrv.status = RA_SVCPASSED;
      }
   }

   if (Rflag)
      if (!((argus->argus_far.src.count == 1) && (argus->argus_far.dst.count == 1)))
         ArgusThisSrv.status = RA_SVCFAILED;

   RaProcessThisSrvRecord (&ArgusThisSrv);
}


void
process_arp (struct ArgusRecord *argus)
{
}


void
process_non_ip (struct ArgusRecord *argus)
{
}


void
RaSendArgusRecord(struct ArgusRecordStore *store)
{
/*
   char buf[MAXSTRLEN], *ptr = buf;
   struct ArgusRecordData *data;
   struct ArgusRecord *argus = NULL;
   struct ArgusFarHeaderStruct *farhdr;
   int i;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if ((data = store->data[i]) != NULL) {
         argus = data->argus;

         if (argus && (data->status & RA_MODIFIED)) {
            if (data->act.n > 0) {
               data->agr.act.n = data->act.n;
               data->agr.act.mean = data->act.sumtime/data->act.n;
               data->agr.act.stdev = sqrt (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
            }
            if (data->idle.n > 0) {
               data->agr.idle.n = data->idle.n;
               data->agr.idle.mean = data->idle.sumtime/data->idle.n;
               data->agr.idle.stdev = sqrt (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
            }
      
            bcopy ((char *) argus, buf, argus->ahdr.length);
            argus = (struct ArgusRecord *) ptr;
      
            if (data->agr.count > 0) {
               bcopy ((char *)&data->agr,&buf[argus->ahdr.length], data->agr.length);
               argus->ahdr.length += data->agr.length;
               argus->ahdr.status |= ARGUS_MERGED;
      
            } else 
               argus->ahdr.status &= ~(ARGUS_MERGED);
      
            argus->argus_far.time.last = data->agr.lasttime;
      
            if (wfile) {
#ifdef _LITTLE_ENDIAN
               ArgusHtoN(argus);
#endif _LITTLE_ENDIAN

               ArgusWriteNewLogfile (wfile, argus);
      
#ifdef _LITTLE_ENDIAN
               ArgusNtoH(argus);
#endif _LITTLE_ENDIAN
            } else {
      
               if (argus->ahdr.type & ARGUS_MAR)
                  printf ("%s\n", get_man_string (argus));
                  
               else {
                  ArgusThisFarStatus = ArgusIndexRecord(argus, ArgusThisFarHdrs);
      
                  switch (argus->ahdr.status & 0xFFFF) {
                     case ETHERTYPE_IP:
                        switch (argus->argus_far.flow.ip_flow.ip_p) {
                           case IPPROTO_TCP:              
                              printf ("%s\n", get_tcp_string (argus));
                              break;
            
                           case IPPROTO_ICMP:              
                              printf ("%s\n", get_icmp_string (argus));
                              break;
            
                           default:
                              printf ("%s\n", get_ip_string (argus));
                              break;
                        }
                        break;
            
                     case ETHERTYPE_ARP:
                     case ETHERTYPE_REVARP:
                        printf ("%s\n", get_arp_string (argus));
                        break;
            
                     default:
                        printf ("%s\n", get_nonip_string (argus));
                        break;
                  }
               }
               fflush (stdout);
            }
      
            argus = data->argus;

            if (argus->ahdr.type & ARGUS_FAR) {
               int farlen, length = argus->ahdr.length - sizeof(argus->ahdr);;
               farhdr = (struct ArgusFarHeaderStruct *)((char *)argus + sizeof(argus->ahdr));
      
               while (length > 0) {
                  switch (farhdr->type) {
                     case ARGUS_FAR: {
                        struct ArgusFarStruct *far = (struct ArgusFarStruct *) farhdr;
                        far->time.start.tv_sec = 0x7FFFFFFF; far->time.start.tv_usec = 0;
                        far->time.last.tv_sec = 0x7FFFFFFF; far->time.last.tv_usec = 0;
                        far->src.count = 0; far->src.bytes = 0;
                        far->dst.count = 0; far->dst.bytes = 0;
                        break;
                     }
      
                     case ARGUS_TCP_DSR: {
                        struct ArgusTCPObject *tcp = (struct ArgusTCPObject *) farhdr;
      
                        tcp->state = 0;
                        tcp->src.seqbase = 0; tcp->src.ackbytes = 0;
                        tcp->src.rpkts = 0; tcp->src.win = 0; tcp->src.flags = 0;
                        tcp->dst.seqbase = 0; tcp->dst.ackbytes = 0;
                        tcp->dst.rpkts = 0; tcp->dst.win = 0; tcp->dst.flags = 0;
      
                        break;
                     }
      
                     case ARGUS_AGR_DSR: {
                        struct ArgusAGRStruct *agr = (struct ArgusAGRStruct *) farhdr;
      
                        agr->count = 0;
                        agr->act.n = 0;
                        agr->act.min = 0x7FFFFFFF;   agr->act.mean = 0;
                        agr->act.stdev = 0;  agr->act.max = 0;
                        agr->idle.n = 0;
                        agr->idle.min = 0x7FFFFFFF; agr->idle.mean = 0;
                        agr->idle.stdev = 0; agr->idle.max = 0;
                        break;
                     }
                  }
                  farlen = farhdr->length;
                  if ((farhdr->type == ARGUS_SRCUSRDATA_DSR) ||
                      (farhdr->type == ARGUS_DSTUSRDATA_DSR))
                     farlen = farlen * 4;
                  length -= farlen;
                  farhdr = (struct ArgusFarHeaderStruct *)((char *)farhdr + farlen);
               }
            }
      
            data->agr.count = 0;
            data->agr.act.min = 0x7FFFFFFF;   data->agr.act.mean = 0;
            data->agr.act.stdev = 0;  data->agr.act.max = 0;
            data->agr.idle.min = 0x7FFFFFFF; data->agr.idle.mean = 0;
            data->agr.idle.stdev = 0; data->agr.idle.max = 0;
            data->act.sumtime = 0; data->act.n = 0;
            data->idle.sumtime = 0; data->idle.n = 0;
      
            data->argus->ahdr.status &= ~ARGUS_MERGED;
            data->status &= ~RA_MODIFIED;
         }
      }
   }
*/

   store->qhdr.logtime = ArgusGlobalTime;

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaSendArgusRecord(0x%x) done.\n", store);
#endif
}


#include <stdio.h>
#include <errno.h>

#define RA_MAXQSCAN  256


int RaSortCountsArgusStore (const void *, const void *);
int RaSortAddrsArgusStore (const void *, const void *);

int
RaSortCountsArgusStore (const void *void1, const void *void2)
{
   int retn = 0, i;
   struct ArgusRecord *argus;
   struct ArgusRecordStore *store1 = *(struct ArgusRecordStore **)void1;
   struct ArgusRecordStore *store2 = *(struct ArgusRecordStore **)void2;

   if (store1->sortag == 0) {
      for (i = 0; i < RaHistoTimeSeries; i++) {
         if ((store1->data[i]) && ((argus = store1->data[i]->argus) != NULL)) {
            store1->sortag += argus->argus_far.src.count;
            store1->sortag += argus->argus_far.dst.count;
         }
      }
   }

   if (store2->sortag == 0) {
      for (i = 0; i < RaHistoTimeSeries; i++) {
         if ((store2->data[i]) && ((argus = store2->data[i]->argus) != NULL)) {
            store2->sortag += argus->argus_far.src.count;
            store2->sortag += argus->argus_far.dst.count;
         }
      }
   }

   retn = (store2->sortag - store1->sortag);

   return (retn);
}


int
RaSortAddrsArgusStore (const void *void1, const void *void2)
{
   int retn = 0, i;
   struct ArgusRecordStore *store1 = *(struct ArgusRecordStore **)void1;
   struct ArgusRecordStore *store2 = *(struct ArgusRecordStore **)void2;
   struct ArgusRecord *argus1 = NULL, *argus2 = NULL;

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if (store1->data[i]) {
         argus1 = store1->data[i]->argus;
         break;
      }
   }

   for (i = 0; i < RaHistoTimeSeries; i++) {
      if (store2->data[i]) {
         argus2 = store2->data[i]->argus;
         break;
      }
   }

   if (argus1 && argus2)
      if (!(retn = (argus2->argus_far.flow.ip_flow.ip_dst - argus1->argus_far.flow.ip_flow.ip_dst)))
         retn = (argus2->argus_far.flow.ip_flow.ip_src - argus1->argus_far.flow.ip_flow.ip_src);

   return (retn);
}


#include <netdb.h>

struct RaSrvDescTableEntry {
   int srvDport, srvDaddr, proto, port;
   char *name, *desc;
};

struct RaSrvDescTableEntry *RaFindSrvHashObject (int, int);

void
RaPrintOutQueue (struct RaQueueStruct *queue, int level)
{
   struct ArgusIPFlow *ipFlow = NULL;
   struct ArgusRecordStore *obj = NULL;
   struct ArgusRecordData *data = NULL;
   struct RaSrvDescTableEntry *dtblent = NULL;
   double value;
   int i = 0, n, num = Nflag;

   if (Nflag == 0)
      num = queue->count;

   for (n = 0; n < num; n++) {
      if ((obj = (struct ArgusRecordStore *) queue->array[n]) != NULL) {

         for (i = 0; i < RaHistoTimeSeries; i++) {
            if ((data = obj->data[i]) != NULL) {
               ipFlow = &data->argus->argus_far.flow.ip_flow;

               switch (level) {
                  case ARGUS_SERVICE:
                     switch (ipFlow->ip_p) {
                        case IPPROTO_TCP:
                           if ((dtblent = RaFindSrvHashObject(ipFlow->ip_p, ipFlow->dport)) != NULL)
                              printf("Service: %-12s tcp port %-5d \"%s\"\n", dtblent->name, dtblent->port, dtblent->desc); 
                           fflush(stdout);
                           break;
      
                        case IPPROTO_UDP:
                           if ((dtblent = RaFindSrvHashObject(ipFlow->ip_p, ipFlow->dport)) != NULL)
                              printf("Service: %-12s udp port %-5d \"%s\"\n", dtblent->name, dtblent->port, dtblent->desc); 
                           fflush(stdout);
                           break;

                        case IPPROTO_IGMP:
                           printf("Service:  igmp:\n");
                           fflush(stdout);
                           break;
      
                        default:
                           printf("Service:  %s\n", get_ip_string(data->argus));
                           fflush(stdout);
                           break;
                     }
                     break;
      
                  case ARGUS_SERVICE_SUBNET:
//                   printf("Service_Subnet: %s\n", ipaddr_string(&ipFlow->ip_dst));
                     break;

                  case ARGUS_SERVER:
                     printf("   Server: %-20.20s Trans         Mean (uSec)               Range              Pkts       Bytes\n",
                                  ipaddr_string(&ipFlow->ip_dst));
                     fflush(stdout);
                     break;

                  case ARGUS_CLIENT_SUBNET:
//                   printf("        Client Subnet %s  Clients %d\n", ipaddr_string(&ipFlow->ip_src), obj->queue->count);
//                   fflush(stdout);
                     break;

                  case ARGUS_CLIENT:
//                   printf("  Clients %d\n", queue->count);
//                   fflush(stdout);

                     if (data->act.n > 0) {
                        data->agr.act.mean = data->act.sumtime/data->act.n;
                        value = (data->act.sumsqrd/data->act.n - pow (data->act.sumtime/data->act.n, 2.0));
                        data->agr.act.stdev = sqrt (value);
                        data->agr.act.n = data->act.n;
                     }
                     if (data->idle.n > 0) {
                        data->agr.idle.mean = data->idle.sumtime/data->idle.n;
                        value = (data->idle.sumsqrd/data->idle.n - pow (data->idle.sumtime/data->idle.n, 2.0));
                        data->agr.idle.stdev = sqrt (value);
                        data->agr.idle.n = data->idle.n;
                     }
      
      
                     printf("          %15.15s:      %5d  %10d  +- %-10d  %7d - %-12d   %-9d  %-9d\n",
                                         ipaddr_string(&ipFlow->ip_src),
                                         data->agr.count, data->agr.act.mean, data->agr.act.stdev,
                                         data->agr.act.min,  data->agr.act.max,
                                         data->argus->argus_far.src.count + data->argus->argus_far.dst.count,
                                         data->argus->argus_far.src.bytes + data->argus->argus_far.dst.bytes);
                     break;
               }
            }
         }

         if (Hflag && (level == ARGUS_CLIENT))
            printf("\n");

         fflush(stdout);

         if (obj->queue != NULL) {
            RaSortQueue (obj->queue);
            RaPrintOutQueue (obj->queue, level + 1);
         }

      } else
         break;
   }
}


void
RaProcessQueue(struct RaQueueStruct *queue, unsigned char status)
{
   struct ArgusRecordStore *obj = NULL;
   struct ArgusRecord *argus = NULL;
   int cnt = 0;
 
   switch (status) {
      case ARGUS_STOP:
         while (queue->count) {
            if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
               if (obj->htable) {
                  if (obj->htable->array)
                     ArgusFree(obj->htable->array);
                  ArgusFree(obj->htable);
               }
               RaTimeoutArgusStore(obj);
            }
         }
         break;

      default:
         if ((cnt = ((queue->count > RA_MAXQSCAN) ? RA_MAXQSCAN : queue->count)) != 0) {
            while (cnt--) {
               if ((obj = (struct ArgusRecordStore *) RaPopQueue(queue)) != NULL) {
                  argus = obj->data[0]->argus;
                  if (RaCheckTimeout(obj, argus)) {
                     if (obj->htable) {
                        if (obj->htable->array)
                           ArgusFree(obj->htable->array);
                        ArgusFree(obj->htable);
                     }
                     RaTimeoutArgusStore(obj);
                  } else
                     RaAddToQueue(queue, &obj->qhdr);

               } else
                  cnt++;
            }
         }
         break;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaProcessQueue (0x%x, %d) returning\n", queue, status);
#endif
}


struct RaSrvDescTableEntry *
RaFindSrvHashObject (int proto, int port)
{
   struct RaSrvDescTableEntry *retn = NULL;
   struct RaSrvHashTableHeader *head = NULL, *target = NULL;
   struct RaSrvHash RaThisSrvHash;
   unsigned short hash = 0;

   hash += ((unsigned short *)&proto)[0];
   hash += ((unsigned short *)&proto)[1];
   hash += ((unsigned short *)&port)[0];
   hash += ((unsigned short *)&port)[1];

   RaThisSrvHash.srvProto = proto;
   RaThisSrvHash.srvPort  = port;

   if ((target = RaSrvHashTable.array[hash % RaSrvHashTable.size]) != NULL) {
      head = target;
      do {
         if (!(bcmp ((char *)&RaThisSrvHash, (char *) &target->srvhash, sizeof(RaThisSrvHash)))) {
               retn = target->storeobj;
               break;
            } else
               target = target->nxt;
      } while (target != head);

   }

#ifdef ARGUSDEBUG
   ArgusDebug (5, "RaFindSrvHashObject: returning 0x%x\n", retn);
#endif

   return (retn);
}



struct RaSrvHashTableHeader *
RaAddSrvHashEntry (struct RaSrvDescTableEntry *store)
{
   struct RaSrvHashTableHeader *retn = NULL;
   struct RaSrvHashTableHeader *start = NULL;
   struct RaSrvHash RaThisSrvHash;
   unsigned short hash = 0;

   if ((retn = (struct RaSrvHashTableHeader *) ArgusCalloc (1, sizeof (struct RaSrvHashTableHeader))) != NULL) {
      retn->storeobj = store;
      bzero (&RaThisSrvHash, sizeof(RaThisSrvHash));
      RaThisSrvHash.srvProto = store->proto;
      RaThisSrvHash.srvPort  = store->port;

      hash += ((unsigned short *)&store->proto)[0];
      hash += ((unsigned short *)&store->proto)[1];
      hash += ((unsigned short *)&store->port)[0];
      hash += ((unsigned short *)&store->port)[1];

      retn->srvhash = RaThisSrvHash;
      retn->hash = hash;
      
      if ((start = RaSrvHashTable.array[hash % RaSrvHashTable.size]) != NULL) {
         retn->nxt = start;
         retn->prv = start->prv;
         retn->prv->nxt = retn;
         retn->nxt->prv = retn;
      } else
         retn->prv = retn->nxt = retn;

      RaSrvHashTable.array[hash % RaSrvHashTable.size] = retn;
      RaSrvHashTable.count++;
   }

#ifdef ARGUSDEBUG
   ArgusDebug (6, "RaAddSrvHashEntry (0x%x) returning 0x%x\n", store, retn);
#endif

   return (retn);
}


void
RaRemoveSrvHashEntry (struct RaSrvHashTableHeader *rahtblhdr)
{
   unsigned short hash = rahtblhdr->hash;

   rahtblhdr->prv->nxt = rahtblhdr->nxt;
   rahtblhdr->nxt->prv = rahtblhdr->prv;

   if (rahtblhdr == RaSrvHashTable.array[hash % RaSrvHashTable.size]) {
      if (rahtblhdr == rahtblhdr->nxt)
         RaSrvHashTable.array[hash % RaSrvHashTable.size] = NULL;
      else
         RaSrvHashTable.array[hash % RaSrvHashTable.size] = rahtblhdr->nxt;

      RaSrvHashTable.count--;
   }

   ArgusFree (rahtblhdr);
   RaAllocHashTableHeaders--;
 
#ifdef ARGUSDEBUG
   ArgusDebug (4, "RaRemoveSrvHashObject (0x%x) returning\n", rahtblhdr);
#endif
}


int RaCreateSrvTableEntry(int, char *);

int
RaCreateSrvTableEntry(int linenum, char *str)
{
  struct RaSrvDescTableEntry *dtblent;
  char namebuf[1024], descbuf[1024], *ptr, *tmp;
  int retn = 0;

  bzero (namebuf, 1024);
  bzero (descbuf, 1024);

  if ((dtblent = (struct RaSrvDescTableEntry *) ArgusCalloc (1, sizeof(*dtblent))) != NULL) {
     if (sscanf (str, "%d %d %d %d", &dtblent->srvDport, &dtblent->srvDaddr,
                                     &dtblent->proto, &dtblent->port)) {

        if ((ptr = strchr (str, (int) '"')) != NULL)
           if ((tmp = strchr ((++ptr), (int) '"')) != NULL) {
              *tmp++ = '\0';
              dtblent->name = strdup(ptr);
              str = tmp;
           }

        if ((ptr = strchr (str, (int) '"')) != NULL)
           if ((tmp = strchr (++ptr, (int) '"')) != NULL) {
              *tmp++ = '\0';
              dtblent->desc = strdup(ptr);
           }
     }

     if (dtblent != NULL)
        RaAddSrvHashEntry(dtblent);
  }

  return (retn);
}


int
RaReadServicesFile(char *srvfile)
{
   int retn = 0, RaSrvFileLineNumber = 0;
   char buf[MAXSTRLEN], *str;
   FILE *fd = NULL;

   if (srvfile != NULL) {
      if ((fd = fopen (srvfile, "r")) != NULL) {
         while ((str = fgets (buf, MAXSTRLEN, fd)) != NULL) {
            RaSrvFileLineNumber++;
   
            while (isspace((int)*str))
               str++;
            if (strlen(str)) {
               switch (buf[0]) {
                  case '#':
                  case '\n':
                  case '!':
                     break;
   
                  default:
                     if ((str = strdup(str)) != NULL) {
                        if (RaCreateSrvTableEntry(RaSrvFileLineNumber, str) < 0) {
                           fprintf (stderr, "%s: RaReadServicesFile: parse error line %d\n", ArgusProgramName, RaSrvFileLineNumber);
                           exit (-1);
                        }
   
                     } else
                        ArgusLog (LOG_ERR, "RaReadServiceFile: error %s\n", strerror(errno));
               }
            }
         }
   
         fclose(fd);
      }

   } else
      ArgusLog (LOG_ERR, "RaReadServicesFile: %s %s\n", srvfile, strerror(errno));

   return (retn);
}
