/*
 * Argus Client Software.  Tools to read, analyze and manage Argus data.
 * Copyright (c) 2000-2003 QoSient, LLC
 * All rights reserved.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * rahosts - argus data client  that printouts the unique IP addrs in
 *           in data stream.
 *
 * written by Carter Bullard
 * QoSient, LLC
 *
 */


#include <argus_client.h>
#include <rahosts.h>

#define RAMAP_ETHER_MAC_ADDR            0x1
#define RAMAP_IP_ADDR                   0x10

struct RaMapHashTableStruct {
   int size;
   struct RaMapHashTableHeader **array;
};

struct RaMapHashTableHeader {
   struct ArgusQueueHeader qhdr;
   struct RaMapHashTableHeader *nxt, *prv;
   unsigned int hash;
   int type, len, value, mask;
   struct RaClassNets *net;
   void *obj, *sub;
};

struct RaMapHashTableHeader *RaMapFindHashObject (struct RaMapHashTableStruct *, void *, int, int);
struct RaMapHashTableHeader *RaMapAddHashEntry (struct RaMapHashTableStruct *, void *, int, int);
void RaMapRemoveHashEntry (struct RaMapHashTableStruct *, struct RaMapHashTableHeader *);
unsigned int RaMapCalcHash (void *, int, int);

int RaSortThisDataRoutine (const void *, const void *);
void RaSortThisQueue (struct RaQueueStruct *);

struct RaQueueStruct *RaMapEtherAddrQueue = NULL;
struct RaQueueStruct *RaMapIPAddrQueue = NULL;

struct RaMapHashTableStruct RaMapHashTable;

#define MAX_OBJ_SIZE            1024
int RaInitialized = 0;

unsigned int RaMapHash = 0;
unsigned int RaHashSize  = 0;

#define ARGUS_TRACK_CLASS_ADDR		1
#define ARGUS_TRACK_CLASS_AADDR		2
#define ARGUS_TRACK_CLASS_BADDR		3
#define ARGUS_TRACK_CLASS_CADDR		4

int ArgusTrackClassAddrs = 0;
int ArgusTrackIPAddrs  = 0;
int ArgusTrackMacAddrs = 0;


void
ArgusClientInit ()
{
   struct ArgusModeStruct *mode;

   if (!(RaInitialized)) {
      RaWriteOut = 0;
      RaCumulativeMerge = 1;

      bzero ((char *) RaPrintAlgorithms, sizeof(RaPrintAlgorithms));

      RaHashSize = 0x10000;

      if ((RaMapHashTable.array = (struct RaMapHashTableHeader **)
                  ArgusCalloc (RaHashSize, sizeof (struct RaMapHashTableHeader))) == NULL)
         ArgusLog (LOG_ERR, "RaMapInit: ArgusCalloc error %s\n", strerror(errno));

      RaMapHashTable.size = RaHashSize;

      if ((mode = ArgusModeList) != NULL) {
         while (mode) {
            if (!(strcasecmp (mode->mode, "ether"))) {
               ArgusTrackMacAddrs++;
            } else
            if (!(strcasecmp (mode->mode, "ip"))) {
               ArgusTrackIPAddrs++;
            } else
            if (!(strcasecmp (mode->mode, "all"))) {
               ArgusTrackMacAddrs++;
               ArgusTrackIPAddrs++;
            }

            if (!(strcasecmp (mode->mode, "class")))
               ArgusTrackClassAddrs = ARGUS_TRACK_CLASS_ADDR;
            else
            if (!(strcasecmp (mode->mode, "classA")))
               ArgusTrackClassAddrs = ARGUS_TRACK_CLASS_AADDR;
            else
            if (!(strcasecmp (mode->mode, "classB")))
               ArgusTrackClassAddrs = ARGUS_TRACK_CLASS_BADDR;
            else
            if (!(strcasecmp (mode->mode, "classC")))
               ArgusTrackClassAddrs = ARGUS_TRACK_CLASS_CADDR;

            mode = mode->nxt;
         }
      }

      if (!(ArgusTrackIPAddrs || ArgusTrackMacAddrs))
         ArgusTrackIPAddrs++;
   }
}

int RaParseCompleting = 0;

void
RaParseComplete (int sig)
{
   if ((sig >= 0) && (!RaParseCompleting)) {
      RaParseCompleting++;

      if (RaMapEtherAddrQueue) {
         struct RaMapHashTableHeader *retn = NULL;

         RaSortThisQueue(RaMapEtherAddrQueue);
         while ((retn = (struct RaMapHashTableHeader *) RaPopQueue(RaMapEtherAddrQueue)) != NULL)
            fprintf (stdout, "%s\n", etheraddr_string (retn->obj));
      }

      if (RaMapIPAddrQueue) {
         struct RaMapHashTableHeader *retn = NULL;

         RaSortThisQueue(RaMapIPAddrQueue);
         while ((retn = (struct RaMapHashTableHeader *) RaPopQueue(RaMapIPAddrQueue)) != NULL)
            fprintf (stdout, "%s\n", ipaddr_string (retn->obj));
      }

      fflush (stdout);
   }
}

void
ArgusClientTimeout ()
{
}

void
parse_arg (int argc, char**argv)
{ 
}


void
usage ()
{
   extern char version[];
   fprintf (stderr, "Ratemplate Version %s\n", version);
   fprintf (stderr, "usage: %s \n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -S remoteServer  [- filter-expression]\n", ArgusProgramName);
   fprintf (stderr, "usage: %s [options] -r argusDataFile [- filter-expression]\n\n", ArgusProgramName);

   fprintf (stderr, "options: -b                dump packet-matching code.\n");
   fprintf (stderr, "         -C                treat the remote source as a Cisco Netflow source.\n");
   fprintf (stderr, "         -D <level>        specify debug level\n");
   fprintf (stderr, "         -F <conffile>     read configuration from <conffile>.\n");
   fprintf (stderr, "         -h                print help.\n");
   fprintf (stderr, "         -n                don't convert numbers to names.\n");
   fprintf (stderr, "         -r <file>         read argus data <file>. '-' denotes stdin.\n");
   fprintf (stderr, "         -S <host[:port]>  specify remote argus <host> and optional port number.\n");
   fprintf (stderr, "         -t <timerange>    specify <timerange> for reading records.\n");
   fprintf (stderr, "                  format:  timeSpecification[-timeSpecification]\n");
   fprintf (stderr, "                           timeSpecification: [mm/dd[/yy].]hh[:mm[:ss]]\n");
   fprintf (stderr, "                                               mm/dd[/yy]\n");
   fprintf (stderr, "                                               -%%d{yMhdms}\n");
   fprintf (stderr, "         -T <secs>         attach to remote server for T seconds.\n");
#ifdef ARGUS_SASL
   fprintf (stderr, "         -U <user/auth>    specify <user/auth> authentication information.\n");
#endif
   exit(1);
}


void
RaProcessRecord (struct ArgusRecord *argus)
{
   if (argus->ahdr.type & ARGUS_MAR) {

   } else {
      if (ArgusTrackMacAddrs && (ArgusThisFarStatus & ARGUS_MAC_DSR_STATUS)) {
         struct ArgusMacStruct *mac = (struct ArgusMacStruct *) ArgusThisFarHdrs[ARGUS_MAC_DSR_INDEX];

         RaMapInventory (&mac->phys_union.ether.ethersrc, RAMAP_ETHER_MAC_ADDR, 6);
         RaMapInventory (&mac->phys_union.ether.etherdst, RAMAP_ETHER_MAC_ADDR, 6);
      }

      if (ArgusTrackIPAddrs) {
         RaMapInventory((unsigned char *)&argus->argus_far.flow.ip_flow.ip_src, RAMAP_IP_ADDR, 4);
         RaMapInventory((unsigned char *)&argus->argus_far.flow.ip_flow.ip_dst, RAMAP_IP_ADDR, 4);
      }
   }
}

int RaSendArgusRecord(struct ArgusRecordStore *argus) {return 0;}

#include <netinet/in.h>

void
RaMapInventory (void *oid, int type, int len)
{
   struct RaMapHashTableHeader *retn = NULL;
   unsigned int addr = *(unsigned int *)oid;

   if (ArgusTrackClassAddrs) {
      switch (ArgusTrackClassAddrs) {
         case ARGUS_TRACK_CLASS_ADDR:
            if (IN_CLASSA(addr)) addr &= IN_CLASSA_NET; else
            if (IN_CLASSB(addr)) addr &= IN_CLASSB_NET; else
            if (IN_CLASSC(addr)) addr &= IN_CLASSC_NET;
            break;

         case ARGUS_TRACK_CLASS_AADDR:
            addr &= IN_CLASSA_NET;
            break;
         case ARGUS_TRACK_CLASS_BADDR:
            addr &= IN_CLASSB_NET;
            break;
         case ARGUS_TRACK_CLASS_CADDR:
            addr &= IN_CLASSC_NET;
            break;
      }
      *(unsigned int *)oid = addr;
   }

   if (!(retn = RaMapFindHashObject (&RaMapHashTable, oid, type, len))) {
      if (!(retn = RaMapAddHashEntry (&RaMapHashTable, oid, type, len)))
         ArgusLog (LOG_ERR, "RaMapInventory: RaMapAddHashEntry error %s\n", strerror(errno));

      switch (type) {
         case RAMAP_ETHER_MAC_ADDR: {
            if (!(RaMapEtherAddrQueue))
               if ((RaMapEtherAddrQueue = RaNewQueue()) == NULL)
                  ArgusLog (LOG_ERR, "RaNewQueue: error %s\n", strerror(errno));

            RaAddToQueue(RaMapEtherAddrQueue, &retn->qhdr);
            break;     
         }

         case RAMAP_IP_ADDR: {
            if (!(RaMapIPAddrQueue))
               if ((RaMapIPAddrQueue = RaNewQueue()) == NULL)
                  ArgusLog (LOG_ERR, "RaNewQueue: error %s\n", strerror(errno));

            RaAddToQueue(RaMapIPAddrQueue, &retn->qhdr);
            break;     
         }
      }
   }
 
#ifdef ARGUSDEBUG
   ArgusDebug (3, "RaMapInventory (0x%x, %d, %d) returning\n", oid, type, len);
#endif
}


unsigned int
RaMapCalcHash (void *obj, int type, int len)
{
   u_char buf[MAX_OBJ_SIZE];
   unsigned int retn = 0;

   switch (type) {
      case RAMAP_ETHER_MAC_ADDR:
          len = 6;
          break;

      case RAMAP_IP_ADDR:
          len = 4;
          break;

      default:
          break;
   }

   bzero (buf, sizeof buf);

   if (RaHashSize <= 0x100) {
      unsigned char hash = 0, *ptr = (unsigned char *) buf;
      int i, nitems = len;

      bcopy ((char *) obj, (char *)&buf, len);

      for (i = 0; i < nitems; i++)
         hash += *ptr++;

      retn = hash;

   } else
   if (RaHashSize <= 0x10000) {
      unsigned short hash = 0, *ptr = (unsigned short *) buf;
      int i, nitems = (len / sizeof(unsigned short)) + 2;

      bcopy ((char *) obj, &buf[1], len);

      for (i = 0; i < nitems; i++)
         hash += *ptr++;

      retn = hash;

   } else {
      unsigned int hash = 0, *ptr = (unsigned int *) buf;
      int i, nitems = (len /sizeof(unsigned int)) + 2;

      bcopy ((char *) obj, &buf[3], len);

      for (i = 0; i < nitems; i++)
         hash += *ptr++;

      retn = hash;
   }

   return (retn);
}



struct RaMapHashTableHeader *
RaMapFindHashObject (struct RaMapHashTableStruct *table, void *obj, int type, int len)
{
   struct RaMapHashTableHeader *retn = NULL, *head = NULL, *target;
   int RaMapHash = 0;

   RaMapHash = RaMapCalcHash (obj, type, len);

   if ((target = table->array[RaMapHash % table->size]) != NULL) {
      head = target;
      do {
         if ((type == target->type) && (len == target->len)) {
            if (!(bcmp ((char *) obj, (char *) target->obj, len))) {
               retn = target;
               break;
            }
         }

         target = target->nxt;
      } while (target != head);
   }

#ifdef TCPCLEANDEBUG
   RaMapDebug (6, "RaMapFindHashEntry () returning 0x%x RaMapHash %d\n", retn, RaMapHash);
#endif
 
   return (retn);
}


struct RaMapHashTableHeader *
RaMapAddHashEntry (struct RaMapHashTableStruct *table, void *oid, int type, int len)
{
   struct RaMapHashTableHeader *retn = NULL, *start = NULL;

   if ((retn = (struct RaMapHashTableHeader *) ArgusCalloc (1, sizeof (struct RaMapHashTableHeader))) != NULL) {
      RaMapHash = RaMapCalcHash (oid, type, len);

      retn->hash = RaMapHash;
      retn->type = type;
      retn->len  = len;

      if ((retn->obj = (void *) ArgusCalloc (1, len)) == NULL)
         ArgusLog (LOG_ERR, "RaMapAddHashEntry: ArgusCalloc error %s\n", strerror(errno));
      else
         bcopy ((char *) oid, (char *)retn->obj, len);
      
      if ((start = table->array[RaMapHash % table->size]) != NULL) {
         retn->nxt = start;
         retn->prv = start->prv;
         retn->prv->nxt = retn;
         retn->nxt->prv = retn;
      } else
         retn->prv = retn->nxt = retn;

      table->array[RaMapHash % table->size] = retn;
   }

#ifdef TCPCLEANDEBUG
   RaMapDebug (3, "RaMapAddHashEntry (0x%x, %d, %d) returning 0x%x\n", oid, type, len, retn);
#endif

   return (retn);
}

 
void
RaMapRemoveHashEntry (struct RaMapHashTableStruct *table, struct RaMapHashTableHeader *htblhdr)
{
   unsigned short hash = htblhdr->hash;

   htblhdr->prv->nxt = htblhdr->nxt;
   htblhdr->nxt->prv = htblhdr->prv;

   if (htblhdr == table->array[hash % table->size]) {
      if (htblhdr == htblhdr->nxt)
         table->array[hash % table->size] = NULL;
      else
         table->array[hash % table->size] = htblhdr->nxt;
   }

   ArgusFree (htblhdr);

#ifdef TCPCLEANDEBUG
   RaMapDebug (6, "RaMapRemoveHashEntry (0x%x) returning\n", htblhdr);
#endif
}


int
RaSortThisDataRoutine (const void *void1, const void *void2)
{
   struct RaMapHashTableHeader *data1 = *(struct RaMapHashTableHeader **)void1;
   struct RaMapHashTableHeader *data2 = *(struct RaMapHashTableHeader **)void2;
   int retn = 0;

   if (data1->type == data2->type)
      switch (data1->type) {
         case RAMAP_ETHER_MAC_ADDR:
            retn = bcmp((char *)data2->obj, (char *)data1->obj, data1->len);
            break;

         case RAMAP_IP_ADDR: {
            unsigned int addr1, addr2;

            addr1 = *(unsigned int *)data1->obj;
            addr2 = *(unsigned int *)data2->obj;
            retn = (addr2 > addr1) ? 1 : ((addr2 == addr1) ? 0 : -1);
            break;
         }
      }

   return (retn);
}

void
RaSortThisQueue (struct RaQueueStruct *queue)
{
   int i = 0, cnt = queue->count;
   struct ArgusQueueHeader *qhdr;

   if (queue->array != NULL) {
      ArgusFree(queue->array);
      queue->array = NULL;
   }

   if (cnt > 0) {
      if ((queue->array = (struct ArgusQueueHeader **) ArgusCalloc(sizeof(struct ArgusQueueHeader *), cnt + 1)) != NULL) {
         while ((qhdr = RaPopQueue(queue)) != NULL) {
            queue->array[i] = qhdr;
            i++;
         }

         qsort ((char *) queue->array, i, sizeof (struct ArgusQueueHeader *), RaSortThisDataRoutine);

         for (i = 0 ; i < cnt; i++)
            RaPushQueue(queue, queue->array[i]);

         ArgusFree(queue->array);
         queue->array = NULL;

      } else
         ArgusLog (LOG_ERR, "RaSortQueue: ArgusCalloc(%d, %d) %s\n", sizeof(struct ArgusRecord *),
                                                                     cnt, strerror(errno));
   }
}
