/*

*************************************************************************

ArmageTron -- Just another Tron Lightcycle Game in 3D.
Copyright (C) 2000  Manuel Moos (manuel@moosnet.de)

**************************************************************************

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
***************************************************************************

*/

#ifndef ArmageTron_TESS_H
#define ArmageTron_TESS_H

#include "tMemManager.h"
#include "tList.h"
#include "tEventQueue.h"
#include "eCoord.h"
#include "tHeap.h"

#ifdef POWERPAK_DEB
#include "PowerPak/powerpak.h"
#include "PowerPak/powerdraw.h"
#endif

#include <iostream.h>

#define MAX_VIEWERS 4

extern int se_debugExt;

class ePoint;
class eEdge;
class eFace;
class eWall;

inline int se_Left(int i){return (i+2)%3;}
inline int se_Right(int i){return (i+1)%3;}

inline int se_X_ToScreen(REAL x){
  int X=10+int(x);
  if (X<=3) X=3;
  if (X>=635) X=635;
  return X;
}
inline int se_Y_ToScreen(REAL y){
  int Y=470-int(y);
  if (Y<=3) Y=3;
  if (Y>=475) Y=475;
  return Y;
}




class ePoint:public eCoord{
  friend class eEdge;

  int id;
  tArray<eFace *> f;

#ifdef DEBUG
public:
#endif
  static List<ePoint> points;
  int refCtr;
#ifndef DEBUG
public:
#endif
  void AddRef(){refCtr++;} // only to be used by off-grid objects
  void Release(){
    refCtr--; 
#ifdef DEBUG
    if (refCtr<0)
      tERR_ERROR_INT("Additional unreg!");
#endif
  }

  ePoint();
  ePoint(REAL X,REAL Y);
  ePoint(const eCoord &c);
  ~ePoint();

  // Calculations:

  bool operator==(const ePoint &a) const{return eCoord::operator==(a);}
  bool operator!=(const ePoint &a) const{return !eCoord::operator==(a);}
  eCoord operator-(const ePoint &a) const{return eCoord(x-a.x,y-a.y);}
  eCoord operator+(const ePoint &a) const{return eCoord(x+a.x,y+a.y);}
  REAL   operator*(const ePoint &a) const{return -x*a.y+y*a.x;}
  const eCoord &operator=(const eCoord &a){x=a.x;y=a.y;return *this;}

  bool operator==(const eCoord &a) const{return eCoord::operator==(a);}
  bool operator!=(const eCoord &a) const{return !eCoord::operator==(a);}
  eCoord operator-(const eCoord &a) const{return eCoord(x-a.x,y-a.y);}
  eCoord operator-() const{return eCoord(-x,-y);}
  eCoord operator+(const eCoord &a) const{return eCoord(x+a.x,y+a.y);}
  REAL   operator*(const eCoord &a) const{return -x*a.y+y*a.x;}
  const eCoord &operator=(const ePoint &a){x=a.x;y=a.y;return *this;}


  // adds eFace f to neighbour list
  void AddNeighbour(eFace *F);

  // removes it.
  void RemoveNeighbour(eFace *F);

  // replaces all known pointers to *this with pointers to *P.
  void Replace(ePoint *P);

  // adds a new Point *end, adds an eEdge from *this to *end with
  // type eWall. Modifies other faces and non-eWall-edges; Visibility
  // information needs to be updated afterwards.

  ePoint * DrawLineTo(ePoint *end,eWall *wal=NULL,bool change_grid=1);
  
  // almost the same
  ePoint * DrawLineTo(const eCoord &dest,eWall *wal=NULL,
			      bool change_grid=1){
    ePoint *ret=tNEW(ePoint)(dest);
    return DrawLineTo(ret,wal,change_grid);
  }

  static ePoint * InsertIntoGrid(eCoord x);
  void InsertIntoGrid();

  static void Clear();
  static void Check(bool lp=true);

#ifdef POWERPAK_DEB
  void DebugPlot(PD_Color c);
#endif
};



// a base class for the next two classes: connects an eEdge and a viewer
class eEdgeViewer{

  friend class eEdge;
 protected:
  //  tCHECKED_PTR(eEdgeViewer) next;
  eEdgeViewer *next;

  tCHECKED_PTR(eEdge) e;      // the eEdge we belong to
  int viewer;               // and the viewer
     
  eEdgeViewer **Ref();

public:
  eEdgeViewer(eEdge *E,int v);

  virtual ~eEdgeViewer();
  
  virtual void Render();
};

// tEvents for a viewer crossing an eEdge (that is, the sraight
// prolongiation of the eEdge):

class eViewerCrossesEdge: public tEvent,public eEdgeViewer{
protected:
  virtual tHeapBase *Heap();
public:
  eViewerCrossesEdge(eEdge *E,int v);
  virtual ~eViewerCrossesEdge();
  
  virtual bool Check(REAL dist);

  virtual void Render();
};



class eEdge{
  friend class ePoint;
  friend class eWall;
  friend class eFace;
  friend class eEdgeViewer;

  int id;

  //  tCHECKED_PTR(eEdgeViewer) edgeViewers; // ancor for attached edgeViewers
  eEdgeViewer *edgeViewers; // ancor for attached edgeViewers

  tCHECKED_PTR(eWall) wall;  // is it a eWall? what type?

public:
  void calc_Len() const;

  static List<eEdge> edges;

  tCHECKED_PTR(ePoint) p[2]; // the endpoints
  tCHECKED_PTR(eFace)  f[2]; // the faces left and right to the eEdge (may be NULL)

  REAL len;

  eWall *Wall()const {return wall;}
  void SetWall(eWall *w);

  eEdge(ePoint *p1,ePoint *p2,eWall *W=NULL);
  ~eEdge();

  void AddNeighbour(eFace * F);
  void RemoveNeighbour(eFace * F);

  REAL Ratio(const eCoord &c)const{return eCoord::V(*p[0],c,*p[1]);}

  eCoord Vec(){return *p[1]-*p[0];}

  // flip: swap p[0] and p[1]
  void Flip();
  // flips, if not P==p[0]
  void MakeFirst(ePoint *P){if (!(P==p[0])) Flip();}

  // are we allowed to replace it?
  bool Movable() const{return !wall;}

  // splitting
  bool Splittable() const;

  // split eEdge at s;
  void Split(eEdge *& e1,eEdge *& e2,ePoint *s);

  // does the same, but guarantees that first lies in e1.
  void SplitOriented(eEdge *& e1,eEdge *& e2,ePoint *s,ePoint *first);

  // different split along an eEdge::
  // this:   XXXXXWWWWZZZZWWWWWWWWWW
  // e:      ...........

  // after:
  // e1:                ZZWWWWWWWWWW
  // e:      XXXXXWWWWZZ

  void Split(eEdge * e,eEdge *& e1);


  // get the intersection point of two edges: 
  // stupid intersection without checks; returned point
  // does not allways lie within the two edges
  ePoint * IntersectWithCareless(const eEdge *e2) const;

  // the same, but with checks: if the two edges don't intersect,
  // NULL is returned.
  ePoint * IntersectWith(const eEdge *e2) const;

  // inserts an eEdge into the grid
  void CopyIntoGrid(bool change_grid=1);

  // mark this as edge of vision
  void MarkCheckVis(int i);
  static void MarkCheckVisAll(int i);

  // checks the visibility; return value: do we need to check
  // it again in the near future?
  bool UpdateVis(int i);

  // checks all edges on visibility for viewer i
  static void UpdateVisAll();
  static void UpdateVisAll(int viewer);

  // I/O:
  void Print(ostream &s) const;

  static void Clear();
  static void Check();

  eFace *Other(eFace *F){
    if (f[0]==F) return f[1];
    else if (f[1]==F) return f[0];
    else tERR_ERROR_INT("Face is not neighbour to eEdge!");
  }

  eFace * NearerToViewer(int i);    // which of the neighbours is
  // closer to viewer i?
  //static void check_vis(REAL time); // checks visibility of all edges

  static void SeethroughHasChanged(); // some eWalls changed their 
  // blocking height; check that!

  bool Simplify(int dir);
  // try to get rid of this eEdge; return value: can we delete it?

  static void SimplifyNum(int e,int dir);
  // try to get rid of eEdge number e

  static void SimplifyAll(int count=1);
  // try to get rid of count edges

#ifdef POWERPAK_DEB
  void DebugPlot(PD_Color c);
  void DebugPlotRefresh();
#endif
};





class eFace{
  friend class ePoint;
  friend class eEdge;
  friend class eGameObject;

  int id;

public:
  REAL   visHeight[MAX_VIEWERS]; // at which height can the
  // cameras see into this eFace?
  static int s_currentCheckVis[MAX_VIEWERS];
  
  tCHECKED_PTR(ePoint) p[3]; // points counterclockwise
  tCHECKED_PTR(eEdge)  e[3]; // eEdge i is opposite to point i

public:
  static List<eFace> faces;

  eFace (eEdge *e1,eEdge *e2,eEdge *e3);
  ~eFace();

  eFace *F(int i){return e[i]->Other(this);}

  void Print(ostream &s) const;

  int FindPoint(const ePoint *P) const; // returns -1, if P is not a point
  // of this eFace, else the p[FindPoint(P)]=P.
  int FindEdge(const eEdge *E) const; // same

  eCoord LeftVec(int i){return (*p[se_Left(i)])-(*p[i]);}
  eCoord RightVec(int i){return (*p[se_Right(i)])-(*p[i]);}

  // visibility by viewer i
  void SetVisHeight(int i,REAL h);
  static void SetVisHeightAll(int i,REAL h);

  bool IsInside(const eCoord &coord);

  static void UpdateVisAll(int num=1);    // removes faces from the vis list
  void UpdateVis(int i);      // removes this eFace from the vis list

  static void Clear();
  static void Check();

#ifdef POWERPAK_DEB
  void DebugPlot(PD_Color c);
#endif
};


inline ostream & operator<<(ostream &s,const ePoint &x){x.Print(s);return s;}
inline ostream & operator<<(ostream &s,const eEdge &x){x.Print(s);return s;}
inline ostream & operator<<(ostream &s,const eFace &x){x.Print(s);return s;}


void se_CreateGrid();
void se_ClearGrid();

void se_GridRange(REAL range_squared);

void se_CheckGrid(bool lp=true); // checks the grid for consistency

// displays the grid ,eWalls and gameobjects
void Render(int viewer);

void se_ResetVisibles(int viewer);  // reset the visibility information


#endif






