/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	An abstract class for writing documents
 *	to file.
 *
 *	by Tony Sideris	(10:12AM Aug 07, 2002)
 *================================================*/
#include "arson.h"

#include <qtextstream.h>
#include <qlistview.h>

#include <kio/netaccess.h>
#include <klocale.h>

#include "filewriter.h"
#include "tempfile.h"

/*========================================================*/
/*	Base file writer
 *========================================================*/

ArsonFileWriter::ArsonFileWriter (ArsonDocWidget *pd, const KURL &url)
	: ArsonBaseWriter(pd, url),
	m_pt(NULL), m_file(new ArsonTempFile)
{
	if (m_file->status() == 0)
		m_pt = m_file->textStream();
}

/*========================================================*/

ArsonFileWriter::~ArsonFileWriter (void)
{
	m_file->close();

    if (!(url().isEmpty()))
    {
        if (!(KIO::NetAccess::upload(m_file->name(), url())))
        {
            arsonErrorMsg(
            i18n("Failed to write file %1.").arg(url().path()));
        }
    }

	delete m_file;
}

/*========================================================*/

void ArsonFileWriter::writeLine (const QString &text)
{
	stream() << text << endl;
}

/*========================================================*/
/*	Base XML file writer
 *========================================================*/

ArsonXmlFileWriter::ArsonXmlFileWriter (const QString &docname, ArsonDocWidget *pd, const KURL &url)
	: ArsonFileWriter(pd, url), m_xml(stream(), docname)
{
	m_tags.setAutoDelete(true);
	pd->editRootElement(m_xml);
}

ArsonXmlFileWriter::~ArsonXmlFileWriter (void) { }

/*========================================================*/

void ArsonXmlFileWriter::beginDocument (void)
{
	ArsonFileWriter::beginDocument();
	m_xml.begin();
}

void ArsonXmlFileWriter::endDocument (void)
{
	ArsonFileWriter::endDocument();
	m_xml.end();
}

/*========================================================*/

void ArsonXmlFileWriter::pushTag (ArsonXmlTag *tag)
{
	tag->begin();
	m_tags.push(tag);
}

void ArsonXmlFileWriter::popTag (void)
{
	if (ArsonXmlTag *pt = m_tags.pop())
	{
		pt->end();
		delete pt;
	}
}

/*========================================================*/
/*	Write a file list to XML
 *========================================================*/

ArsonXmlFileListWriter::ArsonXmlFileListWriter (ArsonFileListDoc *pd, const KURL &url)
	: ArsonXmlFileWriter(QString("arson-") + pd->propDocType(), pd, url)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonXmlFileListWriter::beginBranch (QListViewItem *pi, ArsonListItem *pl)
{
	ArsonXmlTag *pt = new ArsonXmlTag(xml(), document()->propFolder());

	ArsonXmlFileWriter::beginBranch(pi, pl);

	pt->addAttribute("name", KURL::encode_string(pi->text(0)));
	pushTag(pt);
	return true;
}

void ArsonXmlFileListWriter::endBranch (void)
{
	ArsonXmlFileWriter::endBranch();
	popTag();
}

/*========================================================*/

bool ArsonXmlFileListWriter::visit (QListViewItem *pi, ArsonListItem *pl)
{
	ArsonXmlTag tag (xml(), document()->propItem());

	tag.addAttribute("url",
		KURL::encode_string(pl->persist()));

	tag.doit();
	return true;
}

/*========================================================*/
/*	Write a plain text file
 *========================================================*/

ArsonTextFileWriter::ArsonTextFileWriter (ArsonDocWidget *pd, const KURL &url)
	: ArsonFileWriter(pd, url), m_level(0)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonTextFileWriter::beginBranch (QListViewItem *pi, ArsonListItem *pl)
{
	if (ArsonFileWriter::beginBranch(pi, pl))
	{
		const QString name (pl->display());

		if (!name.isEmpty())
			writeLine(indent() + name);
		
		++m_level;
		return true;
	}

	return false;
}

void ArsonTextFileWriter::endBranch (void)
{
	ArsonFileWriter::endBranch();
	--m_level;
}

/*========================================================*/

bool ArsonTextFileWriter::visit (QListViewItem *pi, ArsonListItem *pl)
{
	writeLine(indent() + pl->persist());
	return true;
}

/*========================================================*/

QString ArsonTextFileWriter::indent (void) const
{
	QString str;
	
	str.fill(QChar(' '), m_level);
	return str;
}

/*========================================================*/
/*	Write an m3u file
 *========================================================*/

ArsonM3uFileWriter::ArsonM3uFileWriter (ArsonFileListDoc *pd, const KURL &url)
	: ArsonFileWriter(pd, url)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonM3uFileWriter::visit (QListViewItem *pi, ArsonListItem *pl)
{
	ArsonFileListItem *pf = (ArsonFileListItem *) pl;
	
	writeLine(pf->persist());
	return true;
}

/*========================================================*/
