/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __LOOKUP_H__
#define __LOOKUP_H__

#include <qvaluelist.h>
#include <qstring.h>

/*========================================================*/

class ArsonCdInfo;

class ArsonLookup
{
public:
	enum {
		unknown = -1,
		freedbHttp = 0,
		freedbLocal,
		cdindex,
		_lookup_max,
	};

	ArsonLookup (uint type = unknown);
	virtual ~ArsonLookup (void);

	uint type (void) const { return m_type; }
	const QString &typeDesc (void) const;

	virtual ArsonLookup *clone (void) const = 0;
	virtual QString display (void) const = 0;
	virtual QString toString (void) const = 0;
	virtual void fromString (const QString &str) = 0;
	virtual bool submit (const ArsonCdInfo &info) const = 0;
	virtual bool cdInfo (ArsonCdInfo &info) const = 0;

	static ArsonLookup *create (const QString &str);
	static const QString &typeDesc (int type);

#ifdef ARSONDBG
	static void show_instance_count (void);
	static int instance_count;
#endif	//	ARSONDBG
	
private:
	uint m_type;
};

/*==========================================================*/

class ArsonInetLookup : public ArsonLookup
{
public:
	ArsonInetLookup (uint type, const QString &addr = QString::null, short port = 0)
		: ArsonLookup(type), m_addr(addr), m_port(port) { }

	const QString &addr (void) const { return m_addr; }
	short port (void) const { return m_port; }

	void setAddr (const QString &addr) { m_addr = addr; }
	void setPort (short port) { m_port = port; }

	virtual QString display (void) const;
	virtual QString toString (void) const;
	virtual void fromString (const QString &str);

private:
	QString m_addr;
	short m_port;
};

/*==========================================================*/

class ArsonCdindexLookup : public ArsonInetLookup
{
public:
	ArsonCdindexLookup (const QString &addr = QString::null, short port = 0)
		: ArsonInetLookup(cdindex, addr, port) { }

	virtual ArsonLookup *clone (void) const
	{ return new ArsonCdindexLookup(addr(), port()); }
	virtual bool submit (const ArsonCdInfo &info) const;
	virtual bool cdInfo (ArsonCdInfo &info) const;
};

/*==========================================================*/

class ArsonFreedbHttpLookup : public ArsonInetLookup
{
public:
	ArsonFreedbHttpLookup (const QString &addr = QString::null, short port = 0)
		: ArsonInetLookup(freedbHttp, addr, port) { }

	virtual ArsonLookup *clone (void) const
	{ return new ArsonFreedbHttpLookup(addr(), port()); }
	virtual bool submit (const ArsonCdInfo &info) const;
	virtual bool cdInfo (ArsonCdInfo &info) const;
};

/*==========================================================*/

class ArsonFreedbLocalLookup : public ArsonLookup
{
public:
	ArsonFreedbLocalLookup (const QString &path = QString::null, bool as = false)
		: ArsonLookup(freedbLocal), m_path(path), m_autosave(as) { }

	const QString &path (void) const { return m_path; }
	void setPath (const QString &path) { m_path = path; }
	bool autosave (void) const { return m_autosave; }
	void setAutosave (bool as) { m_autosave = as; }

	virtual ArsonLookup *clone (void) const
	{ return new ArsonFreedbLocalLookup(path(), autosave()); }
	virtual QString display (void) const;
	virtual QString toString (void) const;
	virtual void fromString (const QString &str);
	virtual bool submit (const ArsonCdInfo &info) const;
	virtual bool cdInfo (ArsonCdInfo &info) const;

private:
	QString findFile (const ArsonCdInfo &info, bool create, QString *pcateg = NULL) const;
	QString findCateg (const QString &id) const;

	QString m_path;
	bool m_autosave;
};

/*==========================================================*/

class KConfig;

class ArsonLookupOrder
{
	typedef QValueList<ArsonLookup*> LOOKUPS;

public:
	ArsonLookupOrder (void);
	ArsonLookupOrder (const ArsonLookupOrder &obj) { (*this) = obj; }
	~ArsonLookupOrder (void) { clear(); }

	ArsonLookupOrder &operator= (const ArsonLookupOrder &obj);

	int addLookup (ArsonLookup *ptr);
	void delLookup (int index);
	void clear (void);

	void autoSubmit (const ArsonCdInfo &info) const;

	void save (KConfig *pk);
	int load (KConfig *pk);

	const ArsonLookup *lookup (int index) const { return m_lookups[index]; }
	ArsonLookup *lookup (int index) { return m_lookups[index]; }

	uint count (void) const { return m_lookups.count(); }

private:
	LOOKUPS m_lookups;
};

/*========================================================*/
#endif	/*	__LOOKUP_H__	*/
