/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Arson wizard code.
 *
 *	by Tony Sideris	(05:32PM Nov 12, 2002)
 *================================================*/
#include "arson.h"

#include <qabstractlayout.h>
#include <qfileinfo.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qpushbutton.h>
#include <qwidgetstack.h>
#include <qfontmetrics.h>
#include <qpainter.h>
#include <qframe.h>
#include <qtl.h>

#include <kglobalsettings.h>
#include <kfiledialog.h>
#include <klocale.h>
#include <kapp.h>

#include "progressdlg.h"
#include "wizard.h"
#include "utils.h"
#include "layout.h"

/*========================================================*/
/**
 *	A generic wizard page containing an instructional
 *	label (ie: Please enter a filename), a line edit
 *	(for file/directory paths), and a Browse button.
 *	Customize in subclasses.
 */

ArsonWizardPathPage::ArsonWizardPathPage (const QString &label,
	QWidget *parent, const char *name)
	: QWidget(parent, name)
{
	ArsonLayout vl (new QVBoxLayout(this, 11, 6));
	ArsonLayout hl (new QHBoxLayout(this, 0, 4));
	QPushButton *pb = new QPushButton(i18n("&Browse..."), this);
	QSpacerItem *space = new QSpacerItem(0, 0,
		QSizePolicy::Expanding,
		QSizePolicy::Expanding);

	m_pMsg = new QLabel(label, this);
	m_pMsg->setSizePolicy(
		QSizePolicy(
			QSizePolicy::MinimumExpanding, QSizePolicy::Maximum));

	m_pEdit = new QLineEdit(this);

	hl << m_pEdit << pb;
	vl << m_pMsg << hl << space;

	QObject::connect(pb, SIGNAL(clicked()), this, SLOT(slotBrowse()));
	QObject::connect(m_pEdit, SIGNAL(textChanged(const QString &)),
		this, SLOT(slotTextChange(const QString &)));

	m_pEdit->setFocus();
}

/*========================================================*/

void ArsonWizardPathPage::setMessage (const QString &msg)
{
	m_pMsg->setText(msg);
}

QString ArsonWizardPathPage::path (void) const
{
	QString path (m_pEdit->text());

	if (path.left(2) == "~/")
		path.replace(0, 1, QDir::home().absPath());

	return path;
}

/*========================================================*/

void ArsonWizardPathPage::slotBrowse (void)
{
	const QString str (selectPath());

	if (str != QString::null)
		m_pEdit->setText(str);
}

/*========================================================*/

void ArsonWizardPathPage::slotTextChange (const QString &text)
{
	uint btns = (uint) ArsonWizard::All;

	Trace("checking: %s\n", text.latin1());
	
	if (text.isEmpty() || !isValid(text))
		btns &= ~((uint) ArsonWizard::Next);

	emit validate(btns);
}
	
/*========================================================*/
/*	A wiard path page allowing selection of a directory.
 *========================================================*/

ArsonWizardDirPage::ArsonWizardDirPage (QWidget *parent, const char *name)
	: ArsonWizardPathPage(i18n("Please select the directory you wish to burn:"),
		parent, name)
{ }

QString ArsonWizardDirPage::selectPath (void)
{
	return KFileDialog::getExistingDirectory();
}

bool ArsonWizardDirPage::isValid (const QString &text) const
{
	return QFileInfo(text).isDir();
}

/*========================================================*/
/*	A wizard path page allowing selection of an
 *	image file.
 *========================================================*/

ArsonWizardFilePage::ArsonWizardFilePage (QWidget *parent, const char *name)
	: ArsonWizardPathPage(i18n("Please select the image file you wish to burn:"),
		parent, name)
{ }

void ArsonWizardFilePage::fileNameFilter (ArsonFileFilter &filter) const
{
	filter.addFilter(ArsonFileFilter::allFiles);
}

QString ArsonWizardFilePage::selectPath (void)
{
	ArsonFileFilter filter;
	fileNameFilter(filter);

	return KFileDialog::getOpenFileName(
		QString::null,
		filter.toString());
}

bool ArsonWizardFilePage::isValid (const QString &text) const
{
	const QFileInfo fi (text);
	
	return fi.exists() && !fi.isDir();
}

/*========================================================*/
/*	The wizard manager class impl
 *========================================================*/

ArsonWizardFactory::MANAGERS *ArsonWizardFactory::allFactories = NULL;

ArsonWizardFactory::ArsonWizardFactory (const char *name, const char *desc, ActionType type)
	: m_desc(desc), m_type(type)
{
	if (!allFactories)
		allFactories = new MANAGERS;

	Assert(!allFactories->contains(name));
	allFactories->insert(name, this);
}

/*========================================================*/

ArsonWizardFactory *ArsonWizardFactory::byName (const char *name)
{
	if (allFactories->contains(name))
		return (*allFactories)[name];

	return NULL;
}

/*========================================================*/
/*	Decorative sidebar widget
 *========================================================*/

class sidebarWidget : public QFrame
{
public:
	sidebarWidget (QWidget *parent, const char *name = NULL)
		: QFrame(parent, name), XTRA(24)
	{
		QFont font (font());
//		font.setPointSize(48);
		font.setPointSize(12);
		setFont(font);

		setFrameStyle(Panel | Sunken);
		setLineWidth(2);
		setMargin(XTRA);

		setSizePolicy(
			QSizePolicy(
				QSizePolicy::Fixed,
				QSizePolicy::MinimumExpanding));

		setBackgroundColor(QColor(0,0,0));
		setBackgroundMode(FixedColor);

		m_text = "<insert cool logo or graphic>";
	}


	QSize minimumSizeHint (void) const
	{ return sizeHint(); }

	QSize sizeHint (void) const
	{
		const QRect rc (fontMetrics().boundingRect(m_text));

		return QSize(
			128, 128	//	TEMP TEMP TEMP, real code below
//			4 + rc.height() + XTRA * 2, 4 + rc.width() + XTRA * 2
			);
	}

private:
	virtual void drawContents (QPainter *ptr)
	{
		QFrame::drawContents(ptr);

		ptr->rotate(90.0);
		ptr->setPen(QColor(255,255,255));
		ptr->drawText(XTRA, -(XTRA + 2), m_text);
	}

	const int XTRA;
	QString m_text;
};

/*========================================================*/
/*	The wizard dialog class impl
 *========================================================*/

bool g_useIcons = false;

ArsonWizard::ArsonWizard (QWidget *parent, const char *name, bool modal)
	: QDialog(parent, name, modal),
	m_pages(new QWidgetStack(this, "ws")),
	m_count(0), m_page(0),
	m_title(NULL)
{
	QFont font;
	KConfig *pc = KGlobal::config();

	ArsonLayout top (new QVBoxLayout(this, 11, 6));		//	Toplevel layout
	ArsonLayout bl (new QHBoxLayout(this, 0, 6));		//	Button layout
	ArsonLayout center ((m_pCenterLayout = new QHBoxLayout(this, 0, 6)));
	ArsonLayout right ((m_pRightLayout = new QVBoxLayout(this, 0, 6)));
	QLabel *pl = new QLabel(this);

	QFrame *p1 = new QFrame(this);						//	The two divider lines
	QFrame *p2 = new QFrame(this);
	p1->setFrameStyle(QFrame::HLine | QFrame::Sunken);
	p2->setFrameStyle(QFrame::HLine | QFrame::Sunken);

	setTitleWidget(pl);
	font = m_title->font();
	font.setBold(true);
	m_title->setFont(font);
	m_title->setSizePolicy(
		QSizePolicy(
			QSizePolicy::Expanding,
			QSizePolicy::Maximum));

	right << p1 << m_pages;
	center << right;
	top << center << p2;

	setDecorationWidget(new sidebarWidget(this));

	pc->setGroup("KDE");
	g_useIcons = pc->readBoolEntry("ShowIconsOnPushButtons", false);

	createButtons();

	bl << new QSpacerItem(0, 2, QSizePolicy::Expanding, QSizePolicy::Fixed)
	   << m_back << m_next << m_finish << m_cancel;

	top << bl;

	QObject::connect(m_finish, SIGNAL(clicked()), this, SLOT(accept()));
	QObject::connect(m_cancel, SIGNAL(clicked()), this, SLOT(reject()));
	QObject::connect(m_back, SIGNAL(clicked()), this, SLOT(back()));
	QObject::connect(m_next, SIGNAL(clicked()), this, SLOT(next()));
	QObject::connect(this, SIGNAL(titleChange(const QString &)),
		m_title, SLOT(setText(const QString &)));
}

/*========================================================*/

void ArsonWizard::setCurrentPage (int page)
{
	if (InArray(page, m_count))
	{
		QWidget *was = m_pages->visibleWidget();

		m_pages->raiseWidget((m_page = page));

		if (QWidget *now = m_pages->visibleWidget())
		{
			Trace("setting current page: %p %p\n", was, now);

			emit titleChange(m_titles[m_page]);
			emit pageChange(was, now);

			enableButtons();
		}
	}
}

/*========================================================*/

QWidget *ArsonWizard::page (int page) const
{
	return m_pages->widget(page);
}

int ArsonWizard::index (QWidget *page) const
{
	return m_pages->id(page);
}

void ArsonWizard::setCurrentPage (QWidget *page)
{
	setCurrentPage(index(page));
}

/*========================================================*/

void ArsonWizard::next (void)
{
	Trace("nexting: %d/%d\n", m_page, m_count);
	
	if (m_page < (m_count - 1))
		setCurrentPage(m_page + 1);
}

void ArsonWizard::back (void)
{
	if (m_page > 0)
		setCurrentPage(m_page - 1);
}

/*========================================================*/

void ArsonWizard::addPage (ArsonWizardPathPage *ptr, const QString &text)
{
	addPage((QWidget *) ptr, text);

	QObject::connect(ptr, SIGNAL(validate(uint)),
		this, SLOT(enableButtons(uint)));

	enableButtons(Buttons(All & ~Next));
}

void ArsonWizard::addPage (ArsonProgress *ptr, const QString &text)
{
	addPage((QWidget *) ptr, text);
	ptr->setup();

	new ArsonProgressEventFilter(ptr, this,
		m_finish, m_cancel);
}

void ArsonWizard::addPage (QWidget *ptr, const QString &text)
{
	QWidget *cur = m_pages->visibleWidget();

	m_pages->addWidget(ptr, m_count++);
	m_titles.append(text);
	enableButtons();

	if (!cur)
		setCurrentPage(0);
}

/*========================================================*/

void ArsonWizard::setFactory (ArsonWizardFactory *pFactory)
{
	pFactory->addPages(this);

	setCaption(i18n("arson Wizard: ") + pFactory->description());
}

/*========================================================*/

void ArsonWizard::showWizard (const char *name)
{
	if (ArsonWizardFactory *pw = ArsonWizardFactory::byName(name))
	{
		ArsonWizard dlg (kapp->mainWidget(), NULL, true);
		dlg.setFactory(pw);
		dlg.exec();
	}
}

/*========================================================*/

void ArsonWizard::setDecorationWidget (QWidget *widget)
{
	m_pCenterLayout->insertWidget(0, widget);
}

void ArsonWizard::setTitleWidget (QWidget *title)
{
	m_pRightLayout->insertWidget(0, title);

	delete m_title;
	m_title = title;
}

/*========================================================*/

void ArsonWizard::enableButtons (void)
{
	Trace("enable buttons %d/%d\n", m_page, m_count);
	
	if (m_count > 0)
	{
		const bool last = (m_page == (m_count - 1));
		QPushButton *default_yes = m_next;
		QPushButton *default_no = m_finish;
		QWidget *hide = m_finish;
		QWidget *show = m_next;

		m_back->setEnabled(m_page > 0);

		if (last)
		{
			qSwap(default_no, default_yes);
			qSwap(hide, show);
		}

		hide->hide();
		show->show();

		default_yes->setDefault(true);
		default_no->setDefault(false);		
	}
}

void ArsonWizard::enableButtons (uint buttons)
{
	const int cur = currentPage();

	m_finish->setEnabled(buttons & Finish);
	m_cancel->setEnabled(buttons & Cancel);
	m_back->setEnabled(buttons & Back && cur > 0);
	m_next->setEnabled(buttons & Next && cur < pageCount() - 1);
}

/*========================================================*/

inline bool wizardWantIcons (void)
{
	return g_useIcons;
}

#ifdef ARSON_KDE3
#include <kstdguiitem.h>
void wizardSetupButton (QPushButton *pb,
	const KGuiItem &item,
	const QString &text = QString::null)
{
	if (wizardWantIcons())
		pb->setIconSet(item.iconSet());

	pb->setText(text == QString::null
		? item.text() : text);
}
#else
#include <kiconloader.h>
void wizardSetupButton (QPushButton *pb,
	const QString &label, const QString &icon)
{
	pb->setText(label);

	if (wizardWantIcons())
		pb->setIcon(KGlobal::iconLoader()
			->loadIcon(icon, KIcon::Small));
}
#endif

void ArsonWizard::createButtons (void)
{
	m_finish = new QPushButton(this);
	m_cancel = new QPushButton(this);
	m_back = new QPushButton(this);
	m_next = new QPushButton(this);

#ifdef ARSON_KDE3
	wizardSetupButton(m_finish, KStdGuiItem::apply(), i18n("&Finish"));
	wizardSetupButton(m_cancel, KStdGuiItem::cancel());
	wizardSetupButton(m_next, KStdGuiItem::forward());
	wizardSetupButton(m_back, KStdGuiItem::back());
#else
	wizardSetupButton(m_finish, i18n("&Finish"), "apply");
	wizardSetupButton(m_cancel, i18n("&Cancel"), "cancel");
	wizardSetupButton(m_back, i18n("&Back"), "back");
	wizardSetupButton(m_next, i18n("&Next"), "forward");
#endif	//	ARSON_KDE3

	if (!wizardWantIcons())
	{
		m_next->setText(i18n("&Next >>"));
		m_back->setText(i18n("<< &Back"));
	}
}

/*========================================================*/
