#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <pthread.h>

#include <auth.h>
#include <protocol-asd-spec.h>
#include <volume.h>
#include <asdutil.h>

#include "asdlib.h"
#include "asderrno.h"
#include "cmdidgen.h"
#include "unix.h"
#include "inet.h"
#include "io.h"


struct _AsdConnection
{
  int fd;
  gboolean auth:1;
  AsdConnectionState state;
};

static int _parse_speaker(gchar *e)
{
  gchar s[256];
  gchar *c;

  strncpy(s, e, sizeof(s));
  if (!(c = strchr(s, ':')))
    {
      asd_set_errno(PROTOCOL_ASD_LOCAL_ERROR_BROKEN_SERVER_SPEC);  
      return -1;
    }

  *(c++) = 0;

  if (strcmp(s, "unix") == 0)
    return unix_try_connect(*c ? c : NULL);

  if (strcmp(s, "inet") == 0)
    {
      gchar *p;
      
      if (!(p = strchr(c, ':')))
	return inet_try_connect(*c ? c : NULL, 0);
      else
	{
	  *(p++) = 0;
	  return inet_try_connect(*c ? c : NULL, atoi(p));
	}
    }

  asd_set_errno(PROTOCOL_ASD_LOCAL_ERROR_BROKEN_SERVER_SPEC);  

  return -1;
}


static gboolean assert_in_control(AsdConnection *c)
{
  g_assert(c);
  if (c->state != ASD_CONNECTION_CONTROL)
    asd_set_errno(PROTOCOL_ASD_LOCAL_ERROR_NOT_IN_CONTROL_STATE);

  return c->state == ASD_CONNECTION_CONTROL;
}

AsdConnection* asd_connection_new(gchar *speaker)
{  
  AsdConnection *c;

  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);

  c = g_new(AsdConnection, 1);
  c->auth = FALSE;
  c->state = ASD_CONNECTION_CONTROL;
  pthread_cleanup_push(g_free, c);

  if (speaker)
    if (!*speaker)
      speaker = NULL;
  
  if (!speaker)
    speaker = getenv("ASDSPEAKER");
  
  if (speaker)
    c->fd = _parse_speaker(speaker);
  else if ((c->fd = unix_try_connect(NULL)) < 0)
    c->fd = inet_try_connect(NULL, 0);
  
  pthread_cleanup_pop(0);

  if (c->fd < 0)
    {
      g_free(c);
      return FALSE;
    }

  return c;
}

void asd_connection_free(AsdConnection *c)
{
  g_assert(c);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);
  close(c->fd);
  g_free(c);
}

int asd_connection_fd(AsdConnection *c)
{
  g_assert(c);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);
  return c->fd;
}

gboolean asd_connection_authenticated(AsdConnection *c)
{
  g_assert(c);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);
  return c->auth;
}

AsdConnectionState asd_connection_state(AsdConnection *c)
{
  g_assert(c);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);
  return c->state;
}

gboolean asd_authenticate(AsdConnection *c)
{
  guint8 dummy_cookie[PROTOCOL_ASD_COOKIE_SIZE];
  guint8* cookie;
  guint16 id;
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);

   if (!assert_in_control(c))
    return FALSE;

  if (c->auth)
    return TRUE;

  auth_init(FALSE);
  
  if (auth_cookie_available)
    cookie = auth_cookie;
  else
    {
      memset(dummy_cookie, 0x0, PROTOCOL_ASD_COOKIE_SIZE);
      cookie = dummy_cookie;
    }

  id = generate_command_id();

  if (asd_write_request(c->fd, id, PROTOCOL_ASD_COMMAND_AUTHENTICATE, PROTOCOL_ASD_COOKIE_SIZE))
    if (asd_write(c->fd, cookie, PROTOCOL_ASD_COOKIE_SIZE))
      if (asd_read_ack(c->fd, id, 0))
        {
          c->auth = TRUE;
          return TRUE;
        }

  return FALSE;
}

gboolean asd_server_version(AsdConnection *c, gchar *s, guint l)
{
  ProtocolAsdServerVersionResponse version;
  guint16 id;
  g_assert(c && s && l);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);
 
  if (!assert_in_control(c))
    return FALSE;

  id = generate_command_id();

  if (asd_write_request(c->fd, id, PROTOCOL_ASD_COMMAND_SERVER_VERSION, 0))
    if (asd_read_ack(c->fd, id, sizeof(version)))
      if (asd_read(c->fd, &version, sizeof(version)))
	{
	  strncpy(s, version.version, l);
	  return TRUE;
	}

  return FALSE;
}

gboolean asd_server_info(AsdConnection *c, ProtocolAsdServerInfoResponse *info)
{
  guint16 id;
  g_assert(c && info);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);

  if (!assert_in_control(c))
    return FALSE;

  asd_authenticate(c);

  id = generate_command_id();

  if (asd_write_request(c->fd, id, PROTOCOL_ASD_COMMAND_SERVER_INFO, 0))
    if (asd_read_ack(c->fd, id, sizeof(ProtocolAsdServerInfoResponse)))
      if (asd_read(c->fd, info, sizeof(ProtocolAsdServerInfoResponse)))
	{
	  info->block_size = GUINT32_FROM_LE(info->block_size);
	  sample_type_from_le(&info->sample_type);
	  info->alloc_blocks = GUINT16_FROM_LE(info->alloc_blocks);
	  info->average_latency = GUINT32_FROM_LE(info->average_latency);
	  info->string[sizeof(info->string)-1] = 0;
	  info->version[sizeof(info->version)-1] = 0;
	  return TRUE;
	}

  return FALSE;
}


gboolean asd_stream(AsdConnection *c, 
                    AsdStreamAction action, 
                    gchar *name, 
                    gchar *device, 
                    SampleType *sample_type, 
                    Volume *volume, 
                    gboolean immediate_stop, 
                    guint8 q_len, 
                    guint8 q_hold, 
                    gchar* shortname, 
                    guint shortname_size, 
                    guint32 *block_size)
{
  ProtocolAsdStreamQuery query;
  ProtocolAsdStreamResponse response;
  guint16 id;
  SampleType st;
  Volume v;
  guint cmd;
  g_assert(c && (action >= ASD_STREAM_PLAY) && (action <= ASD_STREAM_SINK));

  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);
  
  if (!assert_in_control(c))
    return FALSE;

  if (!name)
    name = g_get_prgname();
  
  if (!name)
    name = "client";

  if (!device)
    device = ((action == ASD_STREAM_PLAY) || (action == ASD_STREAM_MONITOR)) ? 
      ASD_DEFAULT_PLAY_SINK : ASD_DEFAULT_CAPTURE_SOURCE;

  if ((action == ASD_STREAM_SOURCE) || (action == ASD_STREAM_SINK))
    device = "";

  if (!sample_type)
    {
      st.channels = 2;
      st.bits = 16;
      st.be = FALSE;
      st.sign = TRUE;
      st.rate = 44100;
      sample_type = &st;
    }

  if (!sample_type_valid(sample_type))
    {
      asd_set_errno(PROTOCOL_ASD_LOCAL_ERROR_SAMPLE_TYPE_INVALID);
      return FALSE;
    }

  if (!volume)
    {
      volume_max(&v);
      volume = &v;
    }

  switch (action) 
    {
    case ASD_STREAM_PLAY:    cmd = PROTOCOL_ASD_COMMAND_STREAM_PLAY; break;
    case ASD_STREAM_CAPTURE: cmd = PROTOCOL_ASD_COMMAND_STREAM_CAPTURE; break;
    case ASD_STREAM_MONITOR: cmd = PROTOCOL_ASD_COMMAND_STREAM_MONITOR; break;
    case ASD_STREAM_SOURCE:  cmd = PROTOCOL_ASD_COMMAND_STREAM_SOURCE; break;
    case ASD_STREAM_SINK:    cmd = PROTOCOL_ASD_COMMAND_STREAM_SINK; break;
    default : 
      cmd = 0;
      g_error("This should not happen.");
    }
  
  asd_authenticate(c);
  
  id = generate_command_id();

  if (asd_write_request(c->fd, id, cmd, sizeof(query)))
    {
      strncpy(query.name, name, sizeof(query.name));
      strncpy(query.device, device, sizeof(query.device));
      query.sample_type = *sample_type;
      sample_type_to_le(&query.sample_type);
      query.volume = *volume;
      volume_to_le(&query.volume);
      query.immediate_stop = immediate_stop;
      query.queue_length = q_len;
      query.queue_hold = q_hold;

      if (atomic_write(c->fd, &query, sizeof(query)) == sizeof(query))
	if (asd_read_ack(c->fd, id, sizeof(response)))
	  if (atomic_read(c->fd, &response, sizeof(response)) == sizeof(response))
	    {
	      if (block_size)
		*block_size = GUINT32_FROM_LE(response.block_size);
	      
	      if (shortname)
		{
		  response.shortname[sizeof(response.shortname)-1] = 0;
		  strncpy(shortname, response.shortname, shortname_size);
		}

	      return TRUE;
	    }
    }
    
  return FALSE;
}

gboolean asd_stream_simple_play(AsdConnection *c, SampleType *sample_type)
{
  return asd_stream(c, ASD_STREAM_PLAY, NULL, NULL, sample_type, NULL, FALSE, 0, 0, NULL, 0, NULL);
}

gboolean asd_stream_simple_capture(AsdConnection *c, SampleType *sample_type)
{
  return asd_stream(c, ASD_STREAM_CAPTURE, NULL, NULL, sample_type, NULL, FALSE, 0, 0, NULL, 0, NULL);
}

gboolean asd_stream_simple_monitor(AsdConnection *c, SampleType *sample_type)
{
  return asd_stream(c, ASD_STREAM_MONITOR, NULL, NULL, sample_type, NULL, FALSE, 0, 0, NULL, 0, NULL);
}

gboolean asd_lock(AsdConnection *c, gboolean b)
{
  guint16 id;
  g_assert(c);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);

  if (!assert_in_control(c))
    return FALSE;

  asd_authenticate(c);

  id = generate_command_id();

  if (asd_write_request(c->fd, id, b ? PROTOCOL_ASD_COMMAND_LOCK : PROTOCOL_ASD_COMMAND_UNLOCK, 0))
    if (asd_read_ack(c->fd, id, 0))
      return TRUE;
 
  return FALSE;
}

gboolean asd_volume_set(AsdConnection *c, gchar *shortname, Volume v)
{
  guint16 id;
  ProtocolAsdVolumeSetQuery query;

  g_assert(c && shortname);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);

  if (!assert_in_control(c))
    return FALSE;

  strncpy(query.shortname, shortname, sizeof(query.shortname));
  query.volume = v;
  volume_to_le(&query.volume);

  asd_authenticate(c);
  id = generate_command_id();

  if (asd_write_request(c->fd, id, PROTOCOL_ASD_COMMAND_VOLUME_SET, sizeof(query)))
    if (atomic_write(c->fd, &query, sizeof(query)) == sizeof(query))
      if (asd_read_ack(c->fd, id, 0))
        return TRUE;

  return FALSE;
}

gboolean asd_volume_get(AsdConnection *c, gchar *shortname, Volume *v)
{
  guint16 id;
  ProtocolAsdVolumeGetQuery query;
  ProtocolAsdVolumeGetResponse response;
  g_assert(c && shortname && v);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);

  if (!assert_in_control(c))
    return FALSE;

  strncpy(query.shortname, shortname, sizeof(query.shortname));

  asd_authenticate(c);
  id = generate_command_id();

  if (asd_write_request(c->fd, id, PROTOCOL_ASD_COMMAND_VOLUME_GET, sizeof(query)))
    if (atomic_write(c->fd, &query, sizeof(query)) == sizeof(query))
      if (asd_read_ack(c->fd, id, sizeof(response)))
        if (atomic_read(c->fd, &response, sizeof(response)) == sizeof(response))
          {
            volume_from_le(&response.volume);
            *v = response.volume;
            return TRUE;
          }

  return FALSE;
}

gboolean asd_info_source(AsdConnection *c, gchar *shortname, ProtocolAsdInfoSourceResponse *info)
{
  guint16 id;
  ProtocolAsdInfoQuery query;
  g_assert(c && shortname && info);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);

  if (!assert_in_control(c))
    return FALSE;

  strncpy(query.shortname, shortname, sizeof(query.shortname));

  asd_authenticate(c);
  id = generate_command_id();

  if (asd_write_request(c->fd, id, PROTOCOL_ASD_COMMAND_INFO_SOURCE, sizeof(query)))
    if (atomic_write(c->fd, &query, sizeof(query)) == sizeof(query))
      if (asd_read_ack(c->fd, id, sizeof(ProtocolAsdInfoSourceResponse)))
        if (atomic_read(c->fd, info, sizeof(ProtocolAsdInfoSourceResponse)) == sizeof(ProtocolAsdInfoSourceResponse))
          {
            info->shortname[ASD_SHORTNAME_LENGTH-1] = 0;
            info->name[ASD_NAME_LENGTH-1] = 0;
            info->type[ASD_TYPE_LENGTH-1] = 0;
            sample_type_from_le(&info->sample_type);
            volume_from_le(&info->volume);
            info->flags = GUINT32_FROM_LE(info->flags);
            info->throughput = GUINT32_FROM_LE(info->throughput);
            info->latency = GUINT32_FROM_LE(info->latency);
            info->byte_counter = GUINT32_FROM_LE(info->byte_counter);
            return TRUE;
          }

  return FALSE;
}


gboolean asd_info_sink(AsdConnection *c, gchar *shortname, ProtocolAsdInfoSinkResponse *info)
{
  guint16 id;
  ProtocolAsdInfoQuery query;
  g_assert(c && shortname && info);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);

  if (!assert_in_control(c))
    return FALSE;

  strncpy(query.shortname, shortname, sizeof(query.shortname));

  asd_authenticate(c);
  id = generate_command_id();

  if (asd_write_request(c->fd, id, PROTOCOL_ASD_COMMAND_INFO_SINK, sizeof(query)))
    if (atomic_write(c->fd, &query, sizeof(query)) == sizeof(query))
      if (asd_read_ack(c->fd, id, sizeof(ProtocolAsdInfoSinkResponse)))
        if (atomic_read(c->fd, info, sizeof(ProtocolAsdInfoSinkResponse)) == sizeof(ProtocolAsdInfoSinkResponse))
          {
            info->shortname[ASD_SHORTNAME_LENGTH-1] = 0;
            info->name[ASD_NAME_LENGTH-1] = 0;
            info->type[ASD_TYPE_LENGTH-1] = 0;
            sample_type_from_le(&info->sample_type);
            volume_from_le(&info->volume);
            info->flags = GUINT32_FROM_LE(info->flags);
            info->throughput = GUINT32_FROM_LE(info->throughput);
            info->byte_counter = GUINT32_FROM_LE(info->byte_counter);
            return TRUE;
          }

  return FALSE;
}

gboolean asd_list(AsdConnection *c, AsdListProc proc, gpointer userdata, gboolean source)
{
  guint16 id;
  g_assert(c && proc);
  asd_set_errno(PROTOCOL_ASD_ERROR_SUCCESS);

  if (!assert_in_control(c))
    return FALSE;

  asd_authenticate(c);
  id = generate_command_id();

  if (asd_write_request(c->fd, id, source ? PROTOCOL_ASD_COMMAND_LIST_SOURCES : PROTOCOL_ASD_COMMAND_LIST_SINKS, 0))
    if (asd_read_ack(c->fd, id, -1))
      {
        GSList *l = NULL, *it;

        for (;;)
          {
            ProtocolAsdListResponse *r = g_new0(ProtocolAsdListResponse, 1);
            if (atomic_read(c->fd, r, sizeof(ProtocolAsdListResponse)) != sizeof(ProtocolAsdListResponse))
              {
                GSList *it;
                g_free(r);
                
                it = l;
                while (it)
                  {
                    g_free(l->data);
                    it = it->next;
                  }
                g_slist_free(l);
                  
                return FALSE;
              }
            
            if (!r->shortname[0])
              {
                g_free(r);
                break;
              }

            r->shortname[ASD_SHORTNAME_LENGTH-1] = 0;
            r->name[ASD_NAME_LENGTH-1] = 0;
            r->type[ASD_TYPE_LENGTH-1] = 0;

            l = g_slist_prepend(l, r);
          }
        
        
        it = l;
        while (it)
          {
            proc((ProtocolAsdListResponse*) it->data, userdata);
            g_free(it->data);
            it = it->next;
          }

        g_slist_free(l);

        return TRUE;
      }

  return FALSE;
}
