#ifndef sinkh
#define sinkh

#include <glib.h>
#include <pthread.h>
#include <libxml/tree.h>

#include <recursive-mutex.h>
#include <notify.h>
#include <sample-type.h>
#include <volume.h>

struct _Sink;
typedef struct _Sink Sink;

#include <throughput.h>
#include <block.h>
#include <block-queue.h>
#include <source.h>
#include <byte-counter.h>

struct _Sink
{
  gchar *shortname;
  gchar *name;
  gchar *type; /*static string!*/

  SampleType sample_type;
  Volume volume;
  
  SinkMode mode;
  guint flags;

  pthread_t thread;
  gboolean thread_running;

  Throughput* throughput;

  GSList *input_block_queues;
  RecursiveMutex *input_block_queues_mutex;

  GSList *monitor_block_queues;
  RecursiveMutex *monitor_block_queues_mutex;

  Source *direct_source;

  Notify notify;

  LatencyBank latency_bank;

  RecursiveMutex *byte_counter_mutex;
  Origin current_block_origin;
  ByteCounter byte_counter;
  gulong byte_counter_since_open;

  gpointer private_data;
  gboolean (*open) (Sink *);
  void (*close) (Sink *);
  void (*flush) (Sink *);
  gboolean (*reopen) (Sink *);
  glong (*write) (Sink *, guint8*, gulong);
  glong (*direct_write) (Sink *, guint8*, gulong);
  gboolean (*write_block) (Sink *, Block *);
  gulong (*get_current_byte) (Sink *s);
  void (*free_private) (Sink *);
  void (*save_private) (Sink *s, xmlNodePtr node);
};

// Creates an abstract instance of a sink
Sink* sink_new(gchar *shortname, gchar *name, gboolean ns_reg);

// Adds/removes a block queue to/from the list of connected input block queues
void sink_add_input_block_queue(Sink *s, BlockQueue *b);
void sink_remove_input_block_queue(Sink *s, BlockQueue *b);

// Adds/removes a block queue to/from the list of connected monitoring block queues
void sink_add_monitor_block_queue(Sink *s, BlockQueue *b);
void sink_remove_monitor_block_queue(Sink *s, BlockQueue *b);

// Convenience functions for conjunction manager
void sink_input_block_queue_death_proc(gpointer p, BlockQueue*b);
void sink_monitor_block_queue_death_proc(gpointer p, BlockQueue*b);

// PRIVATE: Reads a block from all connected block queues
Block* sink_input_read_block(Sink *s);

// PRIVATE: Writes a block to all connected monitoring block queues
gboolean sink_monitor_write_block(Sink *s, Block *b);

// PRIVATE: Enables all connected block queues
void sink_input_block_queues_enable(Sink *s, gboolean b);
void sink_monitor_block_queues_enable(Sink *s, gboolean b);

// Starts the thread
void sink_start(Sink *s);

// Stops the thread
void sink_stop(Sink *s);

// Stops the thread (if available) and removes all conjunctions
void sink_kill(Sink *s);

// PRIVATE: Changes the mode of the sink
void sink_set_mode(Sink *s, SinkMode m);

// Links a sink directly to a source
gboolean sink_direct(Sink *s, gboolean enable, Source* source);

// Sets the volume of the sink
void sink_set_volume(Sink *s, Volume *v);

#endif
