// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CFunctionInfo_h__
#define __CFunctionInfo_h__

// function info (standard functions and methods)

#include "Puma/CStructure.h"
#include "Puma/CTypeInfo.h"
#include "Puma/Array.h"

namespace Puma {


class CArgumentInfo;
class CLabelInfo;
class CRecord;
class CNamespaceInfo;
class CTemplateInfo;

/** \file 
 *  Representation of a function.
 *  \see Puma::CFunctionInfo */

/** \class CFunctionInfo CFunctionInfo.h Puma/infos/CFunctionInfo.h
 *  A CFunctionInfo object represents a function declaration. */
class CFunctionInfo : public CStructure {
  Array<CArgumentInfo*> _Arguments;
  Array<CLabelInfo*> _Labels;

  CTypeInfo     *_ConversionType;
  CTemplateInfo *_TemplateInfo;
  bool           _isTemplate;

  enum MethodType {
    NONE,
    CONSTRUCTOR,
    DESTRUCTOR,
    OPERATOR,
    CONVERSION
  } _MethodType;

protected:
  /** This constructor is to be used by classes inherited from
   *  CFunctionInfo. It makes it possible to set another object id
   *  than \c FUNCTION_INFO for abstractions not representing a 
   *  standard function (e.g. CFctInstance).
   *  \param id The object id to be used instead of \c FUNCTION_INFO. */
  CFunctionInfo (ObjectId id);

public: 
  /** Construct a new function info. */
  CFunctionInfo ();
  /** Destruct the function info. */
  ~CFunctionInfo ();
  
//  bool operator ==(const CFunctionInfo &) const;  

  /** Get the base info object of the info object. <b>The 
   *  corresponding information is not yet set!</b>
   *  \return This method always returns \c NULL.
   *  \warning Do not use this method. */
  CFunctionInfo *BaseObject () const;
  /** Get the info object of the function definition. 
   *  \return The info or \e this if the function is not defined. */
  CFunctionInfo *DefObject () const;
  /** Get the type information object for the function. */
  CTypeFunction *TypeInfo () const;
  /** Return the conversion type if this is a conversion function.
   *  \return The conversion type or \c NULL if not a conversion function. */
  CTypeInfo *ConversionType () const;
  /** Return the return type of the function. */
  CTypeInfo *ReturnType () const;
  /** Get the template info if this is a function template.
   *  \return The template info or \c NULL if not a function template. */
  CTemplateInfo *TemplateInfo () const;
  /** Return the class or union if this is method of a class or union.
   *  \return The record info or \c NULL if not a method. */
  CRecord *Record () const;
  /** Get the namespace info if this function is declared in a namespace.
   *  \return The namespace info or \c NULL if not declared in a namespace. */
  CNamespaceInfo *Namespace () const;

  // Get ...

  /** Get the number of jump labels defined in the function body. */
  unsigned Labels () const;
  /** Get the number of function parameters. */
  unsigned Arguments () const;
  /** Get the n-th jump label.
   *  \param n The number of the jump label.
   *  \return The label info or \c NULL if \e n is invalid. */
  CLabelInfo *Label (unsigned n) const;
  /** Get the jump label with the given name.
   *  \param name The name of the jump label.
   *  \return The label info or \c NULL if no such label. */
  CLabelInfo *Label (const char *name) const;
  /** Get the n-th function parameter.
   *  \param n The number of the function parameter. 
   *  \return The function parameter info or \c NULL if \e n is invalid. */
  CArgumentInfo *Argument (unsigned n) const;
  /** Get the function parameter with the given name. 
   *  \param name The name of the function parameter.
   *  \return The function parameter info or \c NULL if no such parameter. */
  CArgumentInfo *Argument (const char *name) const;
  /** Get the default argument of the n-th function parameter.
   *  \param The number of the function parameter.
   *  \return The syntax tree node of the default argument or 
   *          \c NULL if the n-th parameter has no default argument. */
  CTree *DefaultArgument (unsigned n) const;
  /** Get the initializer of the function. Only pure virtual functions
   *  have an initializer, i.e. "=0".
   *  \return The syntax tree node of the initializer or \c NULL
   *          if function is not pure virtual. */
  CT_ExprList *Init () const;

  // Set ... 
  
  /** Set the conversion type of a conversion function. 
   *  \param type The conversion type. */
  void ConversionType (CTypeInfo *type);
  /** Set the qualification scope of a class/union method or 
   *  function declared in a namespace.
   *  \param scope The qualification scope. */
  void QualifiedScope (CStructure *scope);
  /** Add a function parameter to the function. 
   *  \param info The function parameter info. */
  void addArgument (CArgumentInfo *info);
  /** Add a jump label to the function.
   *  \param label The label info. */
  void addLabel (CLabelInfo *label);
  /** Remove the given function parameter from the function.
   *  \param info The function parameter info. */
  void removeArgument (const CArgumentInfo *);
  /** Remove the given jump label from the function.
   *  \param label The label info. */
  void removeLabel (const CLabelInfo *label);
  /** Set the template info of a function template.
   *  \param info The template info. */
  void TemplateInfo (CTemplateInfo *info);
  /** Set whether the function is a function template.
   *  \param v \c true if the function is a template. */
  void isTemplate (bool v);

  // Create/Delete ...
  
  /** Create a new function parameter. The function parameter info 
   *  is added to the function. */
  CArgumentInfo *newArgument ();
  /** Create a new jump label. The jump label info is added
   *  to the function. */
  CLabelInfo *newLabel ();
  /** Remove and destroy the given function parameter. 
   *  \param info The function parameter info. */
  void deleteArgument (const CArgumentInfo *info);
  /** Remove and destroy the given jump label. 
   *  \param info The jump label info. */
  void deleteLabel (const CLabelInfo *info);

  // additional information
  
  /** Return \e true if the function accepts a variable argument list. */
  bool hasEllipsis () const;
  /** Return \e true if the n-th function parameter has a default argument.
   *  \param n The number of the function parameter. */
  bool hasDefaultArgument (unsigned n) const;
  
  /** Return \e true if this is the info of a function definition. */
  bool isFctDef () const;                    
  /** Return \e true if this is a method of a class or union. */
  bool isMethod () const;                    
  /** Return \e true if this is a static method of a class or union. */
  bool isStaticMethod () const;          
  /** Return \e true if this is a function template. */
  bool isTemplate () const;        
  /** Return \e true if the function is defined. */
  bool isDefined () const;              
  /** Return \e true if this is a destructor. */
  bool isDestructor () const;          
  /** Return \e true if this is a constructor. */
  bool isConstructor () const;              
  /** Return \e true if this is an overloaded operator. */
  bool isOperator () const;           
  /** Return \e true if this is a conversion function. */
  bool isConversion () const;     

  /** Set whether the function is a destructor. 
   *  \param v \e true if the function is a destructor. */
  void isDestructor (bool v);
  /** Set whether the function is a constructor. 
   *  \param v \e true if the function is a constructor. */
  void isConstructor (bool v);
  /** Set whether the function is an overloaded operator. 
   *  \param v \e true if the function is an overloaded operator. */
  void isOperator (bool v);
  /** Set whether the function is a conversion function. 
   *  \param v \e true if the function is a conversion function. */
  void isConversion (bool v);

//  bool isRedefined () const;
  /** Return \e true if the function is pure virtual. */
  bool isPureVirtual () const;
  
  /** Returns \e true if the argument function has the same name and
   *  parameter types
   *  \param fi function info object of the compared function. */
  bool hasSameNameAndArgs (const CFunctionInfo *fi) const;
  
  /** Returns \e true if the function is a non-static member function and
   *  if in any of the base classes there is a function definition of a
   *  virtual function with the same name and argument types */
  bool overridesVirtual () const;
};

inline CFunctionInfo::CFunctionInfo (ObjectId id) :
  CStructure (id),
  _ConversionType ((CTypeInfo*)0),
  _TemplateInfo ((CTemplateInfo*)0),
  _isTemplate (false),
  _MethodType (NONE)
 {}
inline CFunctionInfo::CFunctionInfo () :
  CStructure (CObjectInfo::FUNCTION_INFO),
  _ConversionType ((CTypeInfo*)0),
  _TemplateInfo ((CTemplateInfo*)0),
  _isTemplate (false),
  _MethodType (NONE)
 {}

inline CTypeInfo *CFunctionInfo::ConversionType () const
 { return _ConversionType; }
inline void CFunctionInfo::ConversionType (CTypeInfo *t)
 { _ConversionType = t; }
 
inline CTypeFunction *CFunctionInfo::TypeInfo () const
 { return (CTypeFunction*)CObjectInfo::TypeInfo (); }
inline CFunctionInfo *CFunctionInfo::BaseObject () const
 { return (CFunctionInfo*)CObjectInfo::BaseObject (); }

inline unsigned CFunctionInfo::Arguments () const
 { return _Arguments.length (); }
inline CArgumentInfo *CFunctionInfo::Argument (unsigned n) const
 { return _Arguments.lookup (n); }
inline unsigned CFunctionInfo::Labels () const
 { return _Labels.length (); }
inline CLabelInfo *CFunctionInfo::Label (unsigned n) const
 { return _Labels.lookup (n); }

inline void CFunctionInfo::QualifiedScope (CStructure *s)
 { _QualScope = s; }
 
inline bool CFunctionInfo::isDestructor () const
 { return _MethodType == DESTRUCTOR; }
inline bool CFunctionInfo::isConstructor () const
 { return _MethodType == CONSTRUCTOR; }
inline bool CFunctionInfo::isOperator () const
 { return _MethodType == OPERATOR; }
inline bool CFunctionInfo::isConversion () const
 { return _MethodType == CONVERSION; }
inline bool CFunctionInfo::isMethod () const 
 { return (Record ()); }
inline bool CFunctionInfo::isStaticMethod () const 
 { return (isMethod () && isStatic ()); }

inline void CFunctionInfo::isDestructor (bool v)
 { if (v) _MethodType = DESTRUCTOR; }
inline void CFunctionInfo::isConstructor (bool v)
 { if (v) _MethodType = CONSTRUCTOR; }
inline void CFunctionInfo::isOperator (bool v)
 { if (v) _MethodType = OPERATOR; }
inline void CFunctionInfo::isConversion (bool v)
 { if (v) _MethodType = CONVERSION; }

inline bool CFunctionInfo::isTemplate () const 
 { return _isTemplate; }
inline void CFunctionInfo::isTemplate (bool v)
 { _isTemplate = v; }

inline CTemplateInfo *CFunctionInfo::TemplateInfo () const
 { return _TemplateInfo; }
inline void CFunctionInfo::TemplateInfo (CTemplateInfo *info) 
 { _TemplateInfo = info; }

//inline bool CFunctionInfo::isRedefined () const 
// { return BaseObject () && ! isInherited (); }


} // namespace Puma

#endif /* __CFunctionInfo_h__ */
