// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __recognizer_h__
#define __recognizer_h__

// The token recognizer class, which uses scanner tables generated by the
// orange tool.

#include "Puma/ScanBuffer.h"

namespace Puma {


#ifndef __TTGEN__
#  define __TTGEN__
typedef struct
 { unsigned char Type;
   unsigned char LookAhead;
   int           Expression;
   int           Start;
 } States;                                                                   
#endif


// class declaration

template <typename TScanBuffer>
class OrangeRecognizer {
  // local structures generated by Orange

  char *_map;
  int _start;
  States *_state;
  int *_next;
  int *_control;
  int _tab_size;

public:
  OrangeRecognizer (char *map, int start, States *state, int *next,
                    int *control, int tab_size) : 
    _map (map), _start (start), _state (state), _next (next),
    _control (control), _tab_size (tab_size)
   {}

  // try to recognize the current token in the character buffer
  // returns 1 on success, 0 on eof, -1 on unknown token
  int recognize (TScanBuffer *scan_buffer, int &token, int &len); 
};

template <typename TScanBuffer>
int OrangeRecognizer<TScanBuffer>::recognize (TScanBuffer *scan_buffer,
                                              int &token, int &len) {
  int  NextState;
  char Character;
  char CharClass;
  int  NextStatePos;
  int  State;
  int  LookAheadLen;
  int  AcceptLen;
  int  AcceptState;
  ScanBuffer::State buffer_state = ScanBuffer::STATE_OK;

  State = _start;
  AcceptState = -1;
  LookAheadLen = 0;
  AcceptLen = 0;

  do { 
    Character = scan_buffer->next ();

    CharClass = _map[(unsigned char)Character];
    if (CharClass == -1)
      break;

    NextStatePos = _state[State].Start + CharClass;
    if (NextStatePos < 0 || NextStatePos >= _tab_size ||
        _control[NextStatePos] != State)
      break;

    if ((NextState = _next[NextStatePos]) == -1)
      break;

    State = NextState;

    if (_state[State].Type > 0) {
      AcceptState = State; // remember this (terminal) state
      AcceptLen = scan_buffer->len ();
    }

    if (_state[State].LookAhead)
      LookAheadLen = scan_buffer->len ();

    buffer_state = scan_buffer->state ();
  } while (NextState != -1 && buffer_state == ScanBuffer::STATE_OK);

  if (buffer_state == ScanBuffer::STATE_ERROR)
    return -1;

  if (AcceptState == -1)
    return 0;

  len   = (_state[AcceptState].Type == 2) ? LookAheadLen : AcceptLen;
  token = _state[AcceptState].Expression;

  return 1;
}

} // namespace Puma

#endif /* __orange_recognizer_h__ */
