// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __File_h__
#define __File_h__

#include <string>
using std::string;
#include <set>
using std::set;
#include <iostream>
using std::cout;

#include "RepoXMLNode.h"
#include "JoinPointModelElement.h"

class File : public JoinPointModelElement {
  string _file_name;
  int _len;
  set<int> _tunits; // set for header files => which tunits include this file?
  bool _is_tunit;
public:
  File () {}
  File (const string &file_name, int len, bool is_tunit) :
    _file_name (file_name), _len (len), _is_tunit (is_tunit) {}
  const string &name () const { return _file_name; }
  int len () const { return _len; }
  bool is_tunit () const { return _is_tunit; }
  bool is_in_no_tunit () const { return !_is_tunit && _tunits.size () == 0; }
  bool is_known_in_tunit (File *tunit) const {
    return (_is_tunit && tunit->id () == id ()) ||
           (!_is_tunit && _tunits.find (tunit->id ()) != _tunits.end ());
  }

  void included_by (File *f) { _tunits.insert (f->id ()); }
  void not_included_by (File *f) { _tunits.erase (f->id ()); }

  void print (int indent = 0) const {
    for (int i = 0; i < indent; i++)
      cout << "  ";
    cout << (is_tunit () ? "tunit " : "header ") << id () << " name " 
         << _file_name << " len " << _len;
    if (!is_tunit ()) {
      cout << " tunits";
      for (set<int>::const_iterator i = _tunits.begin ();
        i != _tunits.end (); ++i)
        cout << " " << *i;
      cout << endl;
    }
  }
  bool operator < (const File &f) const {
    return _file_name < f._file_name;
  }
  bool operator == (const File &f) const {
    return _file_name == f._file_name;
  }
  void to_xml (RepoXMLNode parent) const {
    RepoXMLNode fn = parent.make_child (is_tunit () ? "tunit" : "header");
    fn.set_int_prop ("id", id ());
    fn.set_str_prop ("name", _file_name.c_str ());
    fn.set_int_prop ("len", _len);
    if (!is_tunit ()) {
      ostringstream buf;
      for (set<int>::const_iterator i = _tunits.begin ();
        i != _tunits.end (); ++i) {
        if (i != _tunits.begin ())
          buf << " ";
        buf << *i;
      }
      fn.set_str_prop ("tunits", buf.str ().c_str ());
    }
  }
  void from_xml (RepoXMLNode fn) {
    _is_tunit  = fn.has_name ("tunit");
    id (fn.get_int_prop ("id"));
    _len       = fn.get_int_prop ("len");
    _file_name = fn.get_str_prop ("name");
    if (fn.has_prop ("tunits")) {
      string tunits = fn.get_str_prop ("tunits");
      istringstream in (tunits, istringstream::in);
      int id;
      while (!in.eof ()) {
        in >> id;
        if (!in.fail ())
          _tunits.insert (id);
      }  
    }
  }
};

#endif // __File_h__
