// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Utils.h"

// return various string representations

string signature(const ACM_Name &name_param) {
  ACM_Name &name = (ACM_Name&)name_param;
  ACM_Name *parent = (ACM_Name*)name.get_parent();
  if (!parent)
    return "::";
  string result = signature (*parent);
  if (result == "::")
    result = "";
  else {
    result += "::";
    // ignore the anonymous namespace as parent
    if (result[0] == '%')
      result = "";
  }
  result += name.get_name();

  if (name.type_val() & JPT_Function) {
    ACM_Function &func = (ACM_Function&)name;
    list<ACM_Type*> types;
    get_arg_types (func, types);
    result += "(";
    for (list<ACM_Type*>::iterator i = types.begin(); i != types.end(); ++i) {
      if (i != types.begin())
        result +=",";
      result += format_type(**i);
    }
    if (func.get_variadic_args())
      result += ",...";
    result += ")";
    if (func.get_cv_qualifiers() & CVQ_CONST)
      result += " const";
    if (func.get_cv_qualifiers() & CVQ_VOLATILE)
      result += " volatile";
    if (((ACM_Function &)name).has_result_type()) {
      ACM_Type &result_type = *((ACM_Function &)name).get_result_type();
      if (name.get_name().find("operator ") != 0)
        result = format_type (result_type, result);
      else {
        string result_str = format_type(result_type);
        if (name.get_name().find(string("operator ") + result_str) != 0)
          result = format_type (result_type, result);
      }
    }
  }
  return result;
}

string signature(const ACM_Code &code_param) {
  ACM_Code &code = (ACM_Code&)code_param;
  if (code.type_val() == JPT_Call)
    return signature (*((ACM_Call&)code).get_target());
  else
    return signature (*(ACM_Name*)code.get_parent());
}

string format_type(ACM_Type &type) {
  string sig = type.get_signature();
  string::size_type pos = sig.find(" ?)");
  if (pos != string::npos) {
    sig.erase(pos, 2);
  }
  else {
    pos = sig.find("?");
    if (pos != string::npos) {
      sig.erase(pos, 1);
    }
  }
  return sig;
}

string format_type(ACM_Type &type, const string &name) {
  string sig = type.get_signature();
  string::size_type pos = sig.find("?");
  if (pos != string::npos) {
    sig.replace(pos, 1, name);
  }
  else {
    char last = sig[sig.length () -1];
    if (last != '*' && last != '&')
      sig += " ";
    sig += name;
  }
  return sig; 
}

// functions that return the number of argument types

int get_arg_count (ACM_Function &func) {
  return func.get_arg_types().size();
}

int get_arg_count (ACM_Call &code) {
  ACM_Function *func = code.get_target();
  if (code.has_default_args())
    return get_arg_count (*func) - code.get_default_args();
  else
    return get_arg_count (*func) + code.get_variadic_arg_types().size();
}

int get_arg_count (ACM_Code &code) {
  if (code.type_val() == JPT_Call)
    return get_arg_count ((ACM_Call&)code);
  else
    return get_arg_count (*(ACM_Function*)code.get_parent());
}


// functions that return the argument/result types of certain join-points

void add_arg_types (ACM_Function &func, list<ACM_Type*> &type_list) {
  typedef ACM_Container<ACM_Type, true> Container;
  Container &types = func.get_arg_types();
  for (Container::iterator i = types.begin (); i != types.end(); ++i)
    type_list.push_back (*i);
}

void get_arg_types (ACM_Function &func, list<ACM_Type*> &type_list) {
  type_list.clear();
  add_arg_types (func, type_list);
}


void add_arg_types (ACM_Call &jpl, list<ACM_Type*> &type_list) {
  typedef ACM_Container<ACM_Type, true> Container;

  int args = get_arg_count (jpl);
  Container &types = jpl.get_target()->get_arg_types();
  int arg = 0;
  for (Container::iterator i = types.begin (); i != types.end() && arg < args;
       ++i, ++arg)
    type_list.push_back (*i);
  if (arg < args) {
    Container &var_types = jpl.get_variadic_arg_types();
    for (Container::iterator i = var_types.begin (); i != var_types.end(); ++i)
      type_list.push_back (*i);
  }
}

void get_arg_types (ACM_Call &jpl, list<ACM_Type*> &type_list) {
  type_list.clear();
  add_arg_types (jpl, type_list);
}


void add_arg_types (ACM_Code &jpl, list<ACM_Type*> &type_list) {
  // TODO: this type check is dangerous!
  if (jpl.type_val() == JPT_Call)
    add_arg_types ((ACM_Call&)jpl, type_list);
  else
    add_arg_types (*(ACM_Function*)jpl.get_parent(), type_list);
}

void get_arg_types (ACM_Code &jpl, list<ACM_Type*> &type_list) {
  type_list.clear();
  add_arg_types (jpl, type_list);
}

 bool has_result_type (ACM_Code &jpl) {
  if (jpl.type_val() == JPT_Call)
    return ((ACM_Call&)jpl).get_target()->has_result_type();
  else
    return ((ACM_Function*)jpl.get_parent())->has_result_type();
}

ACM_Type *get_result_type (ACM_Code &jpl) {
  if (jpl.type_val() == JPT_Call)
    return ((ACM_Call&)jpl).get_target()->get_result_type();
  else
    return ((ACM_Function*)jpl.get_parent())->get_result_type();
}

// check whether two functions have the same name and argument types
bool have_same_name_and_args (ACM_Function &l, ACM_Function &r) {
  if (get_arg_count (l) != get_arg_count (r) ||
      l.get_variadic_args() != r.get_variadic_args() || 
      l.get_name() != r.get_name())
    return false;
  typedef ACM_Container<ACM_Type, true> Container;
  Container &ltypes = l.get_arg_types();
  Container &rtypes = r.get_arg_types();
  for (Container::const_iterator liter = ltypes.begin(), riter = rtypes.begin();
       liter != ltypes.end(); ++liter, ++riter)
    if ((*liter)->get_signature() != (*riter)->get_signature())
      return false;
  return true;
}

// check whether a join-point location is only a pseudo join-point location
bool is_pseudo(ACM_Any &jpl) {
  if (jpl.type_val() == JPT_Call && !jpl.get_parent ())
    return true;
  return false;
}

// get the lexical scope of a code join-point as needed by 'within'
ACM_Name *lexical_scope (ACM_Code &jpl) {
  ACM_Name *parent = (ACM_Name*)jpl.get_parent();
  if (jpl.type_val() == JPT_Call) {
    if (parent->type_val() == JPT_Variable)
      parent = (ACM_Name*)parent->get_parent();
  }
  else
    parent = (ACM_Name*)parent->get_parent();
  return parent;
}

// check whether a function needs an object to be invoked ("this pointer")
bool needs_this (ACM_Function &func) {
  assert (func.get_kind() != FT_UNKNOWN);
  switch (func.get_kind()) {
  case FT_UNKNOWN:
  case FT_NON_MEMBER:
  case FT_STATIC_NON_MEMBER:
  case FT_STATIC_MEMBER:
    return false;
  case FT_CONSTRUCTOR:
  case FT_DESTRUCTOR:
  case FT_VIRTUAL_DESTRUCTOR:
  case FT_PURE_VIRTUAL_DESTRUCTOR:
    return true;
  case FT_MEMBER:
  case FT_VIRTUAL_MEMBER:
  case FT_PURE_VIRTUAL_MEMBER:
    break;
  }
  if (func.get_name().substr(0,9) != "operator ")
    return true;

  string name = func.get_name().substr(9);
  return name != "new" && name != "new[]" && name != "delete" && name != "delete[]";
}

// check whether a name joinpoint is defined in the project
bool in_project (ACM_Name &name) {
  return name.get_tunits().empty();
}

// describe the proceed code requirements for code join-points
bool proceed_needs_args(ACM_Code &jpl) {
  return get_arg_count (jpl) > 0;
}

bool proceed_needs_result (ACM_Code &jpl) {
  return has_result_type (jpl);
}

bool proceed_needs_that (ACM_Code &jpl) {
  switch (jpl.type_val()) {
  case JPT_Construction:
  case JPT_Destruction:
    return true;
  case JPT_Execution:
    return needs_this(*(ACM_Function*)((ACM_Execution&)jpl).get_parent());
  default:
    return false;
  }
}

bool proceed_needs_target (ACM_Code &jpl) {
  return jpl.type_val() == JPT_Call && needs_this(*((ACM_Call&)jpl).get_target());
}

// manage child name map of ACM_Name nodes

static string map_key (ACM_Name &name) {
  string result = name.get_name();
  if (name.type_val() & JPT_Function) {
    list<ACM_Type*> types;
    get_arg_types ((ACM_Function &)name, types);
    result += "(";
    for (list<ACM_Type*>::iterator i = types.begin(); i != types.end(); ++i) {
      if (i != types.begin())
        result +=", ";
      result += (*i)->get_signature();
    }
    result += ")";
  }
  return result;
}

void map_refresh(ACM_Name &name) {
  typedef ACM_Container<ACM_Any, true> Container;
  NameMap &map = name.get_name_map();
  Container &children = name.get_children();
  map.clear();
  for (Container::iterator i = children.begin(); i != children.end(); ++i) {
    if ((*i)->type_val() & JPT_Name) {
      ACM_Name *child = (ACM_Name*)*i;
      map.insert(NameMapPair(map_key(*child), child));
    }
  }
}

ACM_Name *map_lookup(ACM_Name &parent, const string &key) {
  NameMap &map = parent.get_name_map();
  if (map.size() == 0 && parent.get_children().size() > 0)
    map_refresh(parent);
  NameMap::iterator i = map.find(key);
  if (i != map.end())
    return i->second;
  return 0;
}

void map_insert(ACM_Name &parent, ACM_Name &child, const string &key) {
  NameMap &map = parent.get_name_map();
  map.insert(NameMapPair(key, &child));
}

// context information of a join-point

string filename (ACM_Any &jpl) {
  typedef ACM_Container<ACM_Source, true> Container;
  Container &source = (jpl.type_val() &
      (JPT_Construction|JPT_Destruction|JPT_Execution)) ?
          ((ACM_Code*)jpl.get_parent())->get_source() : jpl.get_source();
  if (source.size() == 0)
    return "<unknown>";

  Container::iterator last;
  for (Container::iterator i = source.begin (); i != source.end (); ++i) {
    last = i;
    if ((*i)->get_kind () == SLK_DEF)
      break;
  }
  return (*last)->get_file()->get_filename();
}

int line (ACM_Any &jpl) {
  typedef ACM_Container<ACM_Source, true> Container;
  Container &source = (jpl.type_val() &
      (JPT_Construction|JPT_Destruction|JPT_Execution)) ?
          ((ACM_Code*)jpl.get_parent())->get_source() : jpl.get_source();
  if (source.size() == 0)
    return -1;

  Container::iterator last;
  for (Container::iterator i = source.begin (); i != source.end (); ++i) {
    last = i;
    if ((*i)->get_kind () == SLK_DEF)
      break;
  }
  return (*last)->get_line();
}

// get the slice that is associated with on introduction
ACM_ClassSlice *get_slice (ACM_Introduction &intro) {
  ACM_ClassSlice *cs = 0;
  if (intro.has_named_slice ())
    cs = intro.get_named_slice ();
  else if (intro.has_anon_slice ())
    cs = intro.get_anon_slice ();
  return cs;
}

// collect the set of abstract member functions and pointcuts of a class or aspect
void get_abstract_members (ACM_Class &cls, set<string> &abstract_members) {
  // iterate over all base classes and get the inherited abstract members
  typedef ACM_Container<ACM_Class, false> Container;
  Container &bases = cls.get_bases();
  for (Container::iterator i = bases.begin (); i != bases.end (); ++i) {
    set<string> base_members; // key strings
    get_abstract_members (**i, base_members);
    // now check each abstract base class member in the current class
    for (set<string>::const_iterator ki = base_members.begin ();
        ki != base_members.end (); ++ki) {
      ACM_Any *elem = map_lookup(cls, *ki);
      bool redefined = false;
      if ((*ki)[0] == '~')
        redefined = true; // pure virtual destructor are defined implicitly
      else if (elem) {
        // check the found member
        if (elem->type_val() == JPT_Pointcut &&
            ((ACM_Pointcut*)elem)->get_kind() != PT_PURE_VIRTUAL)
          redefined = true;
        else if (elem->type_val() == JPT_Function &&
             ((ACM_Function*)elem)->get_kind() == FT_VIRTUAL_MEMBER)
          redefined = true;
      }
      // if it is not redefined here as non-abstract, keep it
      if (!redefined)
        abstract_members.insert (*ki);
    }
  }

  // non check all member functions and pointcuts of this class
  NameMap &name_map = cls.get_name_map();
  for (NameMap::const_iterator i = name_map.begin (); i != name_map.end (); ++i) {
    string key = i->first;
    ACM_Any *elem = i->second;
    if ((elem->type_val() == JPT_Pointcut &&
        ((ACM_Pointcut*)elem)->get_kind() == PT_PURE_VIRTUAL) ||
        (elem->type_val() == JPT_Function &&
        (((ACM_Function*)elem)->get_kind() == FT_PURE_VIRTUAL_MEMBER ||
        ((ACM_Function*)elem)->get_kind() == FT_PURE_VIRTUAL_DESTRUCTOR)))
      abstract_members.insert (key);
  }

//  cout << "result for " << cls.get_name() << endl;
//  for (set<string>::const_iterator ki = abstract_members.begin ();
//      ki != abstract_members.end (); ++ki) {
//    cout << "  " << *ki << endl;
//  }
}

// check whether a class or aspect is abstract
bool is_abstract (ACM_Class &cls) {
  set<string> abstract_members;
  get_abstract_members (cls, abstract_members);
  return abstract_members.size () != 0;
}

// calculate number of nested advice levels below the given code plan
int depth (ACM_CodePlan &plan) {
  return plan.has_next_level() ? depth (*plan.get_next_level()) + 1 : 0;
}
