// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __Introducer_h__
#define __Introducer_h__

// Class that is used for introductions during a parser run.
// The semantics object has a reference to an Introducer and uses
// the interface at some well defined (join) points.

#include "Puma/ACIntroducer.h"
using namespace Puma;

#include <set>
using std::set;
#include <list>
using std::list;
#include <map>
using std::map;

namespace Puma {
  class CT_ClassDef;
  class CT_Program;
  class CCParser;
  class PreprocessorParser;
  class CClassInfo;
  class TokenProvider;
  class TokenStream;
}

class ACConfig;
class IncludeGraph;
class CodeWeaver;
class JPP_Class;
class ACM_Class;
class ACM_Introduction;
class Plan;
class ModelBuilder;
class ACUnit;
class LineDirectiveMgr;
class IntroductionUnit;

class Introducer : public ACIntroducer {
  Plan &_plan;
  CodeWeaver &_code_weaver;
  CCParser *_parser;
  ModelBuilder &_jpm;
  int _intro_level;
  PreprocessorParser *_cpp;
  IncludeGraph &_ig;
  ACConfig &_conf;
  set<const Unit*> _included_aspect_headers;
  list<CTree*> _ah_trees;
  typedef map<CClassInfo*,ACM_Class*> TargetMap;
  TargetMap _targets;
  list<TokenProvider*> _token_providers;
  list<TokenStream*> _token_streams;
    
  // find a member that is/must be link-once (global) code
  static CObjectInfo *link_once_object (CClassInfo *ci);

  // check whether an object was introduced
  static bool is_intro (CObjectInfo *obj);

  // parse code that shall be introduced
  CTree *parse (list<Unit*> units, bool (CCSyntax::*rule)(),
    const char *expected_id, ErrorStream &err);
    
  // checks if an attribute that us returned by the parser is an attribute
  // in the sense of the AspectC++ introspection mechnism
  bool is_attribute (CAttributeInfo *obj);

public:
  // Contructor: every Introducer knows the parser
  Introducer (Plan &pl, CodeWeaver &cw, CCParser &p, ModelBuilder &jpm,
    IncludeGraph &ig, ACConfig &c) : _plan (pl), _code_weaver (cw),
    _parser (&p), _jpm (jpm), _intro_level (0), _cpp (0), _ig (ig), _conf(c) {}
  
  // Destructor: release all allocated resources
  ~Introducer ();
  
  // called when a new class/union/struct/aspect is created, current scope
  // is the global scope
  virtual void class_before (CT_ClassDef*);

  // called when a new class/union/struct/aspect is created
  virtual void class_begin (CT_ClassDef*);

  // called when a new class/union/struct/aspect definition ends
  // (still in the class scope)
  virtual void class_end (CT_ClassDef*);
  
  // called after the parser tried to parse a base clause
  virtual void base_clause_end (CT_ClassDef*, Token*);

  // called after the program has been parsed completely
  virtual void trans_unit_end (CT_Program *);

  // get the list of aspect header syntax trees
  list<CTree*> &ah_trees () { return _ah_trees; }

private:

  // check in the plan if the current class needs an introduction
  ACM_Class *plan_lookup (CClassInfo *ci);

  // create the weaving plan for a given class and return the class object
  ACM_Class *create_plan (CClassInfo *ci);
  
  // insert introspection code
  //  * at the end of class definitions, after AspectC++ introductions
  //  * before a slice, if "precedence" is given
  void insert_introspection_code (CT_ClassDef *cd, int precedence = -1);

  // insert friend declarations for all aspects
  bool insert_aspect_friend_decls (ACUnit &unit, CRecord *rec);

  // insert friend declarations for all aspects
  bool insert_aspectof_function (ACUnit &unit, CClassInfo *ci,
      ErrorStream &err);

  // manage the intro nesting level and the _cpp pointer
  void enter ();
  void leave ();
  
  // generate member intros for a given class
  void gen_intros (ACM_Class *jpl, list<Unit*> &units, ErrorStream &err,
      CStructure *target, LineDirectiveMgr &lmgr, int introduction_depth,
      bool non_inline = false) const;

  // generate base class intros for a given class
  void gen_base_intros (ACM_Class *jpl, list<Unit*> &units, ErrorStream &err,
    CClassInfo *target, LineDirectiveMgr &lmgr) const;

  void gen_base_intro (IntroductionUnit &intro_unit,
    ACM_Introduction *ii, bool first) const;
};

#endif /* __Introducer_h__ */
