#include <typeinfo>

#include <pspell/manager_impl.hh>
#include <pspell/wordlist.h>
#include <pspell/convert.hh>
#include <pspell/error_messages.hh>
#include <pspell/error_impl.hh>

#include "manager.hh"
#include "data.hh"
#include "suggest.hh"
#include "config.hh"
#include "app_string.hh"
#include "language.hh"

namespace pspell_aspell {

  using namespace aspell;
  using namespace autil;

  template <typename T>
  class ConvertStringEmulation : public PspellStringEmulation {
    Emulation<T> real_emul;
    string    temp_str;
    AppString temp_buf;
    const PspellConvert * from_internal;
    LocalWordSetInfo local_info;
  public:
    ConvertStringEmulation(VirEmulation<T> * emul, 
			   const PspellConvert * c,
			   LocalWordSetInfo lc = LocalWordSetInfo())
      : real_emul(emul)
      , temp_buf(&temp_str)
      , from_internal(c) 
      , local_info(lc)
    {}
    ConvertStringEmulation(const ConvertStringEmulation & other) 
      : real_emul(other.real_emul)
      , temp_str(other.temp_str)
      , temp_buf(&temp_str)
      , from_internal(other.from_internal)
    {}
    ConvertStringEmulation & operator= (const ConvertStringEmulation & other)
    {
      real_emul = other.real_emul;
      temp_str = other.temp_str;
      from_internal = other.from_internal;
      return *this;
    }
    const char * next();
    bool at_end() const;
    PspellStringEmulation * clone() const;
    void assign(const PspellStringEmulation * other);
  };

  template <>
  const char * ConvertStringEmulation<const char *>::next() {
    const char * w = real_emul.next();
    if (w == 0) return 0;
    temp_str = "";
    from_internal->convert(w, temp_buf);
    return temp_str.c_str();
  }

  template <>
  const char * ConvertStringEmulation<BasicWordInfo>::next() {
    BasicWordInfo wi = real_emul.next();
    if (!wi) return 0;
    string tmp;
    wi.get_word(tmp,local_info.convert);
    temp_str = "";
    from_internal->convert(tmp.c_str(), temp_buf);
    return temp_str.c_str();
  }

  template <typename T>
  bool ConvertStringEmulation<T>::at_end() const {
    return real_emul.at_end();
  }

  template <typename T>
  PspellStringEmulation * ConvertStringEmulation<T>::clone() const {
    return new ConvertStringEmulation(*this);
  }

  template <typename T>
  void ConvertStringEmulation<T>::assign(const PspellStringEmulation * other) {
    *this = *static_cast<const ConvertStringEmulation *>(other);
  }


  template <class C,typename T> 
  class PA_WordList : public PspellWordList
  {
  public:
    const C * wl;
    const PspellConvert * from_internal;

    PA_WordList() : wl(0) {}

    const char * encoding() const
    {
      return from_internal->out_code();
    }
    
    bool empty() const 
    {
      return wl->empty();
    }
     
    int size() const 
    {
      return wl->size();
    }
    
    PspellStringEmulation  * elements() const 
    {
      return new ConvertStringEmulation<T>(wl->elements(), from_internal);
    }

  };
  
  class PA_Manager : public PspellManagerImplBase
  {
  private:
    Manager manager_;
    const PspellConvert * to_internal_;
    const PspellConvert * from_internal_;

    //PA_WordList<LoadableWordSet, BasicWordInfo> main_wl_;
    PA_WordList<BasicWordSet, BasicWordInfo> personal_wl_;
    PA_WordList<BasicWordSet, BasicWordInfo> session_wl_;
    PA_WordList<SuggestionList,const char *> suggestions_;

    PA_Manager(const PA_Manager &);
    PA_Manager & operator= (const PA_Manager &);

  public:

    PA_Manager(PspellConfig * config, PspellManagerLtHandle h);

    void to_internal(const char * in, int size, string & out) const {
      AppString buf(&out);
      to_internal_->convert(in, size, buf);
    }

    void from_internal(const char * in, string & out) const {
      AppString buf(&out);
      from_internal_->convert(in, buf);
    }

    virtual PspellConfig & config();
    
    virtual int check(const char * word, int size);

    virtual bool add_to_personal(const char * word, int size);

    virtual bool add_to_session(const char * word, int size);

    virtual const PspellWordList * master_word_list();

    virtual const PspellWordList * personal_word_list();

    virtual const PspellWordList * session_word_list();
    
    virtual bool save_all_word_lists();

    virtual bool clear_session();
    
    virtual const PspellWordList * suggest(const char * word, int size);
  
    virtual bool store_replacement(const char * mis, int mis_size,
				   const char * cor, int cor_size);
    
  };

  PA_Manager::PA_Manager(PspellConfig * config, PspellManagerLtHandle h)
    : PspellManagerImplBase(h)
    , to_internal_(0)
    , from_internal_(0)
  {
    string encoding = config->retrieve("encoding");

    manager_.init_config();
    
    manager_.config().set_extra(pspell_config_impl_keys_begin,
                                pspell_config_impl_keys_end);
    
    assert(dynamic_cast<PspellConfigImpl *>(config) != 0);
    manager_.setup(*static_cast<PspellConfigImpl *>(config));

    const char * charset = manager_.lang().charset();

    PspellCanHaveError * temp;

    temp = new_pspell_convert(*config, encoding.c_str(), charset);
    if (temp->error_number() != 0) {
      throw temp;
    }
    to_internal_ = static_cast<PspellConvert *>(temp);
      
    temp = new_pspell_convert(*config, charset, encoding.c_str());
    if (temp->error_number() != 0) {
      throw temp;
    }
    from_internal_ = static_cast<PspellConvert *>(temp);

    //main_wl_    .wl = &manager_.main_wl();
    //main_wl_.from_internal = from_internal_;

    personal_wl_.wl = &manager_.personal_wl();
    personal_wl_.from_internal = from_internal_;

    session_wl_ .wl = &manager_.session_wl();
    session_wl_.from_internal = from_internal_;

    suggestions_.from_internal = from_internal_;
  }

  PspellConfig & PA_Manager::config() {
    return manager_.config().real_config();
  }
    
  int PA_Manager::check(const char * word, int size) {
    error_.reset_error();
    string temp;
    to_internal(word, size, temp);
    return manager_.check(temp) != 0;
  }

  bool PA_Manager::add_to_personal(const char * word, int size) {
    try {

      error_.reset_error();
      string temp;
      to_internal(word, size, temp);
      manager_.add_to_personal(temp);
      return true;

    } catch (const Exception & e) {

      error_.set_error(e.number, e.what());
      return false;

    }
  }

  bool PA_Manager::add_to_session(const char * word, int size) {
    try {

      error_.reset_error();
      string temp;
      to_internal(word, size, temp);
      manager_.add_to_session(temp);
      return true;

    } catch (const Exception & e) {

      error_.set_error(e.number, e.what());
      return false;

    }
  }

  const PspellWordList * PA_Manager::master_word_list() 
  {
    error_.reset_error();
    error_.set_error(operation_not_supported, "get master_word_list");
    return 0;
  }

  const PspellWordList * PA_Manager::personal_word_list()
  {
    error_.reset_error();
    return &personal_wl_;
  }

  const PspellWordList * PA_Manager::session_word_list() 
  {
    error_.reset_error();
    return &session_wl_;
  }
    
  bool PA_Manager::save_all_word_lists() {
    error_.reset_error();
    try {
      manager_.save_all_wls();
      return true;
    } catch (const Exception & e) {
      error_.set_error(e.number, e.what());
      return false;
    }
  }
    
  bool PA_Manager::clear_session() {
    error_.reset_error();
    manager_.clear_session();
    return true;
  }
    
  const PspellWordList * PA_Manager::suggest(const char * word, int size) {
    error_.reset_error();
    string temp;
    to_internal(word, size, temp);
    suggestions_.wl = &manager_.suggest(temp);
    return &suggestions_;
  }
  
  bool PA_Manager::store_replacement(const char * mis, int mis_size,
				     const char * cor, int cor_size) 
  {
    error_.reset_error();
    string converted_mis, converted_cor;
    to_internal(mis, mis_size, converted_mis);
    to_internal(cor, cor_size, converted_cor);
    manager_.store_repl(converted_mis, converted_cor);
    return true;
  }

}

extern "C"
PspellCanHaveError *
libpspell_aspell_LTX_new_pspell_manager_class(PspellConfig * config, 
					      PspellManagerLtHandle h)
{
  using namespace pspell_aspell;
  try {
    PA_Manager * m = new PA_Manager(config, h);
    return m;
  } catch (PspellCanHaveError * e) {
    return e;
  } catch (Exception & e) {
    return (new PspellCanHaveErrorImpl())->set_error(e.number, e.what());
  }
}
